﻿using System;
using System.Linq;
using System.Collections.Generic;
using System.Web.UI;
using Framework.Data;

namespace Framework.Web.UI
{
    /// <summary>
    /// コントロールの各種一括操作を行うメソッドが登録してあります。
    /// 一括操作の基本ルールは指定したコントロール以下にあるIMappingControl,IInputControlなどのインタフェースを
    /// 実装したコントロールに対して操作をリクエストします。
    /// 再帰的に子コントロールの検索を行いますが、INamingContainer以下は検索しません。
    /// </summary>
    public static class UIController
    {
        /// <summary>
        /// parentControl以下にあるIMappingControlを実装したコントロールから値を取得します。
        /// </summary>
        /// <param name="parentControls"></param>
        /// <returns></returns>
        public static MappingData GetMappingData(IEnumerable<Control> parentControls)
        {
            if (parentControls == null || parentControls.Any() == false) return null;

            var data = new Framework.Data.MappingData();

            foreach (var parentControl in parentControls)
            {
                var mc = parentControl as IMappingControl;
                if (mc != null)
                {
                    mc.RequestMappingData(data);
                }
                else
                {
                    UIController.RequestMappingData(parentControl, data);
                }
            }

            return data;
        }

        /// <summary>
        /// parentControl以下にあるIMappingControlを実装したコントロールから値を取得します。
        /// </summary>
        /// <param name="parentControl"></param>
        /// <returns></returns>
        public static MappingData GetMappingData(Control parentControl)
        {
            if (parentControl == null) return null;

            var data = new Framework.Data.MappingData();

            var mc = parentControl as IMappingControl;
            if (mc != null)
            {
                mc.RequestMappingData(data);
            }
            else
            {
                UIController.RequestMappingData(parentControl, data);
            }

            return data;
        }

        private static void RequestMappingData(Control parentControl, Framework.Data.MappingData data)
        {
            foreach (Control c in parentControl.Controls)
            {
                var mc = c as IMappingControl;
                if (mc != null)
                {
                    mc.RequestMappingData(data);
                }
                else if (CanSearchChild(c))
                {
                    UIController.RequestMappingData(c, data);
                }
            }
        }

        /// <summary>
        /// parentControl以下にあるIMappingControlを実装したコントロールに値をセットします。
        /// </summary>
        /// <param name="parentControl"></param>
        /// <param name="data"></param>
        public static void SetMappingData(Control parentControl, Framework.Data.MappingData data)
        {
            if (data == null)
            {
                data = new MappingData();
            }

            var mc = parentControl as IMappingControl;
            if (mc != null)
            {
                mc.SetMappingData(data);
            }
            else
            {
                UIController.SetMappingDataSub(parentControl, data);
            }
        }

        private static void SetMappingDataSub(Control parentControl, Framework.Data.MappingData data)
        {
            foreach (Control c in parentControl.Controls)
            {
                var mc = c as IMappingControl;
                if (mc != null)
                {
                    mc.SetMappingData(data);
                }
                else if (CanSearchChild(c))
                {
                    UIController.SetMappingDataSub(c, data);
                }
            }
        }

        /// <summary>
        /// parentControl配下のIListControlのデータソースをリクエストし、データバインドします。
        /// </summary>
        /// <param name="parentControl"></param>
        public static void ListSrcDataBind(Control parentControl)
        {
            var lc = parentControl as IListControl;
            if (lc != null)
            {
                lc.ListSrcDataBind();
            }
            else
            {
                UIController.ListSrcDataBindSub(parentControl);
            }
        }

        private static void ListSrcDataBindSub(Control parentControl)
        {
            foreach (Control c in parentControl.Controls)
            {
                var lc = c as IListControl;
                if (lc != null)
                {
                    lc.ListSrcDataBind();
                }
                else if (UIController.CanSearchChild(c))
                {
                    UIController.ListSrcDataBindSub(c);
                }
            }
        }

        /// <summary>
        /// parentControl以下にあるILocalizeControlを実装したコントロールをローカライズします。
        /// </summary>
        /// <param name="parentControl"></param>
        public static void Localize(Control parentControl)
        {
            UIController.Localize(parentControl, Lang.CurrentCulture);
        }

        /// <summary>
        /// parentControl以下にあるILocalizeControlを実装したコントロールを指定言語(lang)でローカライズします。
        /// </summary>
        /// <param name="parentControl"></param>
        public static void Localize(Control parentControl, System.Globalization.CultureInfo lang)
        {
            var lc = parentControl as ILocalizeControl;
            if (lc != null)
            {
                lc.Localize(lang);
            }
            else
            {
                UIController.LocalizeSub(parentControl, lang);
            }
        }

        private static void LocalizeSub(Control parentControl, System.Globalization.CultureInfo lang)
        {
            foreach (Control c in parentControl.Controls)
            {
                var lc = c as ILocalizeControl;
                if (lc != null)
                {
                    lc.Localize(lang);
                }
                else if (UIController.CanSearchChild(c))
                {
                    UIController.LocalizeSub(c, lang);
                }
            }
        }

        public static System.Drawing.Color ValidateSuccessColor = System.Drawing.Color.Empty;
        public static System.Drawing.Color ValidateErrorColor = System.Drawing.Color.Yellow;

        /// <summary>
        /// parentControl以下にあるIInputControlを実装したコントロールの入力値を検証します。
        /// 不正なコントロールの背景色を黄色にします。不正がある場合はfalseを返します。
        /// </summary>
        /// <param name="parentControl"></param>
        /// <returns></returns>
        public static bool ValidateInput(Control parentControl)
        {
            Control ctrl = null;
            return ValidateInput(parentControl, ref ctrl, false);
        }
        public static bool ValidateInput(Control parentControl, bool dataTypeOnly)
        {
            Control ctrl = null;
            return ValidateInput(parentControl, ref ctrl, dataTypeOnly);
        }

        /// <summary>
        /// parentControls以下にあるIInputControlを実装したコントロールの入力値を検証します。
        /// 不正なコントロールの背景色を黄色にします。不正がある場合はfalseを返します。
        /// </summary>
        /// <param name="parentControls"></param>
        /// <param name="errorControl"></param>
        /// <returns></returns>
        public static bool ValidateInput(IEnumerable<Control> parentControls, ref Control errorControl)
        {
            return ValidateInput(parentControls, ref errorControl, false);
        }

        public static bool ValidateInput(IEnumerable<Control> parentControls, ref Control errorControl, bool dataTypeOnly)
        {
            bool b = true;
            foreach (var pc in parentControls)
            {
                if (ValidateInput(pc, ref errorControl, dataTypeOnly) == false)
                {
                    b = false;
                }
            }
            return b;
        }

        /// <summary>
        /// parentControl以下にあるIInputControlを実装したコントロールの入力値を検証します。
        /// 不正なコントロールの背景色を黄色にします。不正がある場合はfalseを返します。
        /// また最初に見つかった不正なコントロールの参照をerrorControlにセットします。
        /// </summary>
        /// <param name="parentControl"></param>
        /// <param name="errorControl"></param>
        /// <returns></returns>
        public static bool ValidateInput(Control parentControl, ref Control errorControl)
        {
            return UIController.ValidateInput(parentControl, ref errorControl, false);
        }

        public static bool ValidateInput(Control parentControl, ref Control errorControl, bool dataTypeOnly)
        {
            bool ret = true;
            var ic = parentControl as IInputControl;
            if (ic != null)
            {
                ret = ic.ValidateInput(dataTypeOnly);
                ic.SetValidateResult(ret);
                if (ret == false)
                {
                    errorControl = parentControl;
                }

            }
            else
            {
                UIController.ValidateInputSub(parentControl, ref ret, dataTypeOnly);
                if (!ret)
                {
                    errorControl = UIController.FindValidateError(parentControl);
                }
            }

            return ret;
        }

        private static void ValidateInputSub(Control parentControl, ref bool ret, bool dataTypeOnly)
        {
            foreach (Control c in parentControl.Controls)
            {
                var ic = c as IInputControl;
                if (ic != null)
                {
                    bool b = ic.ValidateInput(dataTypeOnly);
                    if (b == false && ret == true)
                    {
                        ret = false;
                    }
                    ic.SetValidateResult(b);
                }
                else if (CanSearchChild(c))
                {
                    UIController.ValidateInputSub(c, ref ret, dataTypeOnly);
                }
            }
        }

        private static Control FindValidateError(Control parentControl)
        {
            foreach (Control c in parentControl.Controls)
            {
                var ic = c as IInputControl;
                if (ic != null)
                {
                    if (!ic.IsValid)
                    {
                        return ic.GetErrorControl();
                    }
                }
                else if (CanSearchChild(c))
                {
                    var ret = UIController.FindValidateError(c);
                    if (ret != null)
                    {
                        return ret;
                    }
                }
            }

            return null;
        }

        /// <summary>
        /// parentControl以下にあるIInputControlを実装したコントロールのいづれかに値が入力されている場合はtrueを返します。
        /// MTextBoxやMDropDownListのようにDefaultValueが設定できるコントロールの場合は値がDefaultValueが同じ場合は入力されていないとみなします。
        /// </summary>
        /// <param name="parentControl"></param>
        /// <returns></returns>
        public static bool IsInputed(Control parentControl)
        {
            var ic = parentControl as IInputControl;
            if (ic != null)
            {
                return ic.IsInputed();
            }
            else
            {
                return IsInputedSub(parentControl);
            }
        }

        private static bool IsInputedSub(Control parentControl)
        {
            foreach (Control c in parentControl.Controls)
            {
                var ic = c as IInputControl;
                if (ic != null)
                {
                    return ic.IsInputed();
                }
                else if (UIController.CanSearchChild(c))
                {
                    if (UIController.IsInputedSub(c))
                    {
                        return true;
                    }
                }
            }

            return false;
        }

        /// <summary>
        /// parentControl以下にあるIInputControlを実装したコントロールの入力値を空にします。
        /// </summary>
        /// <param name="parentControl"></param>
        public static void ClearInput(Control parentControl)
        {
            var ic = parentControl as IInputControl;
            if (ic != null)
            {
                ic.ClearInput();
            }
            else
            {
                ClearInputSub(parentControl);
            }
        }

        private static void ClearInputSub(Control parentControl)
        {
            foreach (Control c in parentControl.Controls)
            {
                var ic = c as IInputControl;
                if (ic != null)
                {
                    ic.ClearInput();
                }
                else if (CanSearchChild(c))
                {
                    UIController.ClearInputSub(c);
                }
            }
        }

        /// <summary>
        /// parentControl以下にあるIInputControlを実装したコントロールの入力の入力エラーを解除します。
        /// </summary>
        /// <param name="parentControl"></param>
        public static void ClearValidateError(Control parentControl)
        {
            var ic = parentControl as IInputControl;
            if (ic != null)
            {
                ic.IsValid = true;
                ic.SetValidateResult(true);
            }
            else
            {
                UIController.ClearValidateErrorSub(parentControl);
            }
        }

        private static void ClearValidateErrorSub(Control parentControl)
        {
            foreach (Control c in parentControl.Controls)
            {
                var ic = c as IInputControl;
                if (ic != null)
                {
                    ic.IsValid = true;
                    ic.SetValidateResult(true);
                }
                else if (UIController.CanSearchChild(c))
                {
                    UIController.ClearValidateErrorSub(c);
                }
            }
        }

        /// <summary>
        /// コントロールのIDが正規表現patternに一致するコントロールの配列を返します。
        /// </summary>
        /// <param name="parentControl"></param>
        /// <param name="pattern"></param>
        /// <returns></returns>
        public static Control[] FindControls(Control parentControl, string pattern)
        {
            var list = new List<Control>();
            var regex = new System.Text.RegularExpressions.Regex(pattern);

            if (regex.IsMatch(parentControl.ID)) list.Add(parentControl);
            FindControlsSub(parentControl, regex, ref list);

            return list.ToArray();
        }

        private static void FindControlsSub(Control parentControl, System.Text.RegularExpressions.Regex regex, ref List<Control> ctrls)
        {
            foreach (Control c in parentControl.Controls)
            {
                if (c.ID.IsNotEmpty() && regex.IsMatch(c.ID))
                {
                    ctrls.Add(c);
                }

                if (CanSearchChild(c))
                {
                    FindControlsSub(c, regex, ref ctrls);
                }
            }
        }

        /// <summary>
        /// コントロールの型がTに一致するコントロールの配列を返します。
        /// </summary>
        /// <param name="parentControl"></param>
        /// <param name="pattern"></param>
        /// <returns></returns>
        public static T[] FindControls<T>(Control parentControl) where T : class
        {
            var list = new List<T>();

            Func<Control, Control> func = null;
            func = (c) =>
            {
                if (c != null)
                {
                    var ct = c as T;
                    if (ct != null) list.Add(ct);

                    c.Controls.OfType<Control>().Foreach(cc =>
                    {
                        var cct = cc as T;
                        if (cct != null) list.Add(cct);

                        if (CanSearchChild(cc))
                        {
                            func(cc);
                        }
                    });
                }
                return null;
            };
            func(parentControl);

            return list.ToArray();
        }


        private static bool CanSearchChild(Control c)
        {
            return c.HasControls() && !(c is INamingContainer);
        }
    }
}
