/*
    SDL_archive
    Copyright (C) 2004  Kazunori Itoyanagi

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Kazunori Itoyanagi
    itkz@users.sourceforge.jp
*/

#ifndef _SDL_archive_H_
#define _SDL_archive_H_


#include<stdio.h>
#include "SDL_version.h"
#include "SDL_rwops.h"

#include "begin_code.h"



#define ARCHIVE_MAJOR_VERSION	0
#define ARCHIVE_MINOR_VERSION	2
#define ARCHIVE_PATCHLEVEL		4

/* This macro can be used to fill a version structure with the compile-time
 * version of the SDL_mixer library.
 */
#define ARCHIVE_VERSION(X)							\
{									\
	(X)->major = ARCHIVE_MAJOR_VERSION;					\
	(X)->minor = ARCHIVE_MINOR_VERSION;					\
	(X)->patch = ARCHIVE_PATCHLEVEL;					\
}


/* Set up for C function definitions, even when using C++ */
#ifdef __cplusplus
extern "C" {
#endif



typedef struct _SDL_Archive
{
	/* protected section */
	void *data;
	
	/* must set up before open */
	int (*open)(struct _SDL_Archive *data, const char *filename);
	int (*numopen)(struct _SDL_Archive *data, const int num);
	int (*name2index)(struct _SDL_Archive *data, const char *filename);
	int (*filenum)(struct _SDL_Archive *data);
	void (*finish)(struct _SDL_Archive *data);
	
	/* not have to set up before open */
	int (*get_char)(struct _SDL_Archive *data);
	int (*read)(struct _SDL_Archive *data, void *mem, const int size, const int maxnum);
	int (*seek)(struct _SDL_Archive *data, const long offset, const int whence);
	long (*size)(struct _SDL_Archive *data);
	long (*tell)(struct _SDL_Archive *data);
	int (*eof)(struct _SDL_Archive *data);
	int (*close)(struct _SDL_Archive *data);
} SDL_Archive;

typedef struct _ArchiverList ArchiverList;


enum
{
	ARCHIVE_SUCCESS = 0,
	ARCHIVE_ERROR_WHENCE,
	ARCHIVE_ERROR_BROKEN_DATA,
	ARCHIVE_ERROR_UNKNOWN,
	ARCHIVE_ERROR_NO_INIT,
	ARCHIVE_ERROR_NO_EXIST_FILE,
	ARCHIVE_ERROR_ALLOCATE,
	ARCHIVE_ERROR_ALREADY_OPEN
};

enum
{
	ARCHIVE_TRUE,
	ARCHIVE_FALSE
};


extern DECLSPEC struct _SDL_Archive *Archive_Alloc(void);
extern DECLSPEC void Archive_FreeMainContext(struct _SDL_Archive *archive);

#define Archive_Open(ctx, file)	(ctx)->open(ctx, file)
#define Archive_NumOpen(ctx, num)	(ctx)->numopen(ctx, num)
#define Archive_GetChar(ctx)	(ctx)->get_char(ctx)
#define Archive_Size(ctx)	(ctx)->size(ctx)
#define Archive_Tell(ctx)	(ctx)->tell(ctx)
#define Archive_Seek(ctx, offset, whence)	(ctx)->seek(ctx, offset, whence)
#define Archive_Read(ctx, mem, size, maxnum)	(ctx)->read(ctx, mem, size, maxnum)
#define Archive_EOF(ctx)	(ctx)->eof(ctx)
#define Archive_Close(ctx)	(ctx)->close(ctx)
#define Archive_NameToIndex(ctx, name)	(ctx)->name2index(ctx, name)
#define Archive_FileNum(ctx)	(ctx)->filenum(ctx)
#define Archive_Finish(ctx)	(ctx)->finish(ctx)



extern DECLSPEC int Archive_IsPKZip(const char *file);
extern DECLSPEC struct _SDL_Archive *Archive_FromPKZip(const char *file);

extern DECLSPEC int Archive_IsTar(const char *file);
extern DECLSPEC struct _SDL_Archive *Archive_FromTar(const char *file);

extern DECLSPEC int Archive_IsDirectory(const char *file);
extern DECLSPEC struct _SDL_Archive *Archive_FromDirectory(const char *file);


typedef int (*Archive_IsOpenable)(const char *file);
typedef struct _SDL_Archive* (*Archive_FromFormat)(const char *file);

extern DECLSPEC ArchiverList *ArchiverList_Create(void);

extern DECLSPEC int ArchiverList_AddArchiver(
	ArchiverList *archiverList,
	Archive_IsOpenable isOpenable,
	Archive_FromFormat fromFormat);
extern DECLSPEC void ArchiverList_Free(ArchiverList *archiverList);
extern DECLSPEC struct _SDL_Archive *ArchiverList_CreateArchive(
	ArchiverList *archiverList,
	const char *file);

extern DECLSPEC struct _SDL_Archive *Archive_FromArchiverList(
	ArchiverList *archiverList,
	const char *file);
extern DECLSPEC struct _SDL_Archive *Archive_FromGZFilter(
	ArchiverList *archiverList,
	const char *file);

extern DECLSPEC SDL_RWops *SDL_RWFromArchive(struct _SDL_Archive *arc, const char *file);
extern DECLSPEC SDL_RWops *SDL_RWFromArchiveNum(struct _SDL_Archive *arc, const int num);




/* Ends C function definitions when using C++ */
#ifdef __cplusplus
}
#endif

#include "close_code.h"



#endif
