/*
    SDL_archive
    Copyright (C) 2004  Kazunori Itoyanagi

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Kazunori Itoyanagi
    itkz@users.sourceforge.jp
*/

#include<stdlib.h>
#include<stdio.h>
#include<string.h>

#ifdef WIN32
#  include<windows.h>
#else
#  include<dirent.h>
#endif

#include"SDL_archive.h"


typedef struct _file_chain
{
	long size;
	char *filename;
} FILE_CHAIN;


typedef struct _directory_DATA
{
	FILE *fp;
	int chainNum;
	int chainAllocated;
	
	int tempStringAllocated;
	char *tempString;
	
	int openNumber;
	FILE_CHAIN *chain;
	char *dirname;
} DIRECTORY_DATA;


static DIRECTORY_DATA *Dir_Create(const char *file);
static int Dir_Open(SDL_Archive *archive, const char *filename);
static int Dir_NumOpen(SDL_Archive *archive, const int num);
static int Dir_NameToIndex(SDL_Archive *archive, const char *filename);
static long Dir_Size(SDL_Archive *archive);
static int Dir_FileNumber(SDL_Archive *archive);
static void Dir_Finish(SDL_Archive *archive);
static int Dir_GetChar(SDL_Archive *archive);
static long Dir_Tell(SDL_Archive *archive);
static int Dir_EOF(SDL_Archive *archive);
static int Dir_Read(SDL_Archive *archive, void *mem, const int size, const int maxnum);
static int Dir_Seek(SDL_Archive *archive, const long offset, const int whence);
static int Dir_Close(SDL_Archive *archive);
static int Dir_LoadDirectory(DIRECTORY_DATA *data);
static int Dir_SetTempPath(DIRECTORY_DATA *data, char *filename);


int Archive_IsDirectory(const char *file)
#ifdef WIN32
{
	char *temp;
	WIN32_FIND_DATA findFile;
	HANDLE handle;
	
	temp = malloc(strlen(file) + strlen("\\*.*") + 1);
	if (temp == NULL) {
		return 0;
	}
	
	strcpy(temp, file);
	strcat(temp, "\\*.*");
	handle = FindFirstFile(temp, &findFile);
	if (handle == INVALID_HANDLE_VALUE) {
		free(temp);
		return 0;
	}
	FindClose(handle);
	free(temp);
	
	return 1;
}
#else
{
	DIR *dir;
	
	dir = opendir(file);
	if (dir == NULL) {
		return 0;
	} else {
		closedir(dir);
		return 1;
	}
}
#endif


SDL_Archive *Archive_FromDirectory(const char *file)
{
	SDL_Archive *archive;
	
	archive = Archive_Alloc();
	if (archive == NULL) {
		return NULL;
	}
	
	archive->data = Dir_Create(file);
	if (archive->data == NULL) {
		Archive_FreeMainContext(archive);
		return NULL;
	}
	
	archive->open = Dir_Open;
	archive->numopen = Dir_NumOpen;
	archive->name2index = Dir_NameToIndex;
	archive->filenum = Dir_FileNumber;
	archive->finish = Dir_Finish;
	archive->get_char = Dir_GetChar;
	archive->read = Dir_Read;
	archive->seek = Dir_Seek;
	archive->size = Dir_Size;
	archive->tell = Dir_Tell;
	archive->eof = Dir_EOF;
	archive->close = Dir_Close;
	
	return archive;
}



int Dir_SetTempPath(DIRECTORY_DATA *data, char *filename)
{
	int allocSize;
	int needSize;
	char *tempMem;
	
	/* directory name + delimiter + file name + '\0' */
	needSize = strlen(data->dirname) + 1 + strlen(filename) + 1;
	allocSize = data->tempStringAllocated;
	while (allocSize < needSize) {
		allocSize *= 2;
	}
	
	if (allocSize > data->tempStringAllocated) {
		tempMem = realloc(data->tempString, allocSize);
		if (tempMem == NULL) {
			return -1;
		}
		data->tempStringAllocated = allocSize;
		data->tempString = tempMem;
	}
	
	strcpy(data->tempString, data->dirname);
#ifdef WIN32
	strcat(data->tempString, "\\");
#else
	strcat(data->tempString, "/");
#endif
	strcat(data->tempString, filename);
	
	return 0;
}


int Dir_LoadDirectory(DIRECTORY_DATA *data)
#ifdef WIN32
{
	WIN32_FIND_DATA findFile;
	HANDLE handle;
	FILE *fp;
	void *tempptr;
	int ret;
	
	ret = Dir_SetTempPath(data, "*.*");
	if (ret != 0) {
		return 0;
	}
	handle = FindFirstFile(data->tempString, &findFile);
	if (handle == INVALID_HANDLE_VALUE) {
		return 0;
	}
	do {
		if (findFile.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
			continue;
		}
		
		ret = Dir_SetTempPath(data, findFile.cFileName);
		if (ret != 0) {
			break;
		}
		fp = fopen(data->tempString, "rb");
		if (fp == NULL) {
			continue;
		}
		fclose(fp);
		
		if (data->chainNum + 1 > data->chainAllocated) {
			data->chainAllocated = data->chainAllocated * 2;
			tempptr = realloc(data->chain,
				sizeof(FILE_CHAIN) * data->chainAllocated);
			if (tempptr == NULL) {
				break;
			}
			data->chain = (FILE_CHAIN*)tempptr;
		}
		
		data->chain[data->chainNum].filename =
			malloc(strlen(findFile.cFileName) + 1);
		if (data->chain[data->chainNum].filename == NULL) {
			break;
		}
		strcpy(data->chain[data->chainNum].filename, findFile.cFileName);
		data->chain[data->chainNum].size = findFile.nFileSizeLow;
		data->chainNum++;
	} while(FindNextFile(handle, &findFile));
	
	FindClose(handle);
	
	return data->chainNum;
}
#else
{
	DIR *dir;
	FILE *fp;
	void *tempptr;
	struct dirent *ent;
	long filesize;
	int needSize;
	int ret;
	
	dir = opendir(data->dirname);
	if (dir == NULL) {
		return 0;
	}
	while ((ent = readdir(dir)) != NULL) {
		if (strcmp(ent->d_name, ".") == 0 || strcmp(ent->d_name, "..") == 0) {
			continue;
		}
		
		ret = Dir_SetTempPath(data, ent->d_name);
		if (ret != 0) {
			break;
		}
		fp = fopen(data->tempString, "rb");
		if (fp == NULL) {
			continue;
		}
		fseek(fp, 0L, SEEK_END);
		filesize = ftell(fp);
		fclose(fp);
		
		if (data->chainNum + 1 > data->chainAllocated) {
			data->chainAllocated = data->chainAllocated * 2;
			tempptr = realloc(data->chain,
				sizeof(FILE_CHAIN) * data->chainAllocated);
			if (tempptr == NULL) {
				break;
			}
			data->chain = (FILE_CHAIN*)tempptr;
		}
		
		data->chain[data->chainNum].filename =
			malloc(strlen(ent->d_name) + 1);;
		if (data->chain[data->chainNum].filename == NULL) {
			break;
		}
		strcpy(data->chain[data->chainNum].filename, ent->d_name);
		data->chain[data->chainNum].size = filesize;
		data->chainNum++;
	}
	
	closedir(dir);
	
	return data->chainNum;
}
#endif


DIRECTORY_DATA *Dir_Create(const char *file)
{
	DIRECTORY_DATA *data;
	int needSize;
	int ret;
	
	data = (DIRECTORY_DATA*)malloc(sizeof(DIRECTORY_DATA));
	if (data == NULL) {
		return NULL;
	}
	
	data->dirname = (char*)malloc(strlen(file) + 1);
	if (data->dirname == NULL) {
		free(data);
		return NULL;
	}
	strcpy(data->dirname, file);
	
	/* directory name + delimiter + '\0' */
	needSize = strlen(data->dirname) + 1 + 1;
	data->tempString = (char*)malloc(needSize);
	if (data->tempString == NULL) {
		free(data->dirname);
		free(data);
		return NULL;
	}
	data->tempStringAllocated = needSize;
	
	data->chain = (FILE_CHAIN*)malloc(sizeof(FILE_CHAIN) * 1);
	if (data->chain == NULL) {
		free(data->tempString);
		free(data->dirname);
		free(data);
		return NULL;
	}
	
	data->fp = NULL;
	data->chainNum = 0;
	data->chainAllocated = 1;
	
	ret = Dir_LoadDirectory(data);
	if (ret == 0) {
		free(data->chain);
		free(data->tempString);
		free(data->dirname);
		free(data);
		return NULL;
	}
	
	return data;
}


void Dir_Finish(SDL_Archive *archive)
{
	int i;
	DIRECTORY_DATA *data;
	
	if (archive != NULL) {
		data = archive->data;
		
		for (i = 0; i < data->chainNum; i++) {
			free(data->chain[i].filename);
		}
		
		free(data->tempString);
		free(data->dirname);
		free(data->chain);
		if (data->fp != NULL) {
			fclose(data->fp);
		}
		free(data);
	}
	Archive_FreeMainContext(archive);
}


int Dir_Open(SDL_Archive *archive, const char *filename)
{
	return Dir_NumOpen(archive, Dir_NameToIndex(archive, filename));
}


int Dir_NumOpen(SDL_Archive *archive, const int num)
{
	DIRECTORY_DATA *data;
	int ret;
	
	data = archive->data;
	
	if (num < 0 || data->chainNum <= num) {
		return ARCHIVE_ERROR_NO_EXIST_FILE;
	}
	
	if (data->fp != NULL) {
		return ARCHIVE_ERROR_ALREADY_OPEN;
	}
	
	ret = Dir_SetTempPath(data, data->chain[num].filename);
	if (ret != 0) {
		return ARCHIVE_ERROR_ALLOCATE;
	}
	data->fp = fopen(data->tempString, "rb");
	if (data->fp == NULL) {
		return ARCHIVE_ERROR_NO_EXIST_FILE;
	}
	
	data->openNumber = num;
	
	return ARCHIVE_SUCCESS;
}


int Dir_NameToIndex(SDL_Archive *archive, const char *filename)
{
	DIRECTORY_DATA *data;
	int i;
	
	data = archive->data;
	
	for (i = 0; i < data->chainNum; i++) {
		if (strcmp(data->chain[i].filename, filename) == 0) {
			return i;
		}
	}
	
	return -1;
}


long Dir_Size(SDL_Archive *archive)
{
	DIRECTORY_DATA *data;
	
	data = archive->data;
	
	return data->chain[data->openNumber].size;
}


int Dir_FileNumber(SDL_Archive *archive)
{
	DIRECTORY_DATA *data;
	
	data = archive->data;
	
	return data->chainNum;
}


int Dir_GetChar(SDL_Archive *archive)
{
	DIRECTORY_DATA *data;
	
	data = archive->data;
	
	return fgetc(data->fp);
}


long Dir_Tell(SDL_Archive *archive)
{
	DIRECTORY_DATA *data;
	
	data = archive->data;
	
	return ftell(data->fp);
}


int Dir_EOF(SDL_Archive *archive)
{
	DIRECTORY_DATA *data;
	
	data = archive->data;
	
	return feof(data->fp);
}


int Dir_Read(SDL_Archive *archive, void *mem, const int size, const int maxnum)
{
	DIRECTORY_DATA *data;
	
	data = archive->data;
	
	return fread(mem, size, maxnum, data->fp);
}


int Dir_Seek(SDL_Archive *archive, const long offset, const int whence)
{
	DIRECTORY_DATA *data;
	
	data = archive->data;
	
	return fseek(data->fp, offset, whence);
}


int Dir_Close(SDL_Archive *archive)
{
	DIRECTORY_DATA *data;
	
	data = archive->data;
	
	fclose(data->fp);
	data->fp = NULL;
	data->openNumber = -1;
	
	return ARCHIVE_SUCCESS;
}
