/*
    SDL_archive
    Copyright (C) 2004  Kazunori Itoyanagi

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Kazunori Itoyanagi
    itkz@users.sourceforge.jp
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#ifdef WIN32
#  include <windows.h>
#else
#  include <dirent.h>
#endif

#include "SDL_archive.h"


typedef struct _file_chain
{
	long size;
	char *fileName;
} FILE_CHAIN;


typedef struct _directory_DATA
{
	FILE *fp;
	
	FILE_CHAIN *chain;
	int chainNum;
	int chainAllocated;
	
	int tempPathBufAllocated;
	char *tempPathBuf;
	
	int openNumber;
	char *dirName;
	
	DirContainer *dirContainer;
} DIRECTORY_DATA;


static DIRECTORY_DATA *Dir_Create(const char *file);
static int Dir_GetFileNumByName(SDL_Archive *archive, const char *fileName);
static int Dir_OpenFile(SDL_Archive *archive, const char *fileName);
static int Dir_GetChildDirNum(SDL_Archive *archive);
static char *Dir_GetChildDirName(SDL_Archive *archive, int dirNum);
static int Dir_GoDownDir(SDL_Archive *archive, const char *dirName);
static int Dir_GoUpDir(SDL_Archive *archive);
static int Dir_MoveDirByPath(SDL_Archive *archive, const char *path);
static char *Dir_GetCurrentDirName(SDL_Archive *archive);
static char *Dir_GetCurrentDirPath(SDL_Archive *archive);
static long Dir_Size(SDL_Archive *archive);
static int Dir_GetFileNumber(SDL_Archive *archive);
static char *Dir_GetFileName(SDL_Archive *archive, int num);
static void Dir_Free(DIRECTORY_DATA *dir);
static void Dir_Finish(SDL_Archive *archive);
static int Dir_GetChar(SDL_Archive *archive);
static long Dir_Tell(SDL_Archive *archive);
static int Dir_EOF(SDL_Archive *archive);
static int Dir_Read(SDL_Archive *archive, void *mem, const int size, const int maxnum);
static int Dir_Seek(SDL_Archive *archive, const long offset, const int whence);
static int Dir_Close(SDL_Archive *archive);
static int Dir_LoadDirectory(DIRECTORY_DATA *dir);
static int Dir_SetTempPath(DIRECTORY_DATA *dir, const char *fileName);
static SDL_Archive *Dir_Clone(SDL_Archive *archive);

static int Dir_ReallocChain(DIRECTORY_DATA *dir);
static int Dir_AddFile(DIRECTORY_DATA *dir, const char *fileName, int fileSize);
static int Dir_AddDirectory(DIRECTORY_DATA *dir, const char *dirName);
static int Dir_IsValidDir(const char *name);



int Archive_IsDirectory(const char *file)
#ifdef WIN32
{
	char *temp;
	WIN32_FIND_DATA findFile;
	HANDLE handle;
	
	if (Dir_IsValidDir(file) == SDL_FALSE) {
		return SDL_FALSE;
	}
	
	temp = malloc(strlen(file) + strlen("\\*.*") + 1);
	if (temp == NULL) {
		return SDL_FALSE;
	}
	
	strcpy(temp, file);
	strcat(temp, "\\*.*");
	handle = FindFirstFile(temp, &findFile);
	if (handle == INVALID_HANDLE_VALUE) {
		free(temp);
		return SDL_FALSE;
	}
	FindClose(handle);
	free(temp);
	
	return SDL_TRUE;
}
#else
{
	DIR *dir;
	
	if (Dir_IsValidDir(file) == SDL_FALSE) {
		return SDL_FALSE;
	}
	
	dir = opendir(file);
	if (dir == NULL) {
		return SDL_FALSE;
	} else {
		closedir(dir);
		return SDL_TRUE;
	}
}
#endif


SDL_Archive *Archive_FromDirectory(const char *file)
{
	SDL_Archive *archive;
	
	if (Dir_IsValidDir(file) == SDL_FALSE) {
		return NULL;
	}
	
	archive = Archive_Alloc();
	if (archive == NULL) {
		return NULL;
	}
	
	archive->data = Dir_Create(file);
	if (archive->data == NULL) {
		Archive_FreeMainContext(archive);
		return NULL;
	}
	
	archive->getChildDirNum = Dir_GetChildDirNum;
	archive->getChildDirName = Dir_GetChildDirName;
	archive->goDownDir = Dir_GoDownDir;
	archive->goUpDir = Dir_GoUpDir;
	archive->moveDirByPath = Dir_MoveDirByPath;
	archive->getCurrentDirName = Dir_GetCurrentDirName;
	archive->getCurrentDirPath = Dir_GetCurrentDirPath;
	
	archive->openFile = Dir_OpenFile;
	archive->getFileNum = Dir_GetFileNumber;
	archive->getFileName = Dir_GetFileName;
	archive->getFileNumByName = Dir_GetFileNumByName;
	archive->finish = Dir_Finish;
	
	archive->clone = Dir_Clone;
	
	archive->getChar = Dir_GetChar;
	archive->read = Dir_Read;
	archive->seek = Dir_Seek;
	archive->size = Dir_Size;
	archive->tell = Dir_Tell;
	archive->eof = Dir_EOF;
	archive->close = Dir_Close;
	
	return archive;
}



int Dir_IsValidDir(const char *name)
{
	int i;
	int length;
	
	if (strcmp(name, "..") == 0 || strcmp(name, ".") == 0) {
		return SDL_FALSE;
	}
	
	length = strlen(name);
	for (i = 0; i < length; i++) {
		if (name[i] == '/' || name[i] == '\\') {
			return SDL_FALSE;
		}
	}
	
	return SDL_TRUE;
}


int Dir_SetTempPath(DIRECTORY_DATA *dir, const char *fileName)
{
	int allocSize;
	int needSize;
	char *tempMem;
	char *currentPath;
	int fileNameLength;
	
	currentPath = DirContainer_GetCurrentPath(dir->dirContainer);
	currentPath = currentPath + 1; /* next '/' */
	
	if (fileName) {
		fileNameLength = strlen(fileName);
	} else {
		fileNameLength = 0;
	}
	/*
	  dir->dirName + delimiter +
	  dirName + delimiter +
	  fileName + '\0'
	 */
	needSize =
		strlen(dir->dirName) + 1 +
		strlen(currentPath) + 1 +
		fileNameLength + 1;
	allocSize = dir->tempPathBufAllocated;
	while (allocSize < needSize) {
		allocSize *= 2;
	}
	
	if (allocSize > dir->tempPathBufAllocated) {
		tempMem = realloc(dir->tempPathBuf, allocSize);
		if (tempMem == NULL) {
			return ARCHIVE_ERROR_ALLOCATE;
		}
		dir->tempPathBufAllocated = allocSize;
		dir->tempPathBuf = tempMem;
	}
	
	strcpy(dir->tempPathBuf, dir->dirName);
	
	if (currentPath != NULL && currentPath[0] != '\0') {
		strcat(dir->tempPathBuf, "/");
		strcat(dir->tempPathBuf, currentPath);
	}
	
	if (fileName) {
		strcat(dir->tempPathBuf, "/");
		strcat(dir->tempPathBuf, fileName);
	}
	
#ifdef WIN32
	{
		int i;
		int stringLength;
		stringLength = strlen(dir->tempPathBuf);
		for (i = 0; i < stringLength; i++) {
			if (dir->tempPathBuf[i] == '/') {
				dir->tempPathBuf[i] = '\\';
			}
		}
	}
#endif
	
	return ARCHIVE_SUCCESS;
}


int Dir_ReallocChain(DIRECTORY_DATA *dir)
{
	int needSize;
	void *tempptr;
	
	if (dir->chainNum >= dir->chainAllocated) {
		needSize = dir->chainAllocated * 2;
		tempptr = realloc(
			dir->chain,
			sizeof(FILE_CHAIN) * needSize);
		if (tempptr == NULL) {
			return ARCHIVE_ERROR_ALLOCATE;
		}
		dir->chain = (FILE_CHAIN*)tempptr;
		dir->chainAllocated = needSize;
	}
	
	return ARCHIVE_SUCCESS;
}


int Dir_AddFile(DIRECTORY_DATA *dir, const char *fileName, int fileSize)
{
	int result;
	
	result = Dir_ReallocChain(dir);
	if (result != ARCHIVE_SUCCESS) {
		return result;
	}
	
	dir->chain[dir->chainNum].fileName =
		malloc(strlen(fileName) + 1);
	if (dir->chain[dir->chainNum].fileName == NULL) {
		return ARCHIVE_ERROR_ALLOCATE;
	}
	strcpy(dir->chain[dir->chainNum].fileName, fileName);
	dir->chain[dir->chainNum].size = fileSize;
	DirContainer_AddFile(dir->dirContainer, dir->chainNum);
	dir->chainNum++;
	
	return ARCHIVE_SUCCESS;
}


int Dir_AddDirectory(DIRECTORY_DATA *dir, const char *dirName)
{
	int result;
	
	result = DirContainer_AddDirectory(
		dir->dirContainer, dirName);
	if (result != ARCHIVE_SUCCESS) {
		return result;
	}
	result = DirContainer_GoDownDir(
		dir->dirContainer, dirName);
	if (result != ARCHIVE_SUCCESS) {
		return result;
	}
	result = Dir_LoadDirectory(dir);
	if (result != ARCHIVE_SUCCESS) {
		return result;
	}
	result = DirContainer_GoUpDir(dir->dirContainer);
	if (result != ARCHIVE_SUCCESS) {
		return result;
	}
	
	return ARCHIVE_SUCCESS;
}


int Dir_LoadDirectory(DIRECTORY_DATA *dir)
#ifdef WIN32
{
	WIN32_FIND_DATA findFile;
	HANDLE handle;
	int result;
	
	result = Dir_SetTempPath(dir, "*");
	if (result != ARCHIVE_SUCCESS) {
		return result;
	}
	handle = FindFirstFile(dir->tempPathBuf, &findFile);
	if (handle == INVALID_HANDLE_VALUE) {
		return ARCHIVE_ERROR_INVALID_NAME;
	}
	do {
		if (
			strcmp(findFile.cFileName, "..") == 0 ||
			strcmp(findFile.cFileName, ".") == 0
		) {
			continue;
		}
		if (findFile.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
			result = Dir_AddDirectory(dir, findFile.cFileName);
			if (result != ARCHIVE_SUCCESS) {
				return result;
			}
		} else {
			result = Dir_SetTempPath(dir, findFile.cFileName);
			if (result != ARCHIVE_SUCCESS) {
				break;
			}
			
			result = Dir_AddFile(dir, findFile.cFileName, findFile.nFileSizeLow);
			if (result != ARCHIVE_SUCCESS) {
				break;
			}
		}
	} while(FindNextFile(handle, &findFile));
	
	FindClose(handle);
	
	return ARCHIVE_SUCCESS;
}
#else
{
	DIR *dir;
	DIR *tempDir;
	FILE *fp;
	struct dirent *ent;
	long fileSize;
	int result;
	
	result = Dir_SetTempPath(dir, NULL);
	if (result != ARCHIVE_SUCCESS) {
		return result;
	}
	dir = opendir(dir->tempPathBuf);
	if (dir == NULL) {
		return ARCHIVE_ERROR_INVALID_NAME;
	}
	while ((ent = readdir(dir)) != NULL) {
		if (
			strcmp(ent->d_name, "..") == 0 ||
			strcmp(ent->d_name, ".") == 0
		) {
			continue;
		}
		
		result = Dir_SetTempPath(dir, ent->d_name);
		if (result != ARCHIVE_SUCCESS) {
			return result;
		}
		tempDir = opendir(dir->tempPathBuf);
		if (tempDir) {
			closedir(tempDir);
			result = Dir_AddDirectory(dir, ent->d_name);
			if (result != ARCHIVE_SUCCESS) {
				return result;
			}
		} else {
			result = Dir_SetTempPath(dir, ent->d_name);
			if (result != ARCHIVE_SUCCESS) {
				break;
			}
			fp = fopen(dir->tempPathBuf, "rb");
			if (fp == NULL) {
				continue;
			}
			fseek(fp, 0L, SEEK_END);
			fileSize = ftell(fp);
			fclose(fp);
			
			result = Dir_AddFile(dir, ent->d_name, fileSize);
			if (result != ARCHIVE_SUCCESS) {
				break;
			}
		}
	}
	
	closedir(dir);
	
	return ARCHIVE_SUCCESS;
}
#endif


DIRECTORY_DATA *Dir_Create(const char *file)
{
	DIRECTORY_DATA *dir;
	int needSize;
	int result;
	
	dir = (DIRECTORY_DATA*)malloc(sizeof(DIRECTORY_DATA));
	if (dir == NULL) {
		return NULL;
	}
	
	dir->dirContainer = NULL;
	dir->tempPathBuf = NULL;
	dir->dirName = NULL;
	dir->fp = NULL;
	dir->chain = NULL;
	
	dir->dirName = (char*)malloc(strlen(file) + 1);
	if (dir->dirName == NULL) {
		Dir_Free(dir);
		return NULL;
	}
	strcpy(dir->dirName, file);
	
	/* directory name + delimiter + '\0' */
	needSize = strlen(dir->dirName) + 1 + 1;
	dir->tempPathBuf = (char*)malloc(needSize);
	if (dir->tempPathBuf == NULL) {
		Dir_Free(dir);
		return NULL;
	}
	dir->tempPathBufAllocated = needSize;
	
	dir->dirContainer = DirContainer_Create();
	if (dir->dirContainer == NULL) {
		Dir_Free(dir);
		return NULL;
	}
	
	dir->chain = (FILE_CHAIN*)malloc(sizeof(FILE_CHAIN) * 1);
	if (dir->chain == NULL) {
		Dir_Free(dir);
		return NULL;
	}
	dir->fp = NULL;
	dir->chainNum = 0;
	dir->chainAllocated = 1;
	
	result = Dir_LoadDirectory(dir);
	if (result != ARCHIVE_SUCCESS) {
		Dir_Free(dir);
		return NULL;
	}
	
	return dir;
}


SDL_Archive *Dir_Clone(SDL_Archive *orgArchive)
{
	SDL_Archive *archive;
	DIRECTORY_DATA *dir;
	DIRECTORY_DATA *orgDir;
	int i;
	int needSize;
	
	orgDir = (DIRECTORY_DATA*)orgArchive->data;
	
	dir = malloc(sizeof(DIRECTORY_DATA));
	if (dir == NULL) {
		return NULL;
	}
	
	dir->dirContainer = NULL;
	dir->tempPathBuf = NULL;
	dir->dirName = NULL;
	dir->fp = NULL;
	dir->chain = NULL;
	
	dir->dirName = (char*)malloc(strlen(orgDir->dirName) + 1);
	if (dir->dirName == NULL) {
		Dir_Free(dir);
		return NULL;
	}
	strcpy(dir->dirName, orgDir->dirName);
	
	/* directory name + delimiter + '\0' */
	needSize = strlen(dir->dirName) + 1 + 1;
	dir->tempPathBuf = (char*)malloc(needSize);
	if (dir->tempPathBuf == NULL) {
		Dir_Free(dir);
		return NULL;
	}
	dir->tempPathBufAllocated = needSize;
	
	dir->chain = (FILE_CHAIN*)malloc(
		sizeof(FILE_CHAIN) * orgDir->chainNum);
	if (dir->chain == NULL) {
		Dir_Free(dir);
		return NULL;
	}
	dir->chainAllocated = orgDir->chainNum;
	
	dir->fp = NULL;
	
	for (i = 0; i < orgDir->chainNum; i++) {
		dir->chain[i].fileName = NULL;
	}
	for (i = 0; i < orgDir->chainNum; i++) {
		dir->chain[i].fileName = (char*)malloc(
			strlen(orgDir->chain[i].fileName) + 1);
		if (dir->chain[i].fileName == NULL) {
			Dir_Free(dir);
			return NULL;
		}
		strcpy(dir->chain[i].fileName, orgDir->chain[i].fileName);
		dir->chain[i].size = orgDir->chain[i].size;
	}
	dir->chainNum = orgDir->chainNum;
	
	dir->dirContainer = DirContainer_Clone(orgDir->dirContainer);
	if (dir->dirContainer == NULL) {
		Dir_Free(dir);
		return NULL;
	}
	
	dir->openNumber = -1;
	
	archive = Archive_Alloc();
	if (archive == NULL) {
		Dir_Free(dir);
		return NULL;
	}
	
	archive->data = dir;
	
	archive->getChildDirNum = Dir_GetChildDirNum;
	archive->getChildDirName = Dir_GetChildDirName;
	archive->goDownDir = Dir_GoDownDir;
	archive->goUpDir = Dir_GoUpDir;
	archive->moveDirByPath = Dir_MoveDirByPath;
	archive->getCurrentDirName = Dir_GetCurrentDirName;
	archive->getCurrentDirPath = Dir_GetCurrentDirPath;
	
	archive->openFile = Dir_OpenFile;
	archive->getFileNum = Dir_GetFileNumber;
	archive->getFileName = Dir_GetFileName;
	archive->getFileNumByName = Dir_GetFileNumByName;
	archive->finish = Dir_Finish;
	
	archive->clone = Dir_Clone;
	
	archive->getChar = Dir_GetChar;
	archive->read = Dir_Read;
	archive->seek = Dir_Seek;
	archive->size = Dir_Size;
	archive->tell = Dir_Tell;
	archive->eof = Dir_EOF;
	archive->close = Dir_Close;
	
	return archive;
}


void Dir_Free(DIRECTORY_DATA *dir)
{
	int i;
	
	if (dir->dirContainer) {
		DirContainer_Free(dir->dirContainer);
	}
	if (dir->tempPathBuf) {
		free(dir->tempPathBuf);
	}
	if (dir->dirName) {
		free(dir->dirName);
	}
	if (dir->fp != NULL) {
		fclose(dir->fp);
	}
	
	if (dir->chain) {
		for (i = 0; i < dir->chainNum; i++) {
			free(dir->chain[i].fileName);
		}
		free(dir->chain);
	}
	
	free(dir);
}


void Dir_Finish(SDL_Archive *archive)
{
	if (archive != NULL) {
		Dir_Free((DIRECTORY_DATA*)archive->data);
	}
	Archive_FreeMainContext(archive);
}


int Dir_GetFileNumByName(SDL_Archive *archive, const char *fileName)
{
	DIRECTORY_DATA *dir;
	int i;
	
	dir = archive->data;
	
	for (i = 0; i < dir->chainNum; i++) {
		if (strcmp(dir->chain[i].fileName, fileName) == 0) {
			return i;
		}
	}
	
	return -1;
}


int Dir_OpenFile(SDL_Archive *archive, const char *fileName)
{
	DIRECTORY_DATA *dir;
	int result;
	int num;
	
	dir = archive->data;
	
	num = Dir_GetFileNumByName(archive, fileName);
	
	if (num < 0 || dir->chainNum <= num) {
		return ARCHIVE_ERROR_NO_EXIST_FILE;
	}
	
	if (dir->fp != NULL) {
		return ARCHIVE_ERROR_ALREADY_OPEN;
	}
	
	result = Dir_SetTempPath(dir, dir->chain[num].fileName);
	if (result != 0) {
		return result;
	}
	dir->fp = fopen(dir->tempPathBuf, "rb");
	if (dir->fp == NULL) {
		return ARCHIVE_ERROR_NO_EXIST_FILE;
	}
	
	dir->openNumber = num;
	
	return ARCHIVE_SUCCESS;
}


int Dir_GetChildDirNum(SDL_Archive *archive)
{
	DIRECTORY_DATA *dir;
	
	dir = (DIRECTORY_DATA*)archive->data;
	
	return DirContainer_GetChildDirNum(dir->dirContainer);
}


char *Dir_GetChildDirName(SDL_Archive *archive, int dirNum)
{
	DIRECTORY_DATA *dir;
	
	dir = (DIRECTORY_DATA*)archive->data;
	
	return DirContainer_GetChildDirName(dir->dirContainer, dirNum);
}


int Dir_GoDownDir(SDL_Archive *archive, const char *dirName)
{
	DIRECTORY_DATA *dir;
	
	dir = (DIRECTORY_DATA*)archive->data;
	
	return DirContainer_GoDownDir(dir->dirContainer, dirName);
}


int Dir_GoUpDir(SDL_Archive *archive)
{
	DIRECTORY_DATA *dir;
	
	dir = (DIRECTORY_DATA*)archive->data;
	
	return DirContainer_GoUpDir(dir->dirContainer);
}


int Dir_MoveDirByPath(SDL_Archive *archive, const char *path)
{
	DIRECTORY_DATA *dir;
	
	dir = (DIRECTORY_DATA*)archive->data;
	
	return DirContainer_ParsePath(dir->dirContainer, path, DIRECTORY_MOVE_DIR, NULL);
}


char *Dir_GetCurrentDirName(SDL_Archive *archive)
{
	DIRECTORY_DATA *dir;
	
	dir = (DIRECTORY_DATA*)archive->data;
	
	return DirContainer_GetCurrentDirName(dir->dirContainer);
}


char *Dir_GetCurrentDirPath(SDL_Archive *archive)
{
	DIRECTORY_DATA *dir;
	
	dir = (DIRECTORY_DATA*)archive->data;
	
	return DirContainer_GetCurrentPath(dir->dirContainer);
}


int Dir_GetFileNumber(SDL_Archive *archive)
{
	DIRECTORY_DATA *dir;
	
	dir = (DIRECTORY_DATA*)archive->data;
	
	return DirContainer_GetFileNum(dir->dirContainer);
}


char *Dir_GetFileName(SDL_Archive *archive, int num)
{
	DIRECTORY_DATA *dir;
	int fileIndex;
	
	dir = (DIRECTORY_DATA*)archive->data;
	fileIndex = DirContainer_GetFileIndex(dir->dirContainer, num);
	
	return dir->chain[fileIndex].fileName;
}


long Dir_Size(SDL_Archive *archive)
{
	DIRECTORY_DATA *dir;
	
	dir = archive->data;
	
	return dir->chain[dir->openNumber].size;
}


int Dir_GetChar(SDL_Archive *archive)
{
	DIRECTORY_DATA *dir;
	
	dir = archive->data;
	
	return fgetc(dir->fp);
}


long Dir_Tell(SDL_Archive *archive)
{
	DIRECTORY_DATA *dir;
	
	dir = archive->data;
	
	return ftell(dir->fp);
}


int Dir_EOF(SDL_Archive *archive)
{
	DIRECTORY_DATA *dir;
	
	dir = archive->data;
	
	return feof(dir->fp);
}


int Dir_Read(SDL_Archive *archive, void *mem, const int size, const int maxnum)
{
	DIRECTORY_DATA *dir;
	
	dir = archive->data;
	
	return fread(mem, size, maxnum, dir->fp);
}


int Dir_Seek(SDL_Archive *archive, const long offset, const int whence)
{
	DIRECTORY_DATA *dir;
	
	dir = archive->data;
	
	return fseek(dir->fp, offset, whence);
}


int Dir_Close(SDL_Archive *archive)
{
	DIRECTORY_DATA *dir;
	
	dir = archive->data;
	
	fclose(dir->fp);
	dir->fp = NULL;
	dir->openNumber = -1;
	
	return ARCHIVE_SUCCESS;
}
