package org.maachang.mimdb.core ;

import org.maachang.mimdb.MimdbException;

/**
 * 組み込み用プリコンパイル済みステートメント.
 * 
 * @version 2014/01/16
 * @author masahito suzuki
 * @since MasterInMemDB 1.02
 */
final class EmbeddedPreparedStatement implements MimdbPreparedStatement {
    
    /** コネクションオブジェクト. **/
    private EmbeddedConnection connection = null ;
    
    /** プリコンパイル済みオブジェクト. **/
    private final MimdbQueryPrepared prepared = new MimdbQueryPrepared() ;
    
    /** コンストラクタ. **/
    private EmbeddedPreparedStatement() {
    }
    
    /** コンストラクタ. **/
    protected EmbeddedPreparedStatement( EmbeddedConnection conn,QueryCompileInfo p ) {
        connection = conn ;
        prepared.create( p ) ;
    }
    
    /**
     * オブジェクトクローズ.
     */
    public void close() {
        connection = null ;
        prepared.clear() ;
    }
    
    /**
     * オブジェクトがクローズしているか取得.
     * @return boolean [true]の場合、クローズしています.
     */
    public boolean isClose() {
        return connection == null || connection.isClose() ;
    }
    
    /** チェック処理. **/
    protected void check() {
        if( isClose() ) {
            throw new MimdbException( "オブジェクトはクローズしています" ) ;
        }
    }
    
    /**
     * QueryCompileInfo オブジェクトを取得.
     * @return QueryCompileInfo オブジェクトが返却されます.
     */
    protected QueryCompileInfo getInfo() {
        check() ;
        return prepared.getInfo() ;
    }
    
    /**
     * DB更新IDを取得.
     * この情報が、結果データと一致しない場合は、その結果データは古くなっています.
     * @return int DB更新IDが返却されます.
     */
    public long getDbId() {
        check() ;
        return prepared.getDbId() ;
    }
    
    /**
     * テーブル名を取得.
     * @return String テーブル名が返却されます.
     */
    public String getName() {
        check() ;
        return prepared.getName() ;
    }
    
    /**
     * preparedパラメータ数を取得.
     * @return int パラメータ数が返却されます.
     */
    public int length() {
        check() ;
        return prepared.paramsLength() ;
    }
    
    /**
     * preparedパラメータをクリア.
     * @return MimdbPreparedStatement このオブジェクトが返却されます.
     */
    public MimdbPreparedStatement clearParams() {
        check() ;
        prepared.clearParams() ;
        return this ;
    }
    
    /**
     * preparedパラメータを設定.
     * @param index 対象のパラメータ項番を設定します.
     * @param value 対象の設定情報を設定します.
     * @return MimdbPreparedStatement このオブジェクトが返却されます.
     */
    public MimdbPreparedStatement setParams( int index,Object value ) {
        check() ;
        prepared.setParams( index,value ) ;
        return this ;
    }
    
    /**
     * 指定パラメータ項番のカラム名を取得.
     * @param index 対象のパラメータ項番を設定します.
     * @return String カラム名が返却されます.
     */
    public String getColumnName( int index ) {
        check() ;
        return prepared.paramNoByColumnName( index ) ;
    }
    
    /**
     * 指定パラメータ項番の型タイプを取得.
     * @param index 対象のパラメータ項番を設定します.
     * @return int 型タイプが返却されます.
     */
    public int getType( int index ) {
        check() ;
        return prepared.paramNoByColumnType( index ) ;
    }
    
    /**
     * 指定パラメータがリスト形式での設定が可能かチェック.
     * @param index 対象のパラメータ項番を設定します.
     * @return boolean [true]の場合、リスト形式での設定が可能です.
     */
    public boolean isListParam( int index ) {
        check() ;
        return prepared.isPamraNoByListParam( index ) ;
    }
    
    /**
     * 表示オフセット値を設定.
     * この条件により、表示位置を確定できます.
     * @param off 表示オフセット値を設定します.
     *            [-1]が設定された場合、表示幅は確定されません.
     * @return MimdbPreparedStatement このオブジェクトが返却されます.
     */
    public MimdbPreparedStatement setOffset( int off ) {
        check() ;
        prepared.setOffset( off ) ;
        return this ;
    }
    
    /**
     * 表示リミット値を設定.
     * この条件により、表示位置を確定できます.
     * @param limit 表示リミット値を設定します.
     *            [-1]が設定された場合、表示幅は確定されません.
     * @return MimdbPreparedStatement このオブジェクトが返却されます.
     */
    public MimdbPreparedStatement setLimit( int limit ) {
        check() ;
        prepared.setLimit( limit ) ;
        return this ;
    }
    
    /** オフセット、リミット値をクリア. **/
    protected void clearOffLimit() {
        prepared.setOffset( -1 ) ;
        prepared.setLimit( -1 ) ;
    }
    
    /**
     * フェッチサイズを設定.
     * ※この値は、サーバーモードでの接続のみ有効となります.
     * @param size フェッチサイズを設定します.
     * @return MimdbPreparedStatement このオブジェクトが返却されます.
     */
    public MimdbPreparedStatement setFetchSize( int size ) {
        check() ;
        return this ;
    }
    
    /**
     * フェッチサイズを取得.
     * ※この値は、サーバーモードでの接続のみ有効となります.
     * @return int フェッチサイズが返却されます.
     */
    public int getFetchSize() {
        check() ;
        return -1 ;
    }
    
    /**
     * クエリー実行.
     * @return MimdbResult 実行結果が返却されます.
     * @exception Exception 例外.
     */
    public MimdbResult executeQuery()
        throws Exception {
        check() ;
        return prepared.executeQuery() ;
    }
    
}
