package org.seasar.transaction;

import java.lang.reflect.Method;
import java.util.Map;

import javax.transaction.Status;

import org.seasar.util.Assertion;
import org.seasar.util.Reflector;
import org.seasar.util.SMap;
import org.seasar.util.SeasarException;

public abstract class TransAttribute {

    public final static String SUPPORTS_NAME = "Supports";
    public final static String REQUIRED_NAME = "Required";
    public final static String REQUIRES_NEW_NAME = "RequiresNew";
    public final static String MANDATORY_NAME = "Mandatory";

    public final static TransAttribute SUPPORTS = new Supports();
    public final static TransAttribute REQUIRED = new Required();
    public final static TransAttribute REQUIRES_NEW = new RequiresNew();
    public final static TransAttribute MANDATORY = new Mandatory();

    private static Map _transAttributes = new SMap();
    
    private String _name;

    static {
    	_transAttributes.put(null, SUPPORTS);
        _transAttributes.put(SUPPORTS_NAME, SUPPORTS);
        _transAttributes.put(REQUIRED_NAME, REQUIRED);
        _transAttributes.put(REQUIRES_NEW_NAME, REQUIRES_NEW);
        _transAttributes.put(MANDATORY_NAME, MANDATORY);
    }

    protected TransAttribute(final String name) {
        _name = name;
    }

    public static TransAttribute getInstance(final String name) {
        TransAttribute type = (TransAttribute) _transAttributes.get(name);
        Assertion.assertFound("name", type);

        return type;
    }
    
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        } else if (o == null || !getClass().isInstance(o)) {
            return false;
        } else {
            return _name.equals(((TransAttribute) o)._name);
        }
    }
    
    public int hashCode() {
    	return _name.hashCode();
    }

    public abstract Object invoke(Method method, Object target, Object[] args)
             throws SeasarException;

    private static final class Supports extends TransAttribute {

        protected Supports() {
            super(SUPPORTS_NAME);
        }

        public final Object invoke(Method method, Object target, Object[] args)
                throws SeasarException {

            return Reflector.invoke(method, target, args);
        }
    }

    private static final class Required extends TransAttribute {

        protected Required() {
            super(REQUIRED_NAME);
        }

        public final Object invoke(Method method, Object target, Object[] args)
                throws SeasarException {

            TransactionManagerImpl tm = TransactionManagerImpl.getInstance();
			boolean began = tm.preRequiredProcess();
            try {
                Object ret = Reflector.invoke(method, target, args);
                tm.postNormalRequiredProcess(began);
                return ret;
            } catch (Throwable t) {
				tm.postExceptionRequiredProcess(began);
				throw SeasarException.convertSeasarException(t);
			}
        }
    }

    private static final class RequiresNew extends TransAttribute {

        protected RequiresNew() {
            super(REQUIRES_NEW_NAME);
        }

        public final Object invoke(Method method, Object target, Object[] args)
                throws SeasarException {

            TransactionManagerImpl tm = TransactionManagerImpl.getInstance();
			TransactionImpl preTx = tm.preRequiresNewProcess();
            try {
                Object ret = Reflector.invoke(method, target, args);
                tm.postNormalRequiresNewProcess();
                return ret;
            } catch (Throwable t) {
				tm.postExceptionRequiresNewProcess();
				throw SeasarException.convertSeasarException(t);
			} finally {
				tm.finalRequiresNewProcess(preTx);
            }
        }
    }

    private static final class Mandatory extends TransAttribute {

        protected Mandatory() {
            super(MANDATORY_NAME);
        }

        public final Object invoke(Method method, Object target, Object[] args)
                throws SeasarException {

			TransactionManagerImpl tm = TransactionManagerImpl.getInstance();
            if (tm.getStatus() == Status.STATUS_NO_TRANSACTION) {
                throw new SeasarException("ESSR0311");
            }
            return Reflector.invoke(method, target, args);
        }
    }
}