package LISM::Storage::GoogleApps;

use strict;
use base qw(LISM::Storage);
use Net::LDAP::Constant qw(:all);
use Net::LDAP::Filter;
use VUser::Google::ProvisioningAPI;
use Encode;
use Data::Dumper;

=head1 NAME

LISM::Storage::GoogleApps - Google Apps storage for LISM

=head1 DESCRIPTION

This class implements the L<LISM::Storage> interface for Google Apps data.

=head1 METHODS

=head2 init

Initialize the configuration data.

=cut

sub init
{
    my $self = shift;

    return $self->SUPER::init();
}

sub _getConnect
{
    my $self = shift;
    my $conf = $self->{_config};

    if (defined($self->{google})) {
        if ($self->{google}->IsAuthenticated()) {
            return 0;
        } else {
            $self->log(level => 'err', message => "Authentication to Google Apps failed");
            undef($self->{google});
        }
    }

    $self->{google} = VUser::Google::ProvisioningAPI->new($conf->{domain}[0], $conf->{admin}[0], $conf->{passwd}[0], '2.0');
    if (!$self->{google}->IsAuthenticated) {
        $self->log(level => 'alert', message => "Authentication to Google Apps failed");
        return -1;
    }

    return 0;
}

sub _checkConfig
{
    my $self = shift;
    my $conf = $self->{_config};
    my $rc = 0;

    foreach my $oname (keys %{$conf->{object}}) {
        if ($oname !~ /^User$/) {
            $self->log(level => 'alert', message => "Invalid Google Apps object $oname");
            return 1
        }
        my $oconf = $conf->{object}{$oname};
        my $param;
        if ($oname eq 'User') {
            $param = 'User';
        }
        $oconf->{attr}{$oconf->{rdn}[0]}->{param}[0] = $param;
    }

    if ($rc = $self->SUPER::_checkConfig()) {
        return $rc;
    }

    return $rc;
}

=pod

=head2 _objSearch($obj, $pkey, $suffix, $sizeLim, $timeLim, $filter)

Search the appropriate records in the object's file.

=cut

sub _objSearch
{
    my $self = shift;
    my ($obj, $pkey, $suffix, $sizeLim, $filter) = @_;
    my $conf = $self->{_config};
    my $oconf = $obj->{conf};
    my @match_entries = ();
    my @match_keys;
    my $rc = LDAP_SUCCESS;

    my $filterStr = $filter->as_string;
    my ($rdn_val) = ($filterStr =~ /$oconf->{rdn}[0]=(.[^\)]*)/i);

    if ($obj->{name} eq 'User') {
        if ($rdn_val && $rdn_val !~ /\*/) {
            my $entry = $self->{google}->RetrieveUser($rdn_val);
            if (!defined($entry)) {
                if ($self->{google}->{result}{reason} !~ / 1301 -/) {
                    $self->log(level => 'err', message => "Searching User $rdn_val in Google Apps failed: ".$self->{google}->{result}{reason});
                    $rc = LDAP_OPERATIONS_ERROR;
                }
            } else {
                my $entryStr = $self->_buildGoogleEntry($obj, $entry);
                if (!Encode::is_utf8($entryStr)) {
                    $entryStr = decode('utf8', $entryStr);
                }

                if ($self->parseFilter($filter, $entryStr)) {
                    push(@match_entries, $entryStr);
                }
            }
        } else {
            my @entries = $self->{google}->RetrieveAllUsers();
            if (!defined(@entries)) {
                $self->log(level => 'err', message => "Searching Google Apps failed: ".$self->{google}->{result}{reason});
                $rc = LDAP_OPERATIONS_ERROR;
            } else {
                foreach my $entry (@entries) {
                    my $entryStr = $self->_buildGoogleEntry($obj, $entry);
                    if (Encode::is_utf8($entryStr)) {
                        $entryStr = decode('utf8', $entryStr);
                    }

                    if ($self->parseFilter($filter, $entryStr)) {
                        push(@match_entries, $entryStr);
                    }
                }
            }
        }
    }

    return ($rc , \@match_keys, @match_entries);
}

=pod

=head2 _objModify($obj, $pkey, $dn, @list)

Write the modified record to the temporary file.

=cut

sub _objModify
{
    my $self = shift;
    my ($obj, $pkey, $dn, @list) = @_;
    my $conf = $self->{_config};
    my $oconf = $obj->{conf};
    my $rc = LDAP_SUCCESS;

    my ($rdn_val) = ($dn =~ /^[^=]+=([^,]+),/);

    my $update_entry;
    if ($obj->{name} eq 'User') {
        $update_entry = VUser::Google::ProvisioningAPI::V2_0::UserEntry->new();;
    }

    while ( @list > 0 && !$rc) {
        my $action = shift @list;
        my $attr    = lc(shift @list);
        my @values;

        while (@list > 0 && $list[0] ne "ADD" && $list[0] ne "DELETE" && $list[0] ne "REPLACE") {
            push(@values, shift @list);
        }

        if (!defined($oconf->{attr}{$attr})) {
            next;
        }

        # can't modify the attribute for rdn
        if ($attr eq $oconf->{rdn}[0]) {
            $rc =  LDAP_CONSTRAINT_VIOLATION;
            last;
        }

        my $param = $oconf->{attr}{$attr}->{param}[0];
        if($action eq "ADD") {
            $update_entry->$param($values[0]);
        } elsif($action eq "DELETE") {
            $update_entry->$param('');
        } elsif($action eq "REPLACE") {
            $update_entry->$param($values[0]);
        }
    }

    if ($rc) {
        return $rc;
    }

    if ($obj->{name} eq 'User') {
        my $entry = $self->{google}->UpdateUser($rdn_val, $update_entry);
        if (!defined($entry)) {
            $self->log(level => 'err', message => "Modifying User $rdn_val to Google Apps failed: ".$self->{google}->{result}{reason});
            if ($self->{google}->{result}{reason} =~ / 1301 -/) {
                $rc = LDAP_NO_SUCH_OBJECT;
            } else {
                $rc = LDAP_OPERATIONS_ERROR;
            }
        }
    }

    return $rc;
}

=pod

=head2 _objAdd($obj, $pkey, $dn, $entryStr)

Copy the object's file to the temporary file and add the record to it.

=cut

sub _objAdd
{
    my $self = shift;
    my ($obj, $pkey, $dn,  $entryStr) = @_;
    my $conf = $self->{_config};
    my $oconf = $obj->{conf};
    my $rc = LDAP_SUCCESS;

    if ($obj->{name} eq 'User') {
        my %values;

        foreach my $attr (keys %{$oconf->{attr}}) {
            ($values{$oconf->{attr}{$attr}->{param}[0]}) = ($entryStr =~ /^$attr: (.*)$/mi);
        }

        my $entry = $self->{google}->CreateUser($values{User}, $values{GivenName}, $values{FamilyName}, $values{Password}, $values{Quota});
        if (!defined($entry)) {
            $self->log(level => 'err', message => "Adding User $values{User} to Google Apps failed: ".$self->{google}->{result}{reason});
            $rc = LDAP_OPERATIONS_ERROR;
        }
    }

    return $rc;
}

=pod

=head2 _objDelete($dn)

Copy the object's file from which the appropriate record is deleted to the temporary file.

=cut

sub _objDelete
{
    my $self = shift;
    my ($obj, $pkey, $dn) = @_;
    my $conf = $self->{_config};
    my $oconf = $obj->{conf};
    my $rc = LDAP_SUCCESS;

    my ($rdn_val) = ($dn =~ /^[^=]+=([^,]+),/);

    if ($obj->{name} eq 'User') {
        # multibyte string
        $rdn_val = encode($conf->{mbcode}[0], $rdn_val);

        if (!$self->{google}->DeleteUser($rdn_val)) {
            $self->log(level => 'err', message => "Deleting User $rdn_val from Google Apps failed: ".$self->{google}->{result}{reason});
            $rc = LDAP_OPERATIONS_ERROR;
        }
    }

    return $rc;
}

sub _buildGoogleEntry
{
    my $self = shift;
    my ($obj, $entry) = @_;
    my $oconf = $obj->{conf};
    my $rdn_val;

    if ($obj->{name} eq 'User') {
        $rdn_val = $entry->User();
    }

    my $entryStr = "dn: ".$oconf->{rdn}[0]."=$rdn_val,".$self->_getParentDn($obj)."\n";

    foreach my $oc (@{$oconf->{oc}}) {
        $entryStr = $entryStr."objectclass: $oc\n";
    }

    foreach my $attr (keys %{$oconf->{attr}}) {
        my $param = $oconf->{attr}{$attr}->{param}[0];
        my $value;

        if ($param eq 'rdn') {
            next;
        } elsif ($param eq 'FullName') {
            $value = $entry->FamilyName()." ".$entry->GivenName();
        } else {
            $value = $entry->$param;
        }

        if ($value) {
            $entryStr = "$entryStr$attr: $value\n";
        }
    }

    return $entryStr;
}

=head1 SEE ALSO

L<LISM>,
L<LISM::Storage>

=head1 AUTHOR

Kaoru Sekiguchi, <sekiguchi.kaoru@secioss.co.jp>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2008 by SECIOSS Corporation

=cut

1;
