#!/usr/bin/python
#
# Copyright 2010 SECIOSS Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
# This script is a modification of OpenID consumer app by Google.

"""
An OpenID consumer app for Google App Engine. Allows users to log into
other OpenID providers, then displays their OpenID login. Also stores and
displays the most recent logins.

Uses JanRain's Python OpenID library, version 2.1.1, licensed under the
Apache Software License 2.0:
  http://openidenabled.com/python-openid/

The JanRain library includes a reference OpenID provider that can be used to
test this consumer. After starting the dev_appserver with this app, unpack the
JanRain library and run these commands from its root directory:

  setenv PYTHONPATH .
  python ./examples/server.py -s localhost

Then go to http://localhost:8080/ in your browser, type in
http://localhost:8000/test as your OpenID identifier, and click Verify.
"""

import datetime
import logging
import os
import re
import sys
import urlparse
from Cookie import SimpleCookie

from google.appengine.ext import db
from google.appengine.ext import webapp
from google.appengine.ext.webapp import template

from openid import fetchers
from openid.consumer.consumer import Consumer
from openid.consumer import discover
from openid.extensions import pape, sreg
import fetcher
import store
from session import OpenIDSession


class OpenIDConfig(db.Model):
  """ OpenID configuration."""
  tenant = db.StringProperty()
  op_url = db.LinkProperty()


class OpenIDHandler(webapp.RequestHandler):
  """A base handler class with a couple OpenID-specific utilities."""
  consumer = None
  session = None

  def __init__(self):
    self.session = OpenIDSession()

  def get_consumer(self):
    """Returns a Consumer instance.
    """
    if not self.consumer:
      fetchers.setDefaultFetcher(fetcher.UrlfetchFetcher())
      if not self.session.load(self.request.cookies.get('sissession', None)):
        return
      self.session.set('claimed_id', self.request.get('openid'))
      self.consumer = Consumer(self.session.args, store.DatastoreStore())

    return self.consumer

  def args_to_dict(self):
    """Converts the URL and POST parameters to a singly-valued dictionary.

    Returns:
      dict with the URL and POST body parameters
    """
    req = self.request
    return dict([(arg, req.get(arg)) for arg in req.arguments()])

  def display_name(self, openid_url):
    """Extracts a short, representative part of an OpenID URL for display.

    For example, it returns "ryan" for:
      ryan.com
      www.ryan.com
      ryan.provider.com
      provider.com/ryan
      provider.com/id/path/ryan

    Adapted from Net::OpenID::Consumer, by Brad Fitzpatrick. See:

    http://code.sixapart.com/svn/openid/trunk/perl/Net-OpenID-Consumer/lib/Net/OpenID/VerifiedIdentity.pm

    Args:
      openid_url: string

    Returns:
      string
    """
    if not openid_url:
      return 'None'

    username_re = '[\w.+-]+'

    scheme, host, path, params, query, frag = urlparse.urlparse(openid_url)

    def sanitize(display_name):
      if '@' in display_name:
        # don't display full email addresses; use just the user name part
        display_name = display_name[:display_name.index('@')]
      return display_name

    # is the username in the params?
    match = re.search('(u|id|user|userid|user_id|profile)=(%s)' % username_re,
                      path)
    if match:
      return sanitize(match.group(2))

    # is the username in the path?
    path = path.split('/')
    if re.match(username_re, path[-1]):
      return sanitize(path[-1])

    # use the hostname
    host = host.split('.')
    if len(host) == 1:
      return host[0]

    # strip common tlds and country code tlds
    common_tlds = ('com', 'org', 'net', 'edu', 'info', 'biz', 'gov', 'mil')
    if host[-1] in common_tlds or len(host[-1]) == 2:
      host = host[:-1]
    if host[-1] == 'co':
      host = host[:-1]

    # strip www prefix
    if host[0] == 'www':
      host = host[1:]

    return sanitize('.'.join(host))


class OpenIDAuthHandler(OpenIDHandler):
  """Handles a POST response to the OpenID login form."""

  def get(self):
    """Handles login requests."""
    logging.info(self.args_to_dict())
    tenant = self.request.get('tenant')
    if tenant == '':
      tenant = 'Default'
    configs = OpenIDConfig.gql('where tenant = :1', tenant)

    if configs.count() != 0:
      config = configs[0]
    else:
      self.response.out.write('No configuration')
      return

    openid_url = config.op_url
    if not openid_url:
      self.report_error('Please enter an OpenID URL.')
      return

    logging.debug('Beginning discovery for OpenID %s' % openid_url)
    try:
      consumer = self.get_consumer()
      if not consumer:
        return
      auth_request = consumer.begin(openid_url)
    except discover.DiscoveryFailure, e:
      self.report_error('Error during OpenID provider discovery.', e)
      return
    except discover.XRDSError, e:
      self.report_error('Error parsing XRDS from provider.', e)
      return

    self.session.set('claimed_id', auth_request.endpoint.claimed_id)
    self.session.set('server_url', auth_request.endpoint.server_url)
    self.session.store()

    sreg_request = sreg.SRegRequest(optional=['nickname', 'fullname', 'email'])
    auth_request.addExtension(sreg_request)

    parts = list(urlparse.urlparse(self.request.uri))
    parts[2] = re.sub('[^/]+$', 'finish', parts[2])
    parts[5] = ''
    return_to = urlparse.urlunparse(parts)
    realm = urlparse.urlunparse(parts[0:2] + [''] * 4)

    redirect_url = auth_request.redirectURL(realm, return_to)
    cookie = SimpleCookie(os.environ.get('HTTP_COOKIE', ''))
    cookie['sissession'] = self.session.getId()
    cookie['sissession']['path'] = '/'
    cookie['sissession']['expires'] = 300
    print cookie.output()
    logging.debug('Redirecting to %s' % redirect_url)
    self.response.set_status(302)
    self.response.headers['Location'] = redirect_url


class OpenIDFinishHandler(OpenIDHandler):
  """Handle a redirect from the provider."""
  def get(self):
    args = self.args_to_dict()
    consumer = self.get_consumer()
    if not consumer:
      return

    response = consumer.complete(args, self.request.uri)

    if response.status == 'success':
      sreg_data = sreg.SRegResponse.fromSuccessResponse(response).items()
      self.session.set('claimed_id', response.endpoint.claimed_id)
      self.session.set('server_url', response.endpoint.server_url)
      self.session.set('status', 'OK')
      self.session.set('display_name', self.display_name(response.endpoint.claimed_id))
      for key, value in sreg_data:
        self.session.set(key, value)
    elif response.status == 'failure':
      self.session.set('status', 'NG')
      logging.error(str(response))

    logging.debug('Login status %s for claimed_id %s' %
                  (response.status, self.session.get('claimed_id')))

    self.redirect('/')
