/* Copyright(C) 2004,2005,2006,2007 Brazil

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "lib/com.h"
#include "lib/store.h"
#include <string.h>
#include <stdio.h>
#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif /* HAVE_SYS_WAIT_H */

#define DEFAULT_PORT 10041
#define DEFAULT_PATH "." PATH_SEPARATOR "senna.db"
#define DEFAULT_DEST "localhost"

int port = DEFAULT_PORT;

static void
usage(void)
{
  fprintf(stderr,
          "Usage: senna [options...] [dest]\n"
          "options:\n"
          "  -a:                 run in standalone mode (default)\n"
          "  -c:                 run in client mode\n"
          "  -s:                 run in server mode\n"
          "  -p <port number>:   server port number (default: %d)\n"
          "dest: <db pathname> or <dest hostname>\n"
          "  <db pathname>: when standalone/server mode (default: \"%s\")\n"
          "  <dest hostname>: when client mode (default: \"%s\")\n",
          DEFAULT_PORT, DEFAULT_PATH, DEFAULT_DEST);
}

sen_rc
show_obj(sen_db_ctx *c, sen_db_obj *obj, sen_rbuf *buf, sen_com_sqtp *cs)
{
  sen_rc rc = sen_success;
  SEN_RBUF_REWIND(buf);
  if (obj) { sen_db_obj_inspect(c->db, obj, buf); }
  if (cs) {
    sen_com_sqtp_header header;
    header.size = SEN_RBUF_VSIZE(buf);
    header.flags = SEN_DB_CTX_TAIL;
    if (sen_com_sqtp_send(cs, &header, buf->head) == -1) {
      rc = sen_other_error;
    }
  } else {
    if (SEN_RBUF_VSIZE(buf)) {
      fwrite(buf->head, 1, SEN_RBUF_VSIZE(buf), stdout);
      putchar('\n');
    }
  }
  return rc;
}

static sen_db_obj *
put_method(sen_db_ctx *c, sen_db_obj *args, sen_db_cont *cont)
{
  if (BULKP(args)) {
    sen_com_sqtp_header header;
    header.size = args->u.b.size;
    header.flags = 0;
    sen_com_sqtp_send((sen_com_sqtp *)c->data.ptr, &header, args->u.b.value);
  }
  return NULL;
}

static void
msg_handler(sen_com_event *ev, sen_com *c)
{
  sen_com_sqtp *cs = (sen_com_sqtp *)c;
  sen_db_ctx *ctx = (sen_db_ctx *)cs->userdata;
  if (cs->rc) { goto exit; }
  if (!ctx) {
    ctx = sen_db_ctx_open((sen_db *)ev->userdata, sen_db_sexp_parser);
    if (!ctx) { goto exit; }
    sen_db_ctx_def_native_method(ctx, "_put", put_method);
    ctx->data.ptr = cs;
    cs->userdata = ctx;
  }
  {
    char *body = SEN_COM_SQTP_MSG_BODY(&cs->msg);
    uint32_t size = SEN_COM_SQTP_MSG_HEADER(&cs->msg)->size;
    sen_db_obj *r = sen_db_ctx_feed(ctx, body, size, 0);
    if (show_obj(ctx, r, &cs->msg, cs)) { goto exit ; }
    return;
  }
exit :
  SEN_LOG(sen_log_notice, "connection closed..");
  if (ctx) { sen_db_ctx_close(ctx); }
  sen_com_sqtp_close(ev, cs);
}

#define BUFSIZE 0x100000
#define MAX_CON 0x100

static int
do_alone(char *path)
{
  int rc = -1;
  if (!sen_init()) {
    sen_rbuf buf;
    if (!sen_rbuf_init(&buf, BUFSIZE)) {
      sen_db *db;
      if ((db = sen_db_open(path)) ||
          (db = sen_db_create(path, 0, sen_enc_none))) {
        sen_db_ctx *ctx;
        if ((ctx = sen_db_ctx_open(db, sen_db_sexp_parser))) {
          while (fputs("> ", stderr), fgets(buf.head, SEN_RBUF_WSIZE(&buf), stdin)) {
            uint32_t size = strlen(buf.head) - 1;
            sen_db_obj *r = sen_db_ctx_feed(ctx, buf.head, size, 0);
            show_obj(ctx, r, &buf, NULL);
          }
          rc = 0;
          sen_db_ctx_close(ctx);
        } else {
          fprintf(stderr, "db_ctx open failed (%s)\n",  path);
        }
        sen_db_close(db);
      } else {
        fprintf(stderr, "db open failed (%s)\n", path);
      }
      sen_rbuf_fin(&buf);
    }
    sen_fin();
  }
  return rc;
}

static int
do_client(char *hostname)
{
  int rc = -1;
  sen_com_sqtp *cs;
  sen_com_init();
  if ((cs = sen_com_sqtp_copen(NULL, hostname, port))) {
    sen_rbuf *buf = &cs->msg;
    sen_com_sqtp_header sheader, *rheader;
    if (!sen_rbuf_reinit(buf, BUFSIZE)) {
      while (fputs("> ", stderr), fgets(buf->head, SEN_RBUF_WSIZE(buf), stdin)) {
        sheader.flags = SEN_DB_CTX_TAIL;
        sheader.size = strlen(buf->head) - 1;
        if (sen_com_sqtp_send(cs, &sheader, buf->head)) { break; }
        do {
          if (sen_com_sqtp_recv(cs, buf)) {
            fprintf(stderr, "sen_com_sqtp_recv failed\n");
            goto exit;
          }
          rheader = SEN_COM_SQTP_MSG_HEADER(buf);
          if (rheader->size) {
            fwrite(SEN_COM_SQTP_MSG_BODY(buf), 1, rheader->size, stdout);
            putchar('\n');
          }
        } while (!(rheader->flags & SEN_DB_CTX_TAIL));
      }
      rc = 0;
    } else {
      fprintf(stderr, "sen_rbuf_reinit failed (%d)\n", BUFSIZE);
    }
    sen_com_sqtp_close(NULL, cs);
  } else {
    fprintf(stderr, "sen_com_sqtp_copen failed (%s:%d)\n", hostname, port);
  }
exit :
  return rc;
}

// todo : use thread
static int
server(char *path)
{
  int rc = -1;
  sen_com_init();
  if (!sen_init()) {
    sen_com_event ev;
    if (!sen_com_event_init(&ev, MAX_CON, sizeof(sen_com_sqtp))) {
      sen_db *db;
      if ((db = sen_db_open(path)) ||
          (db = sen_db_create(path, 0, sen_enc_none))) {
        sen_com_sqtp *cs;
        ev.userdata = db;
        if ((cs = sen_com_sqtp_sopen(&ev, port, msg_handler))) {
          while (!sen_com_event_poll(&ev, -1)) ;
          sen_com_sqtp_close(&ev, cs);
          rc = 0;
        } else {
          fprintf(stderr, "sen_com_sqtp_sopen failed (%d)\n", port);
        }
        sen_db_close(db);
      } else {
        fprintf(stderr, "db open failed (%s)\n", path);
      }
      sen_com_event_fin(&ev);
    } else {
      fprintf(stderr, "sen_com_event_init failed\n");
    }
    sen_fin();
  } else {
    fprintf(stderr, "sen_init failed\n");
  }
  return rc;
}

#ifndef WIN32
static int
do_server(char *path)
{
  pid_t pid;
  switch (fork()) {
  case 0:
    break;
  case -1:
    perror("fork");
    return -1;
  default:
    wait(NULL);
    return 0;
  }
  switch ((pid = fork())) {
  case 0:
    break;
  case -1:
    perror("fork");
    return -1;
  default:
    fprintf(stderr, "%d\n", pid);
    _exit(0);
  }
  return server(path);
}
#endif /* WIN32 */

enum {
  mode_alone,
  mode_client,
  mode_server,
  mode_usage
};

int
main(int argc, char **argv)
{
  char *portstr = NULL;
  int i, mode = mode_alone;
  static sen_str_getopt_opt opts[] = {
    {'p', NULL, NULL, 0, getopt_op_none},
    {'h', NULL, NULL, mode_usage, getopt_op_update},
    {'a', NULL, NULL, mode_alone, getopt_op_update},
    {'c', NULL, NULL, mode_client, getopt_op_update},
    {'s', NULL, NULL, mode_server, getopt_op_update},
    {'\0', NULL, NULL, 0, 0}
  };

  opts[0].arg = &portstr;
  i = sen_str_getopt(argc, argv, opts, &mode);
  if (i < 0) { mode = mode_usage; }
  if (portstr) { port = atoi(portstr); }
  switch (mode) {
  case mode_alone :
    return do_alone(argc <= i ? DEFAULT_PATH : argv[i]);
  case mode_client :
    return do_client(argc <= i ? DEFAULT_DEST : argv[i]);
  case mode_server :
#ifdef WIN32
    return server(argc <= i ? DEFAULT_PATH : argv[i]);
#else /* WIN32 */
    return do_server(argc <= i ? DEFAULT_PATH : argv[i]);
#endif /* WIN32 */
  default :
    usage(); return -1;
  }
  sen_com_fin();
}
