/*
 * Copyright (C) 2007 JasperSoft http://www.jaspersoft.com
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed WITHOUT ANY WARRANTY; and without the 
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see http://www.gnu.org/licenses/gpl.txt 
 * or write to:
 * 
 * Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330,
 * Boston, MA  USA  02111-1307
 */
 
#include <stdsoap2.h>
/**
\struct jasperserver_attachment attachment.h

This struct contains all the fields required by the callbacks to stream a file as attachment using the mechanism
provided by gSOAP.
This struct should not be used by the user. It is used internally by the function jasperserver_attach_file.

\see jasperserver_attach_file
*/
typedef struct jasperserver_attachment
{ 
    unsigned char * __ptr; /**< fd handle which we need in the callbacks */
    int __size; /**< size of the file to read */
    char *id; /**< always set to attachment */
    char *type; /**< always set to application/octet-stream */
    char *options; /**< obtained using soap_dime_option(soap, 0, "attachment") */
} jasperserver_attachment_t;

/** 
 *  \brief Attach a file to a soap struct previously created and initiated.
 *
 *  This function provides an easy way to add an attachment to a SOAP request.
 *  The file is streamed using the mechanism provided by gSOAP. A struct jasperserver_attachment is created
 *  to keep track of the FILE* opened. That structure will be passed to the callback functions to open,
 *  read and close the file.
 *
 *  \param soapPtr A pointer to the struct soap
 *  \param fileName The file name to attach
 *  \return The function return 0 if no errors occur, otherwise a positive integer representing the error.
 *
 *  Possible errors:
 *  \li 1 File to attach not specified!
 *  \li 2 Soap struct NULL!
 *  \li 3 File not found!!
 *  \li 4 Unable to find the size of the file
 *  \li 5 Error setting up DIME attachment
*/
int jasperserver_attach_file(struct soap* soapPtr, const char* fileName);

/** 
 *  \brief Callback function to open a file to read.
 *
 *  This function is used as callback for the gSoap file streaming mechanism.
 *  This funcion is used by jasperserver_attach_file and it should not be called directly by the user.
 *
 *  \param soap A pointer to the struct soap
 *  \param handle It is a ready FILE pointer
 *  \param id Not used
 *  \param type Not used
 *  \param options Not used
 *  \return The handle input parameter.
 *
 *  \see jasperserver_attach_file
*/
void *jasperserver_dime_read_open(struct soap *soap, void *handle, const char *id, const char *type, const char *options);

/** 
 *  \brief Callback function to close the attached file.
 *
 *  This function is used as callback for the gSoap file streaming mechanism.
 *  This funcion is used by jasperserver_attach_file and it should not be called directly by the user.
 *
 *  \param soap A pointer to the struct soap
 *  \param handle It is a FILE pointer
 *  \param id Not used
 *  \param type Not used
 *  \param options Not used
 *
 *  \see jasperserver_attach_file
*/
void jasperserver_dime_read_close(struct soap *soap, void *handle);

/** 
 *  \brief Callback function to read bytes from a file.
 *
 *  This function is used as callback for the gSoap file streaming mechanism.
 *  This funcion is used by jasperserver_attach_file and it should not be called directly by the user.
 *
 *  \param soap A pointer to the struct soap
 *  \param handle It is a ready FILE pointer
 *  \param buf Buffer to store the read bytes
 *  \param len Number of bytes to read
 *  \return The number of read bytes
 *
 *  \see jasperserver_attach_file
*/
size_t jasperserver_dime_read(struct soap *soap, void *handle, char *buf, size_t len);

/** 
 *  \brief Allocate a buffer and put into that the content of a file. 
 *
 *  This function is currently not used.
 *   The user is responsible to free the data allocated to read the file with free(*dataPtr)
 *
 *  \param fileName A file to read
 *  \param dataPtr A pointer to a data pointer (char **).
 *  \param sizePtr Pointer to a size_t to save the size of the allocated buffer
 *  \return 0 if no error occurs, a different value otherwise.
 *
 *  Possible errors:
 *  \li 1 File to attach not specified!
 *  \li 3 File not found!!
 *
 *  \see jasperserver_attach_file
*/
int jasperserver_read_file(const char* fileName, char **dataPtr, size_t *sizePtr);
