/*
 * Copyright (C) 2007 JasperSoft http://www.jaspersoft.com
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed WITHOUT ANY WARRANTY; and without the 
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see http://www.gnu.org/licenses/gpl.txt 
 * or write to:
 * 
 * Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330,
 * Boston, MA  USA  02111-1307
 */
 
#ifndef jasperserver_services_H
#define jasperserver_services_H

#include "stringbuffer.h"
#include "repository.h"

/** 
 *  \brief List service
 *
 *  This function execute the list service passing as input the request structure.
 *  The URL and credentials used to connect to the server are passed using the server structure.
 *  The user is responsible to free the returned jasperserver_operation_result_t.
 *
 *  If a terrible error occurs, the returned jasperserver_operation_result_t can be NULL
 *
 *  \param server A jasperserver_server_t struct with the information to connect to the server
 *  \param request The request
 *  \return a jasperserver_operation_result_t pointer or NULL if something of terrible is happen.
 *
*/
jasperserver_operation_result_t * jasperserver_ws_list(jasperserver_server_t *server, jasperserver_request_t *request);

/** 
 *  \brief Delete service
 *
 *  This function delete the resource pointed by the request.
 *  The URL and credentials used to connect to the server are passed using the server structure.
 *  The user is responsible to free the returned jasperserver_operation_result_t.
 *
 *  If a terrible error occurs, the returned jasperserver_operation_result_t can be NULL
 *
 *  \param server A jasperserver_server_t struct with the information to connect to the server
 *  \param request The request
 *  \return a jasperserver_operation_result_t pointer or NULL if something of terrible is happen.
 *
 *  \warning Don't try to delete the root resource, since it will be removed all the content of your repository!
 *
*/
jasperserver_operation_result_t * jasperserver_ws_delete(jasperserver_server_t *server, jasperserver_request_t *request);

/** 
 *  \brief Get service
 *
 *  This function get a resource from the repository.
 *  The URL and credentials used to connect to the server are passed using the server structure.
 *  The user is responsible to free the returned jasperserver_operation_result_t.
 *  If the resource contains data, it can be stored in a file passed as argument.
 *
 *  If a terrible error occurs, the returned jasperserver_operation_result_t can be NULL
 *
 *  \param server A jasperserver_server_t struct with the information to connect to the server
 *  \param request The request
 *  \param filename optional file to store the resource content
 *  \return a jasperserver_operation_result_t pointer or NULL if something of terrible is happen.
 *
*/
jasperserver_operation_result_t * jasperserver_ws_get(jasperserver_server_t *server, jasperserver_request_t *request, char *filename);

/** 
 *  \brief RunReport service
 *
 *  This function run a report unit.
 *  The URL and credentials used to connect to the server are passed using the server structure.
 *  The user is responsible to free the returned jasperserver_operation_result_t.
 *  The returned report can be saved in a file passed as argument. If there are more than a single attachment,
 *  all the attachments with id different thatn \<report\> and starting with \<img are stored in a directory
 *  called images created in the directory of filename.
 *
 *  If the report requires parameters, they must be added to the resource in the request.
 *
 *  If a terrible error occurs, the returned jasperserver_operation_result_t can be NULL
 *
 *  \param server A jasperserver_server_t struct with the information to connect to the server
 *  \param request The request
 *  \param filename optional file to store the generated report
 *  \return a jasperserver_operation_result_t pointer or NULL if something of terrible is happen.
 *
*/
jasperserver_operation_result_t * jasperserver_ws_runReport(jasperserver_server_t *server, jasperserver_request_t *request, char *filename);

/** 
 *  \brief Put service
 *
 *  This function is used to put or modify a resource in the repository.
 *  The URL and credentials used to connect to the server are passed using the server structure.
 *  The user is responsible to free the returned jasperserver_operation_result_t.
 *  If the resource requires data, a file can be attached to the SOAP message. The file is streammed,
 *  so don't warry about the size.
 *
 *  If a terrible error occurs, the returned jasperserver_operation_result_t can be NULL
 *
 *  \param server A jasperserver_server_t struct with the information to connect to the server
 *  \param request The request
 *  \param filename optional file to attach
 *  \return a jasperserver_operation_result_t pointer or NULL if something of terrible is happen.
 *
*/
jasperserver_operation_result_t * jasperserver_ws_put(jasperserver_server_t *server, jasperserver_request_t *request, char *filename);

#endif
