/*
 * Copyright (C) 2005 - 2007 JasperSoft Corporation.  All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased a commercial license agreement from JasperSoft,
 * the following license terms apply:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published by
 * the Free Software Foundation.
 *
 * This program is distributed WITHOUT ANY WARRANTY; and without the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see http://www.gnu.org/licenses/gpl.txt
 * or write to:
 *
 * Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330,
 * Boston, MA  USA  02111-1307
 */
package com.jaspersoft.jasperserver.api.engine.jasperreports.util;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import net.sf.jasperreports.engine.JRBand;
import net.sf.jasperreports.engine.JRElement;
import net.sf.jasperreports.engine.JRException;
import net.sf.jasperreports.engine.JRExpression;
import net.sf.jasperreports.engine.JRExpressionChunk;
import net.sf.jasperreports.engine.JRFont;
import net.sf.jasperreports.engine.JRGroup;
import net.sf.jasperreports.engine.JRImage;
import net.sf.jasperreports.engine.JRReport;
import net.sf.jasperreports.engine.JRStyle;
import net.sf.jasperreports.engine.JRSubreport;
import net.sf.jasperreports.engine.design.JasperDesign;
import net.sf.jasperreports.engine.xml.JRXmlLoader;

import com.jaspersoft.jasperserver.api.JSExceptionWrapper;
import com.jaspersoft.jasperserver.api.metadata.common.domain.FileResource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Resource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.client.FileResourceImpl;


/**
 * @author Teodor Danciu (teodord@users.sourceforge.net)
 * @version $Id: AbstractAttributedObject.java 2140 2006-02-21 06:41:21Z tony $
 */
public class ResourceCollector
{	
	private static final String REPO_URL_PROTOCOL = "repo:";
	
	private List resources = new ArrayList();
	private Set resourceNames = new HashSet();
	
	/**
	 * 
	 */
	private ResourceCollector()
	{
	}

	/**
	 * 
	 */
	public static Resource[] getResources(InputStream jrxmlStream)
	{
		JasperDesign jasperDesign = null;

		try
		{
			jasperDesign = JRXmlLoader.load(jrxmlStream);
		}
		catch (JRException e)
		{
			throw new JSExceptionWrapper(e);
		}
		
		ResourceCollector collector = new ResourceCollector();
		collector.collect(jasperDesign);
		
		return collector.getResources();
	}

	/**
	 * 
	 */
	private Resource[] getResources()
	{
		return (Resource[]) resources.toArray(new Resource[resources.size()]);
	}

	/**
	 * 
	 */
	private void collect(JRReport report)
	{
		if (report != null)
		{
			JRStyle[] styles = report.getStyles();
			if (styles != null)
			{
				for (int i = 0; i < styles.length; i++)
				{
					collect(styles[i]);
				}
			}
			
			collect(report.getBackground());
			collect(report.getTitle());
			collect(report.getPageHeader());
			collect(report.getColumnHeader());
			collect(report.getDetail());
			collect(report.getColumnFooter());
			collect(report.getPageFooter());
			collect(report.getSummary());
			
			JRGroup[] groups = report.getGroups();
			if (groups != null)
			{
				for(int i = 0; i < groups.length; i++)
				{
					JRGroup group = groups[i];
					collect(group.getGroupHeader());
					collect(group.getGroupFooter());
				}
			}
		}
	}

	/**
	 * 
	 */
	private void collect(JRStyle style)
	{
		collectFont(style.getOwnPdfFontName()); 

		JRStyle[] conditionalStyles = style.getConditionalStyles();
		if (conditionalStyles != null)
		{
			for (int i = 0; i < conditionalStyles.length; i++)
			{
				collect(conditionalStyles[i]);
			}
		}
	}

	/**
	 * 
	 */
	private void collect(JRBand band)
	{
		if (band != null)
		{
			JRElement[] elements = band.getElements();
			if (elements != null)
			{
				for(int i = 0; i < elements.length; i++)
				{
					JRElement element = elements[i];
					if (element instanceof JRFont)
					{
						JRFont font = (JRFont)element;
						collectFont(font.getOwnPdfFontName());
					}
					else if (element instanceof JRImage)
					{
						JRImage image = (JRImage)element;
						collectFromExpression(image.getExpression(), FileResource.TYPE_IMAGE);
					}
					else if (element instanceof JRSubreport)
					{
						JRSubreport subreport = (JRSubreport)element;
						collectFromExpression(subreport.getExpression(), FileResource.TYPE_JRXML);
					}
				}
			}
		}
	}

	/**
	 * 
	 */
	private void collectFont(String pdfFontName)
	{
		if (
			pdfFontName != null 
			&& pdfFontName.startsWith(REPO_URL_PROTOCOL)
			&& pdfFontName.indexOf('/') < 0
			)
		{
			String resourceName = pdfFontName.substring(REPO_URL_PROTOCOL.length());
			if (!resourceNames.contains(resourceName))
			{
				FileResource resource = new FileResourceImpl();
				resource.setName(resourceName);
				resource.setLabel(resource.getName());
				resource.setDescription(resource.getName());
				resource.setFileType(FileResource.TYPE_FONT);

				resources.add(resource);
				resourceNames.add(resourceName);
			}
		}
	}

	/**
	 * 
	 */
	private void collectFromExpression(JRExpression expression, String resourceType)
	{
		if (
			expression != null 
			&& expression.getChunks() != null
			&& expression.getChunks().length == 1
			)
		{
			JRExpressionChunk chunk = expression.getChunks()[0];
			if (chunk.getType() == JRExpressionChunk.TYPE_TEXT)
			{
				String resourceName = chunk.getText().trim();
				if (
					resourceName.startsWith("\"" + REPO_URL_PROTOCOL)
					&& resourceName.indexOf('/') < 0
					&& resourceName.endsWith("\"")
					)
				{
					resourceName = resourceName.substring(REPO_URL_PROTOCOL.length() + 1, resourceName.length() - 1);
					if (!resourceNames.contains(resourceName))
					{
						FileResource resource = new FileResourceImpl();
						resource.setName(resourceName);
						resource.setLabel(resource.getName());
						resource.setDescription(resource.getName());
						resource.setFileType(resourceType);

						resources.add(resource);
						resourceNames.add(resourceName);
					}
				}
			}
		}
	}

}
