/*
 * Copyright (C) 2005 - 2007 JasperSoft Corporation.  All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased a commercial license agreement from JasperSoft,
 * the following license terms apply:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published by
 * the Free Software Foundation.
 *
 * This program is distributed WITHOUT ANY WARRANTY; and without the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see http://www.gnu.org/licenses/gpl.txt
 * or write to:
 *
 * Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330,
 * Boston, MA  USA  02111-1307
 */
package com.jaspersoft.jasperserver.api.engine.scheduling.quartz;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import javax.mail.MessagingException;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeUtility;

import net.sf.jasperreports.engine.JRException;
import net.sf.jasperreports.engine.JRExporterParameter;
import net.sf.jasperreports.engine.JRParameter;
import net.sf.jasperreports.engine.JRPrintPage;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.export.JExcelApiExporter;
import net.sf.jasperreports.engine.export.JExcelApiExporterParameter;
import net.sf.jasperreports.engine.export.JRCsvExporter;
import net.sf.jasperreports.engine.export.JRCsvExporterParameter;
import net.sf.jasperreports.engine.export.JRHtmlExporter;
import net.sf.jasperreports.engine.export.JRHtmlExporterParameter;
import net.sf.jasperreports.engine.export.JRHyperlinkProducerFactory;
import net.sf.jasperreports.engine.export.JRRtfExporter;
import net.sf.jasperreports.engine.export.JRXlsExporterParameter;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.Job;
import org.quartz.JobDataMap;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;
import org.quartz.SchedulerContext;
import org.springframework.context.ApplicationContext;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.mail.javamail.MimeMessageHelper;

import com.jaspersoft.jasperserver.api.JSException;
import com.jaspersoft.jasperserver.api.JSExceptionWrapper;
import com.jaspersoft.jasperserver.api.common.domain.ExecutionContext;
import com.jaspersoft.jasperserver.api.common.domain.LogEvent;
import com.jaspersoft.jasperserver.api.common.domain.impl.ExecutionContextImpl;
import com.jaspersoft.jasperserver.api.common.util.CharacterEncodingProvider;
import com.jaspersoft.jasperserver.api.common.util.LocaleHelper;
import com.jaspersoft.jasperserver.api.engine.common.service.EngineService;
import com.jaspersoft.jasperserver.api.engine.common.service.LoggingService;
import com.jaspersoft.jasperserver.api.engine.common.service.SecurityContextProvider;
import com.jaspersoft.jasperserver.api.engine.common.service.VirtualizerFactory;
import com.jaspersoft.jasperserver.api.engine.jasperreports.common.CsvExportParametersBean;
import com.jaspersoft.jasperserver.api.engine.jasperreports.common.PdfExportParametersBean;
import com.jaspersoft.jasperserver.api.engine.jasperreports.common.XlsExportParametersBean;
import com.jaspersoft.jasperserver.api.engine.jasperreports.domain.impl.ReportUnitRequest;
import com.jaspersoft.jasperserver.api.engine.jasperreports.domain.impl.ReportUnitResult;
import com.jaspersoft.jasperserver.api.engine.jasperreports.util.JExcelApiHyperlinkProducerFactory;
import com.jaspersoft.jasperserver.api.engine.scheduling.domain.ReportJob;
import com.jaspersoft.jasperserver.api.engine.scheduling.domain.ReportJobIdHolder;
import com.jaspersoft.jasperserver.api.engine.scheduling.domain.ReportJobMailNotification;
import com.jaspersoft.jasperserver.api.engine.scheduling.domain.ReportJobRepositoryDestination;
import com.jaspersoft.jasperserver.api.engine.scheduling.service.ReportJobsPersistenceService;
import com.jaspersoft.jasperserver.api.metadata.common.domain.ContentResource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Folder;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Resource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.client.ContentResourceImpl;
import com.jaspersoft.jasperserver.api.metadata.common.service.RepositoryService;

/**
 * @author Lucian Chirita (lucianc@users.sourceforge.net)
 * @version $Id: ReportExecutionJob.java 9931 2007-09-06 16:31:57Z tdanciu $
 */
public class ReportExecutionJob implements Job {
	
	private static final Log log = LogFactory.getLog(ReportExecutionJob.class);
	
	public static final String REPORT_PARAMETER_SCHEDULED_TIME = "_ScheduledTime";
	
	public static final String REPOSITORY_FILENAME_SEQUENCE_SEPARATOR = "-";
	public static final String REPOSITORY_FILENAME_TIMESTAMP_SEQUENCE_PATTERN = "yyyyMMddHHmm";
	
	public static final String SCHEDULER_CONTEXT_KEY_APPLICATION_CONTEXT = "applicationContext";
	
	public static final String SCHEDULER_CONTEXT_KEY_JOB_PERSISTENCE_SERVICE = "jobPersistenceService";
	public static final String SCHEDULER_CONTEXT_KEY_ENGINE_SERVICE = "engineService";
	public static final String SCHEDULER_CONTEXT_KEY_VIRTUALIZER_FACTORY = "virtualizerFactory";
	public static final String SCHEDULER_CONTEXT_KEY_REPOSITORY = "repositoryService";
	public static final String SCHEDULER_CONTEXT_KEY_MAIL_SENDER = "mailSender";
	public static final String SCHEDULER_CONTEXT_KEY_MAIL_FROM_ADDRESS = "mailFromAddress";
	public static final String SCHEDULER_CONTEXT_KEY_LOGGING_SERVICE = "loggingService";
	public static final String SCHEDULER_CONTEXT_KEY_SECURITY_CONTEXT_PROVIDER = "securityContextProvider";
	public static final String SCHEDULER_CONTEXT_KEY_HYPERLINK_PRODUCER_FACTORY = "hyperlinkProducerFactory";
	public static final String SCHEDULER_CONTEXT_KEY_ENCODING_PROVIDER = "encodingProvider";
	public static final String SCHEDULER_CONTEXT_KEY_EXPORT_PARAMETRES_MAP = "exportParametersMap";
	
	public static final String JOB_DATA_KEY_DETAILS_ID = "jobDetailsID";	
	public static final String JOB_DATA_KEY_USERNAME = "jobUser";
	
	public static final String LOGGING_COMPONENT = "reportScheduler";

	protected ApplicationContext applicationContext;
	protected String username;
	protected ReportJob jobDetails;
	protected JobExecutionContext jobContext;
	protected SchedulerContext schedulerContext;
	protected ExecutionContext executionContext;

	public void execute(JobExecutionContext context) throws JobExecutionException {
		try {
			this.jobContext = context;
			this.schedulerContext = jobContext.getScheduler().getContext();

			this.applicationContext = (ApplicationContext) schedulerContext.get(SCHEDULER_CONTEXT_KEY_APPLICATION_CONTEXT);
			this.username = getUsername();
			SecurityContextProvider securityContextProvider = getSecurityContextProvider();
			securityContextProvider.setAuthenticatedUser(this.username);
			try {
				executionContext = getExecutionContext();
				jobDetails = getJobDetails();
				initJobExecution();

				executeAndSendReport();
			} catch (Throwable e) {
				handleException(e);
			} finally {
				securityContextProvider.revertAuthenticatedUser();
			}
		} catch (JobExecutionException e) {
			throw e;
		} catch (Throwable e) {
			handleException(e);
		} finally {
			clear();
		}
	}

	protected void initJobExecution() {
		updateExecutionContextDetails();
	}

	protected void clear() {
		jobContext = null;
		schedulerContext = null;
		jobDetails = null;
		executionContext = null;
		username = null;
	}

	protected String getUsername() {
		JobDataMap jobDataMap = jobContext.getTrigger().getJobDataMap();
		return jobDataMap.getString(JOB_DATA_KEY_USERNAME);
	}

	protected ExecutionContext getExecutionContext() {
		return new ExecutionContextImpl();
	}

	protected void updateExecutionContextDetails() {
		((ExecutionContextImpl) executionContext).setLocale(getJobLocale());	
	}
	
	protected Locale getJobLocale() {
		String localeCode = jobDetails.getOutputLocale();
		Locale locale;
		if (localeCode != null && localeCode.length() > 0) {
			locale = LocaleHelper.getInstance().getLocale(localeCode);
		} else {
			locale = null;
		}
		return locale;
	}
	
	protected void handleException(Throwable exc) throws JobExecutionException {
		log.error(exc, exc);
		
		try {
			logException(exc);
		} catch (Exception e) {
			log.error(e, e);
			throwJobExecutionException(exc);		
		}

		throwJobExecutionException(exc);		
	}

	protected void throwJobExecutionException(Throwable exc) throws JobExecutionException {
		throw exc instanceof Exception ? new JobExecutionException((Exception)exc) : new JobExecutionException(exc.getMessage());
	}

	protected void logException(Throwable e) {
		LoggingService loggingService = getLoggingService();
		LogEvent event = loggingService.instantiateLogEvent();
		event.setComponent(LOGGING_COMPONENT);
		event.setType(LogEvent.TYPE_ERROR);
		event.setMessageCode("log.error.report.job.failed");
		if (jobDetails != null) {
			event.setResourceURI(jobDetails.getSource().getReportUnitURI());
		}
		
		StringWriter writer = new StringWriter();
		PrintWriter printWriter = new PrintWriter(writer);
		if (jobDetails != null) {
			printWriter.println("Job: " + jobDetails.getLabel() + " (ID: " + jobDetails.getId() + ")");
			printWriter.println("Report unit: " + jobDetails.getSource().getReportUnitURI());			
		}
		printWriter.println("Quartz Job: " + jobContext.getJobDetail().getFullName());
		printWriter.println("Quartz Trigger: " + jobContext.getTrigger().getFullName());
		printWriter.println(e.getMessage());
		e.printStackTrace(printWriter);
		printWriter.flush();
		event.setText(writer.toString());
		event.setState(LogEvent.STATE_UNREAD);
		
		loggingService.log(event);
	}

	protected SecurityContextProvider getSecurityContextProvider() {
		return (SecurityContextProvider) schedulerContext.get(SCHEDULER_CONTEXT_KEY_SECURITY_CONTEXT_PROVIDER);
	}

	protected LoggingService getLoggingService() {
		return (LoggingService) schedulerContext.get(SCHEDULER_CONTEXT_KEY_LOGGING_SERVICE);
	}

	protected ReportJob getJobDetails() {
		ReportJobsPersistenceService persistenceService = getPersistenceService();
		JobDataMap jobDataMap = jobContext.getTrigger().getJobDataMap();
		long jobId = jobDataMap.getLong(JOB_DATA_KEY_DETAILS_ID);
		ReportJob job = persistenceService.loadJob(executionContext, new ReportJobIdHolder(jobId));
		return job;
	}

	protected ReportJobsPersistenceService getPersistenceService() {
		return (ReportJobsPersistenceService) schedulerContext.get(SCHEDULER_CONTEXT_KEY_JOB_PERSISTENCE_SERVICE);
	}

	protected EngineService getEngineService() {
		EngineService engineService = (EngineService) schedulerContext.get(SCHEDULER_CONTEXT_KEY_ENGINE_SERVICE);
		return engineService;
	}

	protected VirtualizerFactory getVirtualizerFactory() {
		return (VirtualizerFactory) schedulerContext.get(SCHEDULER_CONTEXT_KEY_VIRTUALIZER_FACTORY);
	}

	protected void executeAndSendReport() throws JobExecutionException
	{
		ReportUnitResult result = null;
		try
		{
			result = executeReport();
			sendToDestinations(result);
		}
		finally
		{
			if (result != null)
			{
				VirtualizerFactory virtualizerFactory = getVirtualizerFactory();
				if (virtualizerFactory != null)
				{
					virtualizerFactory.disposeVirtualizer(result.getVirtualizer());
				}
			}
		}
	}

	protected ReportUnitResult executeReport() {
		Map parametersMap = collectReportParameters();
		
		ReportUnitRequest request = new ReportUnitRequest(getReportUnitURI(), parametersMap);

		EngineService engineService = getEngineService();
		return (ReportUnitResult) engineService.execute(executionContext, request);
	}

	protected Map collectReportParameters() {
		Map params = new HashMap();
		Map jobParams = jobDetails.getSource().getParametersMap();
		if (jobParams != null) {
			params.putAll(jobParams);
		}
		putAdditionalParameters(params);
		return params;
	}

	protected void putAdditionalParameters(Map parametersMap) {
		if (!parametersMap.containsKey(REPORT_PARAMETER_SCHEDULED_TIME)) {
			Date scheduledFireTime = jobContext.getScheduledFireTime();
			parametersMap.put(REPORT_PARAMETER_SCHEDULED_TIME, scheduledFireTime);
		}
		VirtualizerFactory virtualizerFactory = getVirtualizerFactory();
		if (virtualizerFactory != null)
		{
			parametersMap.put(JRParameter.REPORT_VIRTUALIZER, virtualizerFactory.getVirtualizer());
		}
	}

	protected void sendToDestinations(ReportUnitResult result) throws JobExecutionException {
		JasperPrint jasperPrint = result.getJasperPrint();
		
		String baseFilename = jobDetails.getBaseOutputFilename();
		if (jobDetails.getContentRepositoryDestination().isSequentialFilenames()) {
			Date scheduledTime = jobContext.getScheduledFireTime();
			SimpleDateFormat format = new SimpleDateFormat(REPOSITORY_FILENAME_TIMESTAMP_SEQUENCE_PATTERN);
			baseFilename = jobDetails.getBaseOutputFilename() + REPOSITORY_FILENAME_SEQUENCE_SEPARATOR + format.format(scheduledTime);
		} else {
			baseFilename = jobDetails.getBaseOutputFilename();
		}
		
		boolean sendMail = toSendMail(jasperPrint);
		List mailAttachments = null;
		if (sendMail) {
			ReportJobMailNotification mailNotification = jobDetails.getMailNotification();
			boolean hasMailAttachments = mailNotification != null && mailNotification.getResultSendType() == ReportJobMailNotification.RESULT_SEND_ATTACHMENT;
			if (hasMailAttachments) {
				mailAttachments = new ArrayList();
			}
		}
		
		Set outputFormats = jobDetails.getOutputFormats();
		for (Iterator it = outputFormats.iterator(); it.hasNext();) {
			Byte format = (Byte) it.next();
			ReportOutput output = getReportOutput(jasperPrint, format.byteValue(), baseFilename);
			saveToRepository(output);
			if (mailAttachments != null) {
				mailAttachments.add(output);
			}
		}

		if (sendMail) {
			sendMailNotification(mailAttachments, jobDetails);
		}		
	}

	protected boolean toSendMail(JasperPrint jasperPrint) {
		ReportJobMailNotification mailNotification = jobDetails.getMailNotification();
		return mailNotification != null 
				&& !(mailNotification.isSkipEmptyReports()
						&& isEmpty(jasperPrint));
	}

	protected boolean isEmpty(JasperPrint jasperPrint) {
		List pages = jasperPrint.getPages();
		boolean empty;
		if (pages == null || pages.isEmpty()) {
			empty = true;
		} else if (pages.size() == 1) {
			JRPrintPage page = (JRPrintPage) pages.get(0);
			List elements = page.getElements();
			empty = elements == null || elements.isEmpty();
		} else {
			empty = false;
		}
		return empty;
	}

	protected ReportOutput getReportOutput(JasperPrint jasperPrint, byte format, String baseFilename) {
		ReportOutput output;
		switch (format) {
		case ReportJob.OUTPUT_FORMAT_PDF: {
			output = getPdfOutput(jasperPrint, baseFilename);
			break;
		}
		case ReportJob.OUTPUT_FORMAT_HTML: {
			output = getHtmlOutput(jasperPrint, baseFilename);
			break;
		}
		case ReportJob.OUTPUT_FORMAT_XLS: {
			output = getXlsOutput(jasperPrint, baseFilename);
			break;
		}
		case ReportJob.OUTPUT_FORMAT_RTF: {
			output = getRtfOutput(jasperPrint, baseFilename);
			break;
		}
		case ReportJob.OUTPUT_FORMAT_CSV: {
			output = getCsvOutput(jasperPrint, baseFilename);
			break;
		}
		default:
			throw new JSException("jsexception.report.unknown.output.format", new Object[] {new Byte(format)});
		}
		return output;
	}

	protected String getChildrenFolderName(String resourceName) {
		return getRepository().getChildrenFolderName(resourceName);
	}
	
	protected ReportOutput getPdfOutput(JasperPrint jasperPrint, String baseFilename) {
		Map params = new HashMap();
		params.put(JRExporterParameter.JASPER_PRINT, jasperPrint);
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		params.put(JRExporterParameter.OUTPUT_STREAM, baos);
		PdfExportParametersBean exportParams =(PdfExportParametersBean)getExportParametersMap().get("pdf");
		String locale = LocaleContextHolder.getLocale().getLanguage();
		Map localizedFontMap = exportParams.getLocalizedFontMap().get(locale) != null ?
				(Map)exportParams.getLocalizedFontMap().get(locale) :
					(Map)exportParams.getLocalizedFontMap().get(locale.substring(0,2));
		if(localizedFontMap != null){
			params.put(JRExporterParameter.FONT_MAP, localizedFontMap);
		}
		getEngineService().exportToPdf(executionContext, getReportUnitURI(), params);
		byte[] pdfData = baos.toByteArray();
		String filename = baseFilename + ".pdf";
		return new ReportOutput(pdfData, ContentResource.TYPE_PDF, filename);
	}

	protected String getReportUnitURI() {
		return jobDetails.getSource().getReportUnitURI();
	}

	protected ReportOutput getHtmlOutput(JasperPrint jasperPrint, String baseFilename) {
		try {
			String filename = baseFilename + ".html";
			JRHtmlExporter exporter = new JRHtmlExporter();
			exporter.setParameter(JRExporterParameter.JASPER_PRINT, jasperPrint);
			ByteArrayOutputStream bout = new ByteArrayOutputStream();
			exporter.setParameter(JRExporterParameter.CHARACTER_ENCODING, getCharacterEncoding());
			exporter.setParameter(JRExporterParameter.OUTPUT_STREAM, bout);
			Map images = new HashMap();
			exporter.setParameter(JRHtmlExporterParameter.IMAGES_MAP, images);
			exporter.setParameter(JRHtmlExporterParameter.IMAGES_URI,
					getChildrenFolderName(filename) + '/');
			
			JRHyperlinkProducerFactory hyperlinkProducerFactory = getHyperlinkProducerFactory();
			if (hyperlinkProducerFactory != null) {
				exporter.setParameter(JRExporterParameter.HYPERLINK_PRODUCER_FACTORY, hyperlinkProducerFactory);
			}
			
			exporter.exportReport();

			byte[] htmlData = bout.toByteArray();
			ReportOutput htmlOutput = new ReportOutput(htmlData,
					ContentResource.TYPE_HTML, filename);

			for (Iterator it = images.entrySet().iterator(); it.hasNext();) {
				Map.Entry imageEntry = (Map.Entry) it.next();
				String imageName = (String) imageEntry.getKey();
				byte[] imageData = (byte[]) imageEntry.getValue();
				ReportOutput image = new ReportOutput(imageData,
						ContentResource.TYPE_HTML, imageName);//TODO type?
				htmlOutput.addChild(image);
			}

			return htmlOutput;
		} catch (JRException e) {
			throw new JSExceptionWrapper(e);
		}
	}

	protected JRHyperlinkProducerFactory getHyperlinkProducerFactory() {
		JRHyperlinkProducerFactory engineService = (JRHyperlinkProducerFactory) schedulerContext.get(SCHEDULER_CONTEXT_KEY_HYPERLINK_PRODUCER_FACTORY);
		return engineService;
	}

	protected ReportOutput getXlsOutput(JasperPrint jasperPrint, String baseFilename) {
		try {
			JExcelApiExporter exporter = new JExcelApiExporter();
			exporter.setParameter(JRExporterParameter.JASPER_PRINT, jasperPrint);
			ByteArrayOutputStream bout = new ByteArrayOutputStream();
			exporter.setParameter(JRExporterParameter.OUTPUT_STREAM, bout);
			exporter.setParameter(JRExporterParameter.HYPERLINK_PRODUCER_FACTORY, JExcelApiHyperlinkProducerFactory.INSTANCE);
			exporter.setParameter(JExcelApiExporterParameter.CREATE_CUSTOM_PALETTE, Boolean.TRUE);

			XlsExportParametersBean exportParams = (XlsExportParametersBean)getExportParametersMap().get("xls");
			if(exportParams != null)
			{
				if(exportParams.getOnePagePerSheet() != null);
					exporter.setParameter(JRXlsExporterParameter.IS_ONE_PAGE_PER_SHEET, exportParams.getOnePagePerSheet());
				if(exportParams.getDetectCellType() != null);
					exporter.setParameter(JRXlsExporterParameter.IS_DETECT_CELL_TYPE, exportParams.getDetectCellType());
				if(exportParams.getRemoveEmptySpaceBetweenRows() != null);
					exporter.setParameter(JRXlsExporterParameter.IS_REMOVE_EMPTY_SPACE_BETWEEN_ROWS, exportParams.getRemoveEmptySpaceBetweenRows());
				if(exportParams.getRemoveEmptySpaceBetweenColumns() != null);
					exporter.setParameter(JRXlsExporterParameter.IS_REMOVE_EMPTY_SPACE_BETWEEN_COLUMNS, exportParams.getRemoveEmptySpaceBetweenColumns());
				if(exportParams.getWhitePageBackground() != null);
					exporter.setParameter(JRXlsExporterParameter.IS_WHITE_PAGE_BACKGROUND, exportParams.getWhitePageBackground());
				if(exportParams.getIgnoreGraphics() != null);
					exporter.setParameter(JRXlsExporterParameter.IS_IGNORE_GRAPHICS, exportParams.getIgnoreGraphics());
				if(exportParams.getCollapseRowSpan() != null);
					exporter.setParameter(JRXlsExporterParameter.IS_COLLAPSE_ROW_SPAN, exportParams.getCollapseRowSpan());
				if(exportParams.getIgnoreCellBorder() != null);
					exporter.setParameter(JRXlsExporterParameter.IS_IGNORE_CELL_BORDER, exportParams.getIgnoreCellBorder());
				if(exportParams.getFontSizeFixEnabled() != null);
					exporter.setParameter(JRXlsExporterParameter.IS_FONT_SIZE_FIX_ENABLED, exportParams.getFontSizeFixEnabled());
				if(exportParams.getMaximumRowsPerSheet() != null);
					exporter.setParameter(JRXlsExporterParameter.MAXIMUM_ROWS_PER_SHEET, exportParams.getMaximumRowsPerSheet());
				if(exportParams.getXlsFormatPatternsMap() != null && !exportParams.getXlsFormatPatternsMap().isEmpty());
					exporter.setParameter(JRXlsExporterParameter.FORMAT_PATTERNS_MAP, exportParams.getXlsFormatPatternsMap());
			}	
			exporter.exportReport();

			byte[] xlsData = bout.toByteArray();
			String filename = baseFilename + ".xls";
			return new ReportOutput(xlsData, ContentResource.TYPE_XLS, filename);
		} catch (JRException e) {
			throw new JSExceptionWrapper(e);
		}
	}

	protected ReportOutput getRtfOutput(JasperPrint jasperPrint, String baseFilename) {
		try {
			JRRtfExporter exporter = new JRRtfExporter();
			exporter.setParameter(JRExporterParameter.JASPER_PRINT, jasperPrint);
			ByteArrayOutputStream bout = new ByteArrayOutputStream();
			exporter.setParameter(JRExporterParameter.OUTPUT_STREAM, bout);
			exporter.exportReport();

			byte[] rtfData = bout.toByteArray();
			String fileName = baseFilename + ".rtf";
			return new ReportOutput(rtfData, ContentResource.TYPE_RTF, fileName);
		} catch (JRException e) {
			throw new JSExceptionWrapper(e);
		}
	}

	protected ReportOutput getCsvOutput(JasperPrint jasperPrint, String baseFilename) {
		try {
			JRCsvExporter exporter = new JRCsvExporter();
			exporter.setParameter(JRExporterParameter.JASPER_PRINT, jasperPrint);
			ByteArrayOutputStream bout = new ByteArrayOutputStream();
			exporter.setParameter(JRExporterParameter.CHARACTER_ENCODING, getCharacterEncoding());
			exporter.setParameter(JRExporterParameter.OUTPUT_STREAM, bout);
			
			CsvExportParametersBean exportParams = (CsvExportParametersBean)getExportParametersMap().get("csv");
			if(exportParams != null)
				exporter.setParameter(JRCsvExporterParameter.FIELD_DELIMITER, exportParams.getFieldDelimiter());
			
			exporter.exportReport();

			byte[] csvData = bout.toByteArray();
			String fileName = baseFilename + ".csv";
			return new ReportOutput(csvData, ContentResource.TYPE_CSV, fileName);
		} catch (JRException e) {
			throw new JSExceptionWrapper(e);
		}
	}

	protected void saveToRepository(ReportOutput output) {		
		RepositoryService repositoryService = getRepository();
		ReportJobRepositoryDestination repositoryDestination = jobDetails.getContentRepositoryDestination();
		
		List children = output.getChildren();
		List childResources = new ArrayList(children.size());
		for (Iterator it = children.iterator(); it.hasNext();) {
			ReportOutput childOutput = (ReportOutput) it.next();
			
			ContentResource childRes = new ContentResourceImpl();		
			childRes.setName(childOutput.getFilename());
			childRes.setLabel(childOutput.getFilename());
			childRes.setFileType(childOutput.getFileType());
			childRes.setData(childOutput.getData());
			childResources.add(childRes);
		}

		ContentResource contentRes = null;
		if (repositoryDestination.isOverwriteFiles()) {
			String resURI = repositoryDestination.getFolderURI() + Folder.SEPARATOR + output.getFilename();
			Resource res = repositoryService.getResource(executionContext, resURI);
			if (res != null) {
				if (!(res instanceof ContentResource)) {
					String quotedResURI = "\"" + resURI + "\"";
					throw new JSException("jsexception.report.no.content.resource", new Object[] {quotedResURI});
				}
				contentRes = (ContentResource) res;
			}
		}
		
		if (contentRes == null) {
			contentRes = new ContentResourceImpl();
			contentRes.setName(output.getFilename());
			contentRes.setLabel(jobDetails.getBaseOutputFilename());
			contentRes.setParentFolder(repositoryDestination.getFolderURI());
		}
		
		contentRes.setFileType(output.getFileType());
		contentRes.setData(output.getData());
		contentRes.setResources(childResources);

		repositoryService.saveResource(null, contentRes);
	}

	protected RepositoryService getRepository() {
		RepositoryService repositoryService = (RepositoryService) schedulerContext.get(SCHEDULER_CONTEXT_KEY_REPOSITORY);
		return repositoryService;
	}

	protected void sendMailNotification(List mailAttachments, ReportJob job) throws JobExecutionException {
		ReportJobMailNotification mailNotification = job.getMailNotification();
		if (mailNotification != null) {
			JavaMailSender mailSender = getMailSender();
			String fromAddress = getFromAddress();
			try {
				MimeMessage message = mailSender.createMimeMessage();
				MimeMessageHelper messageHelper = new MimeMessageHelper(message, true, getCharacterEncoding());
				messageHelper.setFrom(fromAddress);				
				messageHelper.setSubject(mailNotification.getSubject());
				messageHelper.setText(mailNotification.getMessageText());
				
				addMailRecipients(mailNotification, messageHelper);
				
				if (mailAttachments != null) {
					for (Iterator it = mailAttachments.iterator(); it.hasNext();) {
						ReportOutput output = (ReportOutput) it.next();
						attachOutput(messageHelper, output);
					}
				}

				mailSender.send(message);
			} catch (MessagingException e) {
				log.error("Error while sending report job result mail", e);
				throw new JSExceptionWrapper(e);
			}
		}
	}

	protected String getCharacterEncoding() {
		CharacterEncodingProvider encodingProvider = (CharacterEncodingProvider) schedulerContext.get(SCHEDULER_CONTEXT_KEY_ENCODING_PROVIDER);
		return encodingProvider.getCharacterEncoding();
	}

	protected void addMailRecipients(ReportJobMailNotification mailNotification, MimeMessageHelper messageHelper) throws MessagingException {
		List toAddresses = mailNotification.getToAddresses();
		if (toAddresses != null && !toAddresses.isEmpty()) {
			String[] addressArray = new String[toAddresses.size()];
			toAddresses.toArray(addressArray);
			messageHelper.setTo(addressArray);
		}
		
		List ccAddresses = mailNotification.getCcAddresses();
		if (ccAddresses != null && !ccAddresses.isEmpty()) {
			String[] addressArray = new String[ccAddresses.size()];
			ccAddresses.toArray(addressArray);
			messageHelper.setCc(addressArray);
		}
		
		List bccAddresses = mailNotification.getBccAddresses();
		if (bccAddresses != null && !bccAddresses.isEmpty()) {
			String[] addressArray = new String[bccAddresses.size()];
			bccAddresses.toArray(addressArray);
			messageHelper.setBcc(addressArray);
		}
	}

	protected void attachOutput(MimeMessageHelper messageHelper, ReportOutput output) throws MessagingException, JobExecutionException {
		String attachmentName;
		byte[] attachmentData;
		if (output.getChildren().isEmpty()) {
			attachmentName = output.getFilename();
			attachmentData = output.getData();
		} else {
			ByteArrayOutputStream bout = new ByteArrayOutputStream();
			ZipOutputStream zipOut = new ZipOutputStream(bout);
			try {
				zipOut.putNextEntry(new ZipEntry(output.getFilename()));
				zipOut.write(output.getData());
				zipOut.closeEntry();
				
				for (Iterator it = output.getChildren().iterator(); it.hasNext();) {
					ReportOutput child = (ReportOutput) it.next();
					String childName = getChildrenFolderName(output.getFilename()) + '/' + child.getFilename();
					zipOut.putNextEntry(new ZipEntry(childName));
					zipOut.write(child.getData());
					zipOut.closeEntry();					
				}
				
				zipOut.finish();
				zipOut.flush();
			} catch (IOException e) {
				throw new JSExceptionWrapper(e);
			}

			attachmentData = bout.toByteArray();
			attachmentName = output.getFilename() + ".zip";
		}
		
		try {
			attachmentName = MimeUtility.encodeWord(attachmentName, getCharacterEncoding(), null);

			messageHelper.addAttachment(attachmentName, new ByteArrayResource(attachmentData));
		} catch (UnsupportedEncodingException e) {
			handleException(e);
		}
	}

	protected String getFromAddress() {
		String fromAddress = (String) schedulerContext.get(SCHEDULER_CONTEXT_KEY_MAIL_FROM_ADDRESS);
		return fromAddress;
	}

	protected JavaMailSender getMailSender() {
		JavaMailSender mailSender = (JavaMailSender) schedulerContext.get(SCHEDULER_CONTEXT_KEY_MAIL_SENDER);
		return mailSender;
	}
	
	protected Map getExportParametersMap() {
		return (Map) schedulerContext.get(SCHEDULER_CONTEXT_KEY_EXPORT_PARAMETRES_MAP);
	}

	
	protected static class ReportOutput {
		private final byte[] data;
		private final String fileType;
		private final String filename;
		private final List children;
		
		public ReportOutput(byte[] data, String fileType, String filename) {
			this.data = data;
			this.fileType = fileType;
			this.filename = filename;
			this.children = new ArrayList();
		}

		public byte[] getData() {
			return data;
		}

		public String getFilename() {
			return filename;
		}

		public String getFileType() {
			return fileType;
		}
		
		public List getChildren() {
			return children;
		}
		
		public void addChild(ReportOutput child) {
			children.add(child);
		}
	}
}
