/*
 * Copyright (C) 2005 - 2007 JasperSoft Corporation.  All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased a commercial license agreement from JasperSoft,
 * the following license terms apply:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published by
 * the Free Software Foundation.
 *
 * This program is distributed WITHOUT ANY WARRANTY; and without the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see http://www.gnu.org/licenses/gpl.txt
 * or write to:
 *
 * Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330,
 * Boston, MA  USA  02111-1307
 */
package com.jaspersoft.jasperserver.api.metadata.common.service.impl;

import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.jaspersoft.jasperserver.api.metadata.common.domain.Resource;
import com.jaspersoft.jasperserver.api.metadata.common.service.RepositoryService;


/**
 * This class adds security/permission checking utilities for RepositoryService
 */
public class RepositoryServiceSecurityChecker {
	
	private static final Log log = LogFactory.getLog(RepositoryServiceSecurityChecker.class);
	private RepositoryService securityChecker;
	
	public RepositoryServiceSecurityChecker() {
		
	}
    
	/** Filters allResources.
	 *  Populates removableResources with the resources that can be deleted.
	 *  Populates editableResources with the resources that can be edited.
	 */  
    public void filterResources(List allResources, Map removableResources, Map editableResources) {
		Iterator iter = allResources.iterator();
		while (iter.hasNext()) {
			Resource res = (Resource) iter.next();
			filterResource(res, removableResources, editableResources); 		
		}		
	}
    
    /** Filters resource.
	 *  Adds resource to removableResources if it can be deleted.
	 *  Adds resource to editableResources if it can be edited.
	 */  
    public void filterResource(Resource resource, Map removableResources, Map editableResources) {
		if (isEditable(resource)) {
			editableResources.put(resource.getURIString(), "true");
			log.debug(resource.getURIString()
					+ ": "
					+ (editableResources.containsKey(resource.getURIString()) ? "EDIT"
							: "??edit??"));
		}
		if (isRemovable(resource)) {
			removableResources.put(resource.getURIString(), "true");
			log.debug(resource.getURIString()
					+ ": "
					+ (removableResources.containsKey(resource.getURIString()) ? "DELETE"
							: "??delete??"));
		}
	}
    
    /** Checks whether the given resource can be edited */
    public boolean isEditable(Resource resource) {
    	try {
			securityChecker.saveResource(null,resource);
			return true;
		} catch (Exception e) {
			log.debug("No UPDATE permission for < " + resource.getURIString() + ">:" +  e.getMessage());
			return false;
		}
    }
    
    /** Checks whether the given resource can be deleted */
    public boolean isRemovable(Resource resource) {
    	try {
			securityChecker.deleteResource(null,resource.getURI());
			return true;
		} catch (Exception e) {
			log.debug("No DELETE permission for < " + resource.getURIString() + ">:" +  e.getMessage());
			return false;
		}	
    }

	public RepositoryService getSecurityChecker() {
		return securityChecker;
	}

	public void setSecurityChecker(RepositoryService securityChecker) {
		this.securityChecker = securityChecker;
	}
}
