/*
 * Copyright (C) 2005 - 2007 JasperSoft Corporation.  All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased a commercial license agreement from JasperSoft,
 * the following license terms apply:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published by
 * the Free Software Foundation.
 *
 * This program is distributed WITHOUT ANY WARRANTY; and without the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see http://www.gnu.org/licenses/gpl.txt
 * or write to:
 *
 * Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330,
 * Boston, MA  USA  02111-1307
 */
package com.jaspersoft.jasperserver.api.engine.jasperreports.service.impl;

import java.io.InputStream;
import java.util.HashMap;
import java.util.List;

import junit.framework.TestCase;
import junit.textui.TestRunner;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.JasperReport;

import org.springframework.context.support.ClassPathXmlApplicationContext;

import com.jaspersoft.jasperserver.api.common.domain.ExecutionContext;
import com.jaspersoft.jasperserver.api.common.domain.ValidationDetail;
import com.jaspersoft.jasperserver.api.common.domain.ValidationResult;
import com.jaspersoft.jasperserver.api.common.domain.impl.ExecutionContextImpl;
import com.jaspersoft.jasperserver.api.engine.common.domain.Result;
import com.jaspersoft.jasperserver.api.engine.common.service.EngineService;
import com.jaspersoft.jasperserver.api.engine.jasperreports.domain.impl.ReportUnitRequest;
import com.jaspersoft.jasperserver.api.engine.jasperreports.domain.impl.ReportUnitResult;
import com.jaspersoft.jasperserver.api.engine.jasperreports.domain.impl.TrialReportUnitRequest;
import com.jaspersoft.jasperserver.api.metadata.common.domain.FileResource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Resource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.ResourceReference;
import com.jaspersoft.jasperserver.api.metadata.common.service.RepositoryService;
import com.jaspersoft.jasperserver.api.metadata.jasperreports.domain.ReportUnit;


/**
 * @author Lucian Chirita (lucianc@users.sourceforge.net)
 * @version $Id: EngineServiceImplTest.java 8540 2007-06-04 02:42:13Z tony $
 */
public class EngineServiceImplTest extends TestCase
{
	private EngineService service;
	private RepositoryService repository;
	private ExecutionContext context;
	private ClassPathXmlApplicationContext appContext;

	public EngineServiceImplTest(String name){
		super(name);
	}


	protected void setUp() throws Exception {
		appContext = new ClassPathXmlApplicationContext(
				new String[]{"hibernateConfig.xml", "viewService.xml", "engine.xml", "customDataSources.xml"});

		service = (EngineService) appContext.getBean("engineService");
		repository = (RepositoryService) appContext.getBean("repoService");
		
		context = new ExecutionContextImpl();
	}

	protected void tearDown() throws Exception {
		try {
			service.release();
		} finally {
			appContext.close();
		}
	}


	public static void main(String[] args) {
		TestRunner.run(EngineServiceImplTest.class);
	}

	/**
	 * 
	 */
	public void testExecute() throws Exception
	{
		// make a reportunit, execute it, and delete it
		FileResource reportRes = (FileResource) repository.newResource(null, FileResource.class);
		reportRes.setFileType(FileResource.TYPE_JRXML);
		setCommon(reportRes, "EmployeesJRXML");
		InputStream jrxml = getClass().getResourceAsStream("/reports/jasper/Employees.jrxml");
		reportRes.readData(jrxml);
		ReportUnit unit = (ReportUnit) repository.newResource(null, ReportUnit.class);
		unit.setName("Employees_JDBC");
		unit.setLabel("Employees_JDBC");
		unit.setParentFolder("/reports/samples");
		unit.setDataSourceReference("/datasources/JServerJdbcDS");
		unit.setMainReport(reportRes);

		repository.saveResource(null, unit);
		
		ReportUnitResult result = (ReportUnitResult) service.execute(context, new ReportUnitRequest("/reports/samples/Employees_JDBC", new HashMap()));
		assertNotNull(result);
		JasperPrint jasperPrint = result.getJasperPrint();
		assertNotNull(jasperPrint);
		List pages = jasperPrint.getPages();
		assertNotNull(pages);
		assertTrue(pages.size() > 0);
		
		repository.deleteResource(null,"/reports/samples/Employees_JDBC");
	}

	/**
	 * 
	 */
	public void testGetResources() throws Exception
	{
		ReportUnit reportUnit = (ReportUnit) repository.getResource(context, "/reports/samples/AllAccounts");
		assertNotNull(reportUnit);
		ResourceReference reportRef = reportUnit.getMainReport();
		assertNotNull(reportRef);
		assertTrue(reportRef.isLocal());
		Resource report = reportRef.getLocalResource();
		assertNotNull(report);
		assertTrue(report instanceof FileResource);
		Resource[] resources = service.getResources(new ResourceReference(report));
		assertNotNull(resources);
		assertTrue(resources.length == 2);
	}

	public void testValidate() {
		ReportUnit unit = createUnit();

		ValidationResult result = service.validate(null, unit);
		assertNotNull(result);
		assertEquals(ValidationResult.STATE_ERROR, result.getValidationState());
		List results = result.getResults();
		assertNotNull(results);
		assertTrue(results.size() >= 1);
		ValidationDetail detail = (ValidationDetail) results.get(0);
		assertNotNull(detail);
		assertEquals("SalesByMonthTrialReport", detail.getName());

		addJar(unit);

		result = service.validate(null, unit);
		assertNotNull(result);
		assertEquals(ValidationResult.STATE_VALID, result.getValidationState());
	}

	public void testTrialExecute() {
		ReportUnit unit = createUnit();
		addJar(unit);
		
		TrialReportUnitRequest request = new TrialReportUnitRequest(unit, null);
		Result result = service.execute(null, request);
		assertNotNull(result);
		assertTrue(result instanceof ReportUnitResult);
		ReportUnitResult ruRes = (ReportUnitResult) result;
		JasperPrint print = ruRes.getJasperPrint();
		assertNotNull(print);
		List pages = print.getPages();
		assertNotNull(pages);
		assertTrue(pages.size() > 1);
	}

	private ReportUnit createUnit() {
		ReportUnit unit = (ReportUnit) repository.newResource(null,
				ReportUnit.class);
		setCommon(unit, "SalesByMonthTrial");
		unit.setParentFolder("/reports");

		FileResource mainReport = (FileResource) repository.newResource(null, FileResource.class);
		mainReport.setFileType(FileResource.TYPE_JRXML);
		setCommon(mainReport, "SalesByMonthTrialReport");
		mainReport.readData(getClass().getResourceAsStream("/reports/jasper/SalesByMonth.jrxml"));
		unit.setMainReport(mainReport);

		unit.setDataSourceReference("/datasources/JServerJdbcDS");

		FileResource img = (FileResource) repository.newResource(null, FileResource.class);
		img.setFileType(FileResource.TYPE_IMAGE);
		img.readData(getClass().getResourceAsStream("/images/jasperreports.gif"));
		setCommon(img, "Logo");
		unit.addResource(img);

		FileResource subrep = (FileResource) repository.newResource(null, FileResource.class);
		subrep.setFileType(FileResource.TYPE_JRXML);
		subrep.readData(getClass().getResourceAsStream("/reports/jasper/SalesByMonthDetail.jrxml"));
		setCommon(subrep, "SalesByMonthDetail");
		unit.addResource(subrep);

		FileResource resBdl = (FileResource) repository.newResource(null, FileResource.class);
		resBdl.setFileType(FileResource.TYPE_RESOURCE_BUNDLE);
		resBdl.readData(getClass().getResourceAsStream("/resource_bundles/sales.properties"));
		setCommon(resBdl, "sales.properties");
		unit.addResource(resBdl);

		return unit;
	}


	private void addJar(ReportUnit unit) {
		FileResource jar = (FileResource) repository.newResource(null,
				FileResource.class);
		jar.setFileType(FileResource.TYPE_JAR);
		jar.readData(getClass().getResourceAsStream("/jars/scriptlet.jar"));
		setCommon(jar, "Scriptlet");
		unit.addResource(jar);
	}
	

	private void setCommon(Resource res, String id) {
		res.setName(id);
		res.setLabel(id + "_label");
		res.setDescription(id + " description");
	}
	
	public void testGetMainJasperReport() {
		JasperReport jasperReport = service.getMainJasperReport(null, "/reports/samples/AllAccounts");
		assertNotNull(jasperReport);
		assertEquals("AllAccounts", jasperReport.getName());
	}

	public void testExecuteWithCustomDS() throws Exception
	{
		ReportUnitResult result = (ReportUnitResult) service.execute(context, new ReportUnitRequest("/reports/samples/DataSourceReport", new HashMap()));
		assertNotNull(result);
		JasperPrint jasperPrint = result.getJasperPrint();
		assertNotNull(jasperPrint);
		List pages = jasperPrint.getPages();
		assertNotNull(pages);
		assertTrue(pages.size() > 0);
	}

	public void testExecuteWithTableModelDS() throws Exception
	{
		ReportUnitResult result = (ReportUnitResult) service.execute(context, new ReportUnitRequest("/reports/samples/DataSourceTableModel", new HashMap()));
		assertNotNull(result);
		JasperPrint jasperPrint = result.getJasperPrint();
		assertNotNull(jasperPrint);
		List pages = jasperPrint.getPages();
		assertNotNull(pages);
		assertTrue(pages.size() > 0);
	}
	
	public void testRemoveCustomDataSourceResources() {
		deleteResource("/reports/samples/DataSourceReport");
		deleteResource("/reports/samples/DataSourceTableModel");
		deleteResource("/reports/samples/DataSourceReportTemplate");
		deleteResource("/datasources/CustomDSFromBean");
		deleteResource("/datasources/CustomTableModelDS");
	}
	
	public void deleteResource(String uri) {
		Resource result = repository.getResource(null, uri);
		assertNotNull(result);
		repository.deleteResource(null, uri);
	}
	
}
