/*
 * Copyright (C) 2005 - 2007 JasperSoft Corporation.  All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased a commercial license agreement from JasperSoft,
 * the following license terms apply:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published by
 * the Free Software Foundation.
 *
 * This program is distributed WITHOUT ANY WARRANTY; and without the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see http://www.gnu.org/licenses/gpl.txt
 * or write to:
 *
 * Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330,
 * Boston, MA  USA  02111-1307
 */
package com.jaspersoft.jasperserver.api.metadata;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.context.support.ClassPathXmlApplicationContext;

import com.jaspersoft.jasperserver.api.metadata.common.domain.FileResource;
import com.jaspersoft.jasperserver.api.metadata.common.service.RepositoryService;
import com.jaspersoft.jasperserver.api.metadata.jasperreports.domain.JdbcReportDataSource;
import com.jaspersoft.jasperserver.api.metadata.jasperreports.domain.ReportDataSource;
import com.jaspersoft.jasperserver.api.metadata.olap.domain.MondrianConnection;
import com.jaspersoft.jasperserver.api.metadata.olap.domain.MondrianXMLADefinition;
import com.jaspersoft.jasperserver.api.metadata.olap.domain.OlapClientConnection;
import com.jaspersoft.jasperserver.api.metadata.olap.domain.OlapUnit;
import com.jaspersoft.jasperserver.api.metadata.olap.service.OlapConnectionService;
import com.jaspersoft.jasperserver.api.metadata.olap.service.impl.OlapConnectionServiceImpl;

import junit.framework.TestCase;

/**
 * @author jshih
 */

public class OlapClientTest extends TestCase {

	protected final Log log = LogFactory.getLog(this.getClass());
	
	protected final String OLAP_SCHEMAS = "/olap/schemas/";

	protected final String OLAP_DATASOURCES = "/olap/datasources/";
	
	protected final String OLAP_CONNECTIONS = "/olap/connections/";
	
	protected final String OLAP_DEFINITIONS = "/olap/xmla/definitions/";
	
	protected final String OLAP_VIEWS = "/olap/views/";
	
    protected final String MDX_QUERY =
    	"SELECT {[Measures].[Unit Sales]} ON COLUMNS, " +
    	"{[Time].[1997].[Q1]} ON ROWS FROM [Sales]";

    protected final String XML_SCHEMA = "/olap/schemas/Foodmart.xml";
    
	private ClassPathXmlApplicationContext appContext;
	
	private Properties jdbcProps;

    private RepositoryService repositoryService;
    
    private OlapConnectionService connectionService;
 
	protected void setUp() throws Exception 
	{
	    appContext =
        	new ClassPathXmlApplicationContext(new String[] {"hibernateConfig.xml", "viewService.xml"});

		setRepositoryService((RepositoryService) appContext.getBean("repoService"));
		
		setConnectionService((OlapConnectionService) appContext.getBean("olapConnectionService"));

		loadJdbcProps();
	}
	
	protected void tearDown() {
		appContext.close();
	}
	
    // tests
    
    static FileResource schemaFoodmart;
    
    static FileResource schemaSugarCrm;
    
    public void testUpdateSchema() 
	{
        schemaFoodmart = 
        	(FileResource) repositoryService.newResource( null, FileResource.class );

        String strSchemaFoodmart = "FoodmartSchema";
    	
    	schemaFoodmart =  
    		(FileResource) repositoryService.getResource(null, OLAP_SCHEMAS + strSchemaFoodmart);
    	
    	assertTrue(schemaFoodmart.getParentFolder().equals("/olap/schemas"));
    	
    	assertTrue(schemaFoodmart.getName().equals(strSchemaFoodmart));

    	schemaFoodmart.readData(getClass().getResourceAsStream(XML_SCHEMA));
    	
    	assertNotNull(schemaFoodmart.getDataStream());
    	
    	String strSchemaSugarCrm = "SugarCRMOppsSchema";
    	
    	schemaSugarCrm =  
    		(FileResource) repositoryService.getResource(null, OLAP_SCHEMAS + strSchemaSugarCrm);
    	
    	assertTrue(schemaSugarCrm.getParentFolder().equals("/olap/schemas"));
    	
    	assertTrue(schemaSugarCrm.getName().equals(strSchemaSugarCrm));

    	schemaSugarCrm.readData(getClass().getResourceAsStream(XML_SCHEMA));
    	
    	assertNotNull(schemaSugarCrm.getDataStream());
    	
	}

	static ReportDataSource datasourceFoodmart;
	
   	static ReportDataSource datasourceSugarCrm;

    public void testUpdateDatasource() 
    {
    	String strDatasourceFoodmart = "FoodmartDataSource";

    	datasourceFoodmart =
    		(ReportDataSource) repositoryService.getResource(null, OLAP_DATASOURCES + strDatasourceFoodmart);
    	
    	assertTrue(datasourceFoodmart.getURIString().equals("/olap/datasources/FoodmartDataSource"));
    	
    	String strDatasourceSugarCrm = "SugarCRMDataSource";

    	datasourceSugarCrm =
    		(ReportDataSource) repositoryService.getResource(null, OLAP_DATASOURCES + strDatasourceSugarCrm);
    	
    	assertTrue(datasourceSugarCrm.getURIString().equals("/olap/datasources/SugarCRMDataSource"));
    	
//    	String strDsFoodmart = "FoodmartDS";
//        JdbcReportDataSource datasource =
//        	(JdbcReportDataSource) repositoryService.newResource( null, JdbcReportDataSource.class );
//        datasource.setName(strDsFoodmart);
//        datasource.setLabel(strDsFoodmart);
//        datasource.setDescription(strDsFoodmart);
//        datasource.setDriverClass(jdbcProps.getProperty("foodmart.jdbc.driverClassName"));
//        datasource.setConnectionUrl(jdbcProps.getProperty("foodmart.jdbc.url"));
//        datasource.setUsername(jdbcProps.getProperty("foodmart.jdbc.username"));
//        datasource.setPassword(jdbcProps.getProperty("foodmart.jdbc.password"));
////        getConnectionService().saveResource( null, "/olap/datasources/", datasource );
//        repositoryService.saveResource(null, datasource);
//        
//    	datasourceFoodmart =
//    		(ReportDataSource) repositoryService.getResource(null, OLAP_DATASOURCES + strDsFoodmart);
//    	
//    	assertTrue(datasourceFoodmart.getURIString().equals("/olap/datasources/FoodmartDS"));
    }
    
	static String strConnection;

	static OlapClientConnection connectionOriginal;

    private void setupTestUpdateMondrianConnection() {
    	
    	strConnection = "Foodmart";
    	
    	connectionOriginal =
    		(OlapClientConnection) repositoryService.getResource(null, OLAP_CONNECTIONS + strConnection);
    	
    	assertNotNull(connectionOriginal);
    	
    	assertTrue(connectionOriginal instanceof MondrianConnection);
    	
    	assertTrue(((MondrianConnection) connectionOriginal).getSchema().getReferenceLookup().getName()
    			.equals("FoodmartSchema"));
    	
    	assertTrue(((MondrianConnection) connectionOriginal).getSchema().getReferenceURI()
    			.equals("/olap/schemas/FoodmartSchema"));
    }
    
	public void testUpdateMondrianConnectionSchema() {

		setupTestUpdateMondrianConnection();
    	
    	// new schema
    	((MondrianConnection) connectionOriginal).setSchemaReference(schemaSugarCrm.getURIString());
        
//    	connectionService.saveResource(null, "/olap/connections/", connectionOriginal);
    	repositoryService.saveResource(null, connectionOriginal);
    	
    	OlapClientConnection connectionNew =
    		(OlapClientConnection) repositoryService.getResource(null, OLAP_CONNECTIONS + strConnection);
    	
    	assertNotNull(connectionNew);
    	
    	assertTrue(connectionNew instanceof MondrianConnection);
    	
    	assertTrue(((MondrianConnection) connectionNew).getSchema().getReferenceLookup().getName()
    			.equals("SugarCRMOppsSchema"));
    	
    	assertTrue(((MondrianConnection) connectionNew).getSchema().getReferenceURI()
    			.equals("/olap/schemas/SugarCRMOppsSchema"));
    	
    }
    
    public void testUpdateMondrianConnectionDatasource() {
    	
    	setupTestUpdateMondrianConnection();
    	
    	// new datasource 
    	((MondrianConnection) connectionOriginal).setDataSourceReference(datasourceSugarCrm.getURIString());
    	
//    	connectionService.saveResource(null, "/olap/connections/", connectionOriginal);
    	repositoryService.saveResource(null, connectionOriginal);
    	
    	OlapClientConnection connectionNew =
    		(OlapClientConnection) repositoryService.getResource(null, OLAP_CONNECTIONS + strConnection);
    	
    	assertNotNull(connectionNew);
    	
    	assertTrue(connectionNew instanceof MondrianConnection);
    	
    	assertTrue(((MondrianConnection) connectionNew).getDataSource().getReferenceLookup().getName()
    			.equals("SugarCRMDataSource"));
    	
    	assertTrue(((MondrianConnection) connectionNew).getDataSource().getReferenceURI()
    			.equals("/olap/datasources/SugarCRMDataSource"));
    	
    }

    public void testUpdateXmlaDefinition() 
    {
    	String strDefinition = "FoodmartXmlaDefinition";
    	
    	MondrianXMLADefinition definition = 
//    		(MondrianXMLADefinition) repository.getResource(null, OLAP_DEFINITIONS + strDefinition);
    		(MondrianXMLADefinition) repositoryService.getResource(null, "/olap/xmla/definitions/" + strDefinition);
    	
    	assertNotNull(definition.getURIString()); // null
    }
    
    public void testUpdateXmlaConnection() 
    {
    	String strConnection = "FoodmartXmlaConnection";
    	
    	OlapClientConnection connection =
//    		(OlapClientConnection) repository.getResource(null, OLAP_CONNECTIONS + strConnection);
    		(OlapClientConnection) repositoryService.getResource(null, "/olap/connections/" + strConnection);
    	
    	assertNotNull(connection);
    }
    
    public void testUpdateOlapModel() 
    {
    	String olapViewRef = "/olap/connections/Foodmart";
    	
    	String olapQuery = MDX_QUERY;
    	
        mondrian.olap.Connection mondrianConnection =
        	((OlapConnectionServiceImpl) getConnectionService()).getMondrianConnection(null, olapViewRef);

        mondrian.olap.Query mondrianQuery = mondrianConnection.parseQuery(olapQuery);
        
        assertNotNull(mondrianQuery);
        
        assertTrue(mondrianQuery.getCube().toString().equals("Sales"));
        
//        assertTrue(mondrianQuery.axes[0].exp.toString().equals("{[Measures].[Unit Sales]}"));
//        
//        assertTrue(mondrianQuery.axes[1].exp.toString().equals("{[Time].[1997].[Q1]}"));
    }
    
    public void testUpdateOlapView() 
    {
    	String strOlapView = "Foodmart_sample_unit_1";
    	
    	OlapUnit olapView =
    		(OlapUnit) repositoryService.getResource(null, OLAP_VIEWS + strOlapView);
    	
    	assertNotNull(olapView.getOlapClientConnection());
    	
    	assertTrue(olapView.getMdxQuery()
    			.equals("SELECT {[Measures].[Unit Sales]} ON COLUMNS, {[Time].[1997].[Q1]} ON ROWS FROM [Sales]"));
    }

    // helpers

    protected Properties loadJdbcProps() throws IOException, FileNotFoundException 
	{
		jdbcProps = new Properties();
		
		String jdbcPropFile = System.getProperty("test.hibernate.jdbc.properties");
		
		BufferedInputStream is = new BufferedInputStream(new FileInputStream(jdbcPropFile));
		
		jdbcProps.load(is);
		
		is.close();
		
		return jdbcProps;
	}
	
    public RepositoryService getRepositoryService() 
    {
        return repositoryService;
    }
    
    public void setRepositoryService(RepositoryService repository) 
    {	
        this.repositoryService = repository;
    }

    public OlapConnectionService getConnectionService() 
    {
        return connectionService;
    }

    public void setConnectionService(OlapConnectionService connection) 
    {
        this.connectionService = connection;
    }
}
