/*
 * Copyright (C) 2005 - 2007 JasperSoft Corporation.  All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased a commercial license agreement from JasperSoft,
 * the following license terms apply:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published by
 * the Free Software Foundation.
 *
 * This program is distributed WITHOUT ANY WARRANTY; and without the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see http://www.gnu.org/licenses/gpl.txt
 * or write to:
 *
 * Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330,
 * Boston, MA  USA  02111-1307
 */
package com.jaspersoft.jasperserver.api.metadata;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import junit.framework.TestCase;
import junit.textui.TestRunner;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.context.support.ClassPathXmlApplicationContext;

import com.jaspersoft.jasperserver.api.common.domain.ExecutionContext;
import com.jaspersoft.jasperserver.api.common.domain.ValidationResult;
import com.jaspersoft.jasperserver.api.common.domain.impl.ExecutionContextImpl;
import com.jaspersoft.jasperserver.api.metadata.common.domain.FileResource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Folder;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Resource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.ResourceLookup;
import com.jaspersoft.jasperserver.api.metadata.common.domain.ResourceReference;
import com.jaspersoft.jasperserver.api.metadata.common.domain.client.FolderImpl;
import com.jaspersoft.jasperserver.api.metadata.common.service.RepositoryService;
import com.jaspersoft.jasperserver.api.metadata.jasperreports.domain.JdbcReportDataSource;
import com.jaspersoft.jasperserver.api.metadata.jasperreports.domain.JndiJdbcReportDataSource;
import com.jaspersoft.jasperserver.api.metadata.jasperreports.domain.ReportUnit;
import com.jaspersoft.jasperserver.api.metadata.olap.domain.MondrianConnection;
import com.jaspersoft.jasperserver.api.metadata.olap.domain.MondrianXMLADefinition;
import com.jaspersoft.jasperserver.api.metadata.olap.domain.OlapUnit;
import com.jaspersoft.jasperserver.api.metadata.olap.domain.XMLAConnection;
import com.jaspersoft.jasperserver.api.metadata.olap.service.OlapConnectionService;
import com.jaspersoft.jasperserver.api.metadata.olap.service.impl.OlapConnectionServiceImpl;
import com.jaspersoft.jasperserver.api.metadata.view.domain.FilterCriteria;
import com.jaspersoft.jasperserver.api.metadata.xml.domain.impl.ResourceDescriptor;
import com.tonbeller.jpivot.olap.model.OlapException;
import com.tonbeller.jpivot.olap.model.OlapModel;
import com.tonbeller.jpivot.olap.model.Result;
import mondrian.olap.Member;
import mondrian.olap.Position;
import mondrian.olap.Axis;

/**
 * @author sbirney
 */

public class OlapConnectionTest extends TestCase {

    private static final Log log = LogFactory.getLog(OlapConnectionTest.class);

    protected final String foodMartSchemaURI   = "/queries/FoodMart.xml";
    protected final String foodMartSchema2006URI   = "/queries/FoodmartSchema2006.xml";
    protected final String foodMartJaSchemaURI = "/queries/FoodMart_ja.xml";
    protected final String sugarCRMSchemaURI   = "/queries/SugarCRMOpps.xml";
    protected final String sugarCRMSchemaUpperURI   = "/queries/SugarcrmSchemaUpper.xml";

	private ClassPathXmlApplicationContext appContext;
	
    private Properties jdbcProps;
    
    private ExecutionContext executionContext;

    public static void main(String[] args) {
        TestRunner.run(OlapConnectionTest.class);
    }

    protected void setUp() throws Exception {

        appContext
            = new ClassPathXmlApplicationContext(new String[] {"hibernateConfig.xml", "viewService.xml"});

        setRepository( (RepositoryService)appContext.getBean("repoService") );
        setConnectionService( (OlapConnectionService)appContext.getBean("olapConnectionService") );

        loadJdbcProps();
        
        executionContext = new ExecutionContextImpl();
    }
	
	protected void tearDown() {
		
		/*
		 * Remove repository objects that are not samples in the released product
		 * (ie. objects that users see after installing the application)
		 */
		Resource savedResource;
		/*
		savedResource = mRepository.getResource(executionContext, "/analysis/reports/FoodmartSalesXmlaReport");
		mRepository.delete(executionContext, new String[]{savedResource.getURI()}, null );

		savedResource = mRepository.getResource(executionContext, "/analysis/views/Foodmart_xmla_sample");
		mRepository.delete(executionContext, new String[]{savedResource.getURI()}, null );
		*/

		appContext.close();
	}

    protected Properties loadJdbcProps() throws IOException, FileNotFoundException {
    	jdbcProps = new Properties();
		String jdbcPropFile = System.getProperty("test.hibernate.jdbc.properties");
		BufferedInputStream is = new BufferedInputStream(new FileInputStream(jdbcPropFile));
		jdbcProps.load(is);
		is.close();
		return jdbcProps;
    }
    
    // updated to use FoodmartSchema2006.xml for SuperMart Demo project
    protected void createFoodmartSchemaResource() {
        RepositoryService rep = getRepository();
        FileResource schemaResource = (FileResource) rep.newResource( null, FileResource.class );
        schemaResource.setName( "FoodmartSchema" );
        schemaResource.setLabel( "Foodmart Schema" );
        schemaResource.setDescription( "Foodmart Analysis Schema" );
        schemaResource.setFileType(ResourceDescriptor.TYPE_MONDRIAN_SCHEMA);
        InputStream in = getClass().getResourceAsStream( foodMartSchema2006URI );
        schemaResource.readData(in);
        getConnectionService().saveResource( null, "/analysis/schemas/", schemaResource );
    }

    protected void createFoodmartJaSchemaResource() {
        RepositoryService rep = getRepository();
        FileResource schemaResource = (FileResource) rep.newResource( null, FileResource.class );
        schemaResource.setName( "FoodmartJaSchema" );
        schemaResource.setLabel( "Foodmart Ja Schema" );
        schemaResource.setDescription( "Foodmart Analysis Pseudo Japanese Schema" );
        schemaResource.setFileType(ResourceDescriptor.TYPE_MONDRIAN_SCHEMA);
        InputStream in = getClass().getResourceAsStream( foodMartJaSchemaURI );
        schemaResource.readData(in);
        getConnectionService().saveResource( null, "/analysis/schemas/", schemaResource );
    }

    protected void createSugarCRMSchemaResource() {
        RepositoryService rep = getRepository();
        FileResource schemaResource = (FileResource) rep.newResource( null, FileResource.class );
        schemaResource.setName( "SugarCRMOppsSchema" );
        schemaResource.setLabel( "SugarCRM Opportunities DataMart" );
        schemaResource.setDescription( "SugarCRM Opportunities Data Mart Schema" );
        schemaResource.setFileType(ResourceDescriptor.TYPE_MONDRIAN_SCHEMA);
        InputStream in = getClass().getResourceAsStream( sugarCRMSchemaURI );
        schemaResource.readData(in);
        getConnectionService().saveResource(null, "/analysis/schemas/", schemaResource );
    }

    protected void createSugarCRMSchemaUpperResource() {
        RepositoryService rep = getRepository();
        FileResource schemaResource = (FileResource) rep.newResource( null, FileResource.class );
        schemaResource.setName( "SugarcrmSchemaUpper" );
        schemaResource.setLabel( "SugarCRM Schema Uppercase" );
        schemaResource.setDescription( "SugarCRM Schema Uppercase" );
        schemaResource.setFileType(ResourceDescriptor.TYPE_MONDRIAN_SCHEMA);
        InputStream in = getClass().getResourceAsStream( sugarCRMSchemaUpperURI );
        schemaResource.readData(in);
        getConnectionService().saveResource(null, "/analysis/schemas/", schemaResource );
    }    
    
    protected void createFoodmartDataSourceResource() {
        RepositoryService rep = getRepository();
        JdbcReportDataSource dataSourceResource
            = (JdbcReportDataSource) rep.newResource( null, JdbcReportDataSource.class );
        dataSourceResource.setName( "FoodmartDataSource" );
        dataSourceResource.setLabel( "Foodmart Data Source" );
        dataSourceResource.setDescription( "Foodmart Data Source" );
        dataSourceResource.setDriverClass(jdbcProps.getProperty("foodmart.jdbc.driverClassName"));
        dataSourceResource.setConnectionUrl(jdbcProps.getProperty("foodmart.jdbc.url"));
        dataSourceResource.setUsername(jdbcProps.getProperty("foodmart.jdbc.username"));
        dataSourceResource.setPassword(jdbcProps.getProperty("foodmart.jdbc.password"));
        getConnectionService().saveResource( null, "/analysis/datasources/", dataSourceResource );
    }

    protected void createFoodmartDataSourceResourceJNDI() {
        RepositoryService rep = getRepository();
        JndiJdbcReportDataSource dataSourceResource
            = (JndiJdbcReportDataSource) rep.newResource( null, JndiJdbcReportDataSource.class );
        dataSourceResource.setName( "FoodmartDataSourceJNDI" );
        dataSourceResource.setLabel( "Foodmart Data Source JNDI" );
        dataSourceResource.setDescription( "Foodmart Data Source JNDI" );
        dataSourceResource.setJndiName("jdbc/foodmart");
        getConnectionService().saveResource( null, "/analysis/datasources/", dataSourceResource );
    }
    
    /*
     * datasource for foodmart Derby
     */
    /* Comment out foodmart derby work
    protected void createFoodmartDerbyDataSourceResourceJNDI() {
        RepositoryService rep = getRepository();
        JndiJdbcReportDataSource dataSourceResource
            = (JndiJdbcReportDataSource) rep.newResource( null, JndiJdbcReportDataSource.class );
        dataSourceResource.setName( "FoodmartDerbyJNDI" );
        dataSourceResource.setLabel( "Foodmart Derby JNDI" );
        dataSourceResource.setDescription( "Foodmart Derby JNDI" );
        dataSourceResource.setJndiName("jdbc/foodmartDerby");
        getConnectionService().saveResource( null, "/analysis/datasources/", dataSourceResource );
    } 
    */   
    
    protected void createFoodmartJaDataSourceResource() {
        RepositoryService rep = getRepository();
        JdbcReportDataSource dataSourceResource
            = (JdbcReportDataSource) rep.newResource( null, JdbcReportDataSource.class );
        dataSourceResource.setName( "FoodmartJaDataSource" );
        dataSourceResource.setLabel( "FoodmartJa Data Source" );
        dataSourceResource.setDescription( "FoodmartJa Data Source" );
        dataSourceResource.setDriverClass(jdbcProps.getProperty("foodmart_ja.jdbc.driverClassName"));
        dataSourceResource.setConnectionUrl(jdbcProps.getProperty("foodmart_ja.jdbc.url"));
        dataSourceResource.setUsername(jdbcProps.getProperty("foodmart_ja.jdbc.username"));
        dataSourceResource.setPassword(jdbcProps.getProperty("foodmart_ja.jdbc.password"));
        getConnectionService().saveResource( null, "/analysis/datasources/", dataSourceResource );
    }

    protected void createSugarCRMDataSourceResource() {
        RepositoryService rep = getRepository();
        JdbcReportDataSource dataSourceResource
            = (JdbcReportDataSource) rep.newResource( null, JdbcReportDataSource.class );
        dataSourceResource.setName( "SugarCRMDataSource" );
        dataSourceResource.setLabel( "SugarCRM Data Source" );
        dataSourceResource.setDescription( "SugarCRM Data Source" );
        dataSourceResource.setDriverClass(jdbcProps.getProperty("test.jdbc.driverClassName"));
        dataSourceResource.setConnectionUrl(jdbcProps.getProperty("test.jdbc.url"));
        dataSourceResource.setUsername(jdbcProps.getProperty("test.jdbc.username"));
        dataSourceResource.setPassword(jdbcProps.getProperty("test.jdbc.password"));
        getConnectionService().saveResource( null, "/analysis/datasources/", dataSourceResource );
    }

    protected void createSugarCRMDataSourceResourceJNDI() {
        RepositoryService rep = getRepository();
        JndiJdbcReportDataSource dataSourceResource
            = (JndiJdbcReportDataSource) rep.newResource( null, JndiJdbcReportDataSource.class );
        dataSourceResource.setName( "SugarCRMDataSourceJNDI" );
        dataSourceResource.setLabel( "SugarCRM Data Source JNDI" );
        dataSourceResource.setDescription( "SugarCRM Data Source JNDI" );
        dataSourceResource.setJndiName("jdbc/sugarcrm");
        getConnectionService().saveResource( null, "/analysis/datasources/", dataSourceResource );
    }
    
    protected void createFoodmartMondrianConnectionResource() {
        RepositoryService rep = getRepository();
        MondrianConnection connection = (MondrianConnection) rep.newResource( null, MondrianConnection.class );
        connection.setName("Foodmart");
        connection.setLabel("Foodmart Mondrian Connection");
        connection.setDescription("Foodmart Analysis Connection");
        connection.setSchemaReference("/analysis/schemas/FoodmartSchema");
        connection.setDataSourceReference("/analysis/datasources/FoodmartDataSourceJNDI");
        getConnectionService().saveResource( null, "/analysis/connections/", connection );
    }

    /*
     * Mondian connection for foodmart Derby 
     */
    /* Comment out foodmart derby work
    protected void createFoodmartDerbyMondrianConnectionResource() {
        RepositoryService rep = getRepository();
        MondrianConnection connection = (MondrianConnection) rep.newResource( null, MondrianConnection.class );
        connection.setName("FoodmartDerbyConnection");
        connection.setLabel("Foodmart Derby Connection");
        connection.setDescription("Foodmart Derby Connection");
        connection.setSchemaReference("/analysis/schemas/FoodmartSchema");
        connection.setDataSourceReference("/analysis/datasources/FoodmartDerbyJNDI");
        getConnectionService().saveResource( null, "/analysis/connections/", connection );
    }
    */
    
    protected void createFoodmartJaMondrianConnectionResource() {
        RepositoryService rep = getRepository();
        MondrianConnection connection = (MondrianConnection) rep.newResource( null, MondrianConnection.class );
        connection.setName("FoodmartJa");
        connection.setLabel("FoodmartJa Mondrian Connection");
        connection.setDescription("FoodmartJa Analysis Connection");
        connection.setSchemaReference("/analysis/schemas/FoodmartJaSchema");
        connection.setDataSourceReference("/analysis/datasources/FoodmartJaDataSource");
        getConnectionService().saveResource( null, "/analysis/connections/", connection );
    }

    protected void createSugarCRMMondrianConnectionResource() {
        RepositoryService rep = getRepository();
        MondrianConnection connection = (MondrianConnection) rep.newResource( null, MondrianConnection.class );
        connection.setName("SugarCRM");
        connection.setLabel("SugarCRM Analysis Connection");
        connection.setDescription("SugarCRM Analysis Connection: only opportunities");
        connection.setSchemaReference("/analysis/schemas/SugarCRMOppsSchema");
        connection.setDataSourceReference("/analysis/datasources/SugarCRMDataSourceJNDI");  
        getConnectionService().saveResource(null,  "/analysis/connections/", connection );
    }
        
    protected void createSugarCRMTestMondrianConnectionResource() {
        RepositoryService rep = getRepository();
        MondrianConnection connection = (MondrianConnection) rep.newResource( null, MondrianConnection.class );
        connection.setName("SugarCRMTest");
        connection.setLabel("SugarCRM Analysis Connection");
        connection.setDescription("SugarCRM Analysis Connection: only opportunities");
        connection.setSchemaReference("/analysis/schemas/SugarCRMOppsSchema");
        connection.setDataSourceReference("/analysis/datasources/SugarCRMDataSource");  
        getConnectionService().saveResource(null,  "/analysis/connections/", connection );
    }

    protected void createFoodmartXMLAConnectionResource() {
        RepositoryService rep = getRepository();
        XMLAConnection connection = (XMLAConnection) rep.newResource( null, XMLAConnection.class );
        connection.setName("FoodmartXmlaConnection");
        connection.setLabel("Foodmart XMLA Connection");
        connection.setDescription("Foodmart XML/A Connection");
        connection.setCatalog("Foodmart");
        connection.setDataSource("Provider=Mondrian;DataSource=Foodmart;");
        connection.setURI("http://localhost:8080/jasperserver/xmla");
	// TODO: create a sample USER_XMLA and ROLE_XMLA_USER
	connection.setUsername("tomcat"); 
	connection.setPassword("tomcat"); 
        getConnectionService().saveResource( null, "/analysis/connections/", connection );
    }

    protected void createSugarCRMXMLAConnectionResource() {
        RepositoryService rep = getRepository();
        XMLAConnection connection = (XMLAConnection) rep.newResource( null, XMLAConnection.class );
        connection.setName("SugarCRMXmlaConnection");
        connection.setLabel("SugarCRM XMLA Connection");
        connection.setDescription("SugarCRM XML/A Connection");
        connection.setCatalog("SugarCRM");
        connection.setDataSource("Provider=Mondrian;DataSource=SugarCRM;");
        connection.setURI("http://localhost:8080/jasperserver/xmla");
	// TODO: create a sample USER_XMLA and ROLE_XMLA_USER
	connection.setUsername("tomcat"); 
	connection.setPassword("tomcat"); 
        getConnectionService().saveResource(null,  "/analysis/connections/", connection );
    }

    protected void createFoodmartMondrianXMLADefinitionResource() {
        RepositoryService rep = getRepository();
        MondrianXMLADefinition def = (MondrianXMLADefinition) rep.newResource( null, MondrianXMLADefinition.class );
        def.setName("FoodmartXmlaDefinition");
        def.setLabel("Foodmart XMLA Connection");
        def.setDescription("Foodmart XML/A Connection");
        def.setCatalog("Foodmart");
        def.setMondrianConnectionReference("/analysis/connections/Foodmart");

        getConnectionService().saveResource( null, "/analysis/xmla/definitions/", def );
    }

    protected void createSugarCRMMondrianXMLADefinitionResource() {
        RepositoryService rep = getRepository();
        MondrianXMLADefinition def = (MondrianXMLADefinition) rep.newResource( null, MondrianXMLADefinition.class );
        def.setName("SugarCRMXmlaDefinition");
        def.setLabel("SugarCRM XMLA Connection");
        def.setDescription("SugarCRM XML/A Connection");
        def.setCatalog("SugarCRM");
        def.setMondrianConnectionReference("/analysis/connections/SugarCRM");

        getConnectionService().saveResource( null, "/analysis/xmla/definitions/", def );
    }

    private String SAMPLE_FOODMART_JA_MDX_QUERY =
	"select {[Measures].[Unit Sales日本語], [Measures].[Store Cost日本語], [Measures].[Store Sales日本語]} on columns, {([Promotion Media日本語].[All Media日本語], [Product日本語].[All Products日本語])} ON rows from [Sales日本語] where ([Time日本語].[2006])";


    private String SAMPLE_FOODMART_MDX_QUERY =
	"select {[Measures].[Unit Sales], [Measures].[Store Cost], [Measures].[Store Sales]} on columns, " +
	"{([Promotion Media].[All Media], [Product].[All Products])} ON rows " + 
	"from Sales " +
	"where ([Time].[2006].[Q4].[12])";


    protected void createFoodmartJaOlapUnit() {
	RepositoryService rep = getRepository();
	OlapUnit view = (OlapUnit) rep.newResource( null, OlapUnit.class );
	view.setName("FoodmartJa_sample_unit_1");
	view.setLabel("FoodmartJa Sample Analysis View");
	view.setDescription("Sample Analysis View: 2006 Q1 FoodmartJa Unit Sales");
	view.setOlapClientConnectionReference("/analysis/connections/FoodmartJa");
	view.setMdxQuery(SAMPLE_FOODMART_JA_MDX_QUERY);
	getConnectionService().saveResource( null, "/analysis/views/", view );
	OlapUnit cycledView = (OlapUnit)rep.getResource(null, 
							"/analysis/views/FoodmartJa_sample_unit_1");
    }

    protected void createFoodmartOlapUnit() {
	RepositoryService rep = getRepository();
	OlapUnit view = (OlapUnit) rep.newResource( null, OlapUnit.class );
	view.setName("Foodmart_sample");
	view.setLabel("Foodmart Sample Analysis View");
	view.setDescription("Sample Analysis View: 2006 Q1 Foodmart Unit Sales");
	view.setOlapClientConnectionReference("/analysis/connections/Foodmart");
	view.setMdxQuery(SAMPLE_FOODMART_MDX_QUERY);
	getConnectionService().saveResource( null, "/analysis/views/", view );
	OlapUnit cycledView = (OlapUnit)rep.getResource(null, 
							"/analysis/views/Foodmart_sample");
	// calling validate here with view instead of cycledView throws null pointer
	// XXX jasperserver bug?

	//something tells me foodmart isn't set up right on aphid
	//assertTrue( "validation of Foodmart Olap Unit failed", 
	//	    getConnectionService().validate(null, cycledView).
	//	    getValidationState() != ValidationResult.STATE_ERROR );
    }

    /*
     * Make a view for foodmart in the embedded Derby database
     */
    /* Comment out foodmart derby work
    protected void createFoodmartDerbyOlapUnit() {
    	RepositoryService rep = getRepository();
    	OlapUnit view = (OlapUnit) rep.newResource( null, OlapUnit.class );
    	view.setName("FoodmartDerbyView");
    	view.setLabel("Foodmart Derby View");
    	view.setDescription("Sample Foodmart Derby View");
    	view.setOlapClientConnectionReference("/analysis/connections/FoodmartDerbyConnection");
    	view.setMdxQuery(SAMPLE_FOODMART_MDX_QUERY);
    	getConnectionService().saveResource( null, "/analysis/views/", view );
    	OlapUnit cycledView = (OlapUnit)rep.getResource(null, 
    							"/analysis/views/FoodmartDerbyView");
    }
    */
    
    
    protected void createSugarCRMXmlaOlapUnit() {
	RepositoryService rep = getRepository();
	OlapUnit view = (OlapUnit) rep.newResource( null, OlapUnit.class );
	view.setName("SugarCRM_xmla_sample");
	view.setLabel("SugarCRM Sample XMLA Analysis View");
	view.setDescription("Sample SugarCRM Analysis View (XMLA): Sales Performance by Industry/Account");
	view.setOlapClientConnectionReference("/analysis/connections/SugarCRMXmlaConnection");
	view.setMdxQuery(SAMPLE_SUGAR_CRM_MDX_QUERY);
	getConnectionService().saveResource( null, "/analysis/views/", view );
    }

    protected void createFoodmartXmlaOlapUnit() {
	RepositoryService rep = getRepository();
	OlapUnit view = (OlapUnit) rep.newResource( null, OlapUnit.class );
	view.setName("Foodmart_xmla_sample");
	view.setLabel("Foodmart Sample XMLA Analysis View");
	view.setDescription("Sample XMLA Analysis View: 2006 Q1 Foodmart Unit Sales");
	view.setOlapClientConnectionReference("/analysis/connections/FoodmartXmlaConnection");
	view.setMdxQuery(SAMPLE_FOODMART_MDX_QUERY);
	getConnectionService().saveResource( null, "/analysis/views/", view );
    }

    private String SAMPLE_SUGAR_CRM_MDX_QUERY =
	"select {[Measures].[Total Sale Amount], [Measures].[Number of Sales], [Measures].[Avg Sale Amount], [Measures].[Avg Time To Close (Days)], [Measures].[Avg Close Probability]} ON COLUMNS, " +
	" NON EMPTY {([Account Categorization].[All Accounts], [Close Period].[All Periods])} ON ROWS " +
	" from [SalesAnalysis] " +
	" where [Sale State].[All Types].[Closed Won]";

    protected void createSugarCRMOlapUnit() {
	RepositoryService rep = getRepository();
	OlapUnit view = (OlapUnit) rep.newResource( null, OlapUnit.class );
	view.setName("SugarCRM_sample");
	view.setLabel("SugarCRM Sample view");
	view.setDescription("SugarCRM Sample Analysis View: Sales Performance by Industry/Account");
	view.setOlapClientConnectionReference("/analysis/connections/SugarCRM");
	view.setMdxQuery(SAMPLE_SUGAR_CRM_MDX_QUERY);
	getConnectionService().saveResource(null,  "/analysis/views/", view );
	//OlapUnit cycledView = (OlapUnit)rep.getResource(null, 
	//"/analysis/views/SugarCRM_sample");
	// calling validate here with view instead of cycledView throws null pointer
	// XXX jasperserver bug?
	//assertTrue( "validation of SugarCRM Olap Unit failed", 
	//	    getConnectionService().validate(null, cycledView).
	//	    getValidationState() != ValidationResult.STATE_ERROR );
    }

    
    protected void executeSampleFoodmartQuery() {
	assertQueryShape( "/analysis/connections/Foodmart",
			  SAMPLE_FOODMART_MDX_QUERY,
			  "", 0, 0);
    }

    protected void executeSampleSugarQuery() {
	assertQueryShape( "/analysis/connections/SugarCRM",
			  SAMPLE_SUGAR_CRM_MDX_QUERY,
			  "Total Sale Amount", 2, 5);
    }

    protected void executeSampleSugarTestQuery() {
	assertQueryShape( "/analysis/connections/SugarCRMTest",
			  SAMPLE_SUGAR_CRM_MDX_QUERY, 
			  "Total Sale Amount", 2, 5);
    }

    protected void assertQueryShape( String connResourceName,
				     String mdxQuery,
				     String expectedCaption,
				     int expectedNumAxes,
				     int expectedNumPositions) {
        mondrian.olap.Connection monConnection
            = ((OlapConnectionServiceImpl)getConnectionService())
	    .getMondrianConnection( null, connResourceName );

        // perform olap test query with this mondrian Connection
        mondrian.olap.Query monQuery = monConnection.parseQuery(mdxQuery);
        mondrian.olap.Result monResult = monConnection.execute(monQuery);
        mondrian.olap.Axis[] axes = monResult.getAxes();
	mondrian.olap.Position pos = (Position)axes[0].getPositions().get(0);
        mondrian.olap.Member m0 = (Member)pos.get(0);
        String caption = m0.getCaption();

	log.debug("caption = " + caption + ", " +
		  "axes.length=" + axes.length + ", " +
		  "positions.length= " + axes[0].getPositions().size());

        assertEquals( caption, expectedCaption );
        assertEquals( axes.length, expectedNumAxes );
        assertEquals( axes[0].getPositions().size(), expectedNumPositions );
    }

    // move this to configuration file if people want to use
    // this for testing on a regular basis...
    private final boolean DO_PSEUDO_JAPANESE_FOODMART = false;

    public void testOlapConnection() {
        // create test metadata in the repository
    	createFoodmartSchemaResource();
    	createSugarCRMSchemaResource();
    	createSugarCRMSchemaUpperResource();
        createFoodmartDataSourceResource();
        createFoodmartDataSourceResourceJNDI();
        //createFoodmartDerbyDataSourceResourceJNDI();	// comment out foodmart derby creation
        createSugarCRMDataSourceResource();
        createSugarCRMDataSourceResourceJNDI();
        createFoodmartMondrianConnectionResource();
        //createFoodmartDerbyMondrianConnectionResource();
        createFoodmartXMLAConnectionResource();
        createSugarCRMMondrianConnectionResource();
        createSugarCRMTestMondrianConnectionResource();
        createSugarCRMXMLAConnectionResource();
        createFoodmartOlapUnit();
        //createFoodmartDerbyOlapUnit();
        createSugarCRMOlapUnit();

	    createSugarCRMXmlaOlapUnit();
	    createFoodmartXmlaOlapUnit();
	    createFoodmartMondrianXMLADefinitionResource();
	    createSugarCRMMondrianXMLADefinitionResource();

	if (DO_PSEUDO_JAPANESE_FOODMART) {
	    createFoodmartJaSchemaResource();
	    createFoodmartJaDataSourceResource();
	    createFoodmartJaMondrianConnectionResource();
	    createFoodmartJaOlapUnit();
	}

	Folder analysisReports = createAnalysisReportsFolder();
	createMondrianFoodmartReport(analysisReports);
	createXmlaFoodmartReport(analysisReports);
	
	// test hitting mondrian directly
        //executeSampleFoodmartQuery();
	//executeSampleSugarQuery();
	executeSampleSugarTestQuery();

	// make some jpivot compatible olap models
	//OlapModel sampleFoodmartModel1 = getModel( "Foodmart_sample" );
	//OlapModel sampleSugarModel1 = getModel( "SugarCRM_sample" );
	//OlapModel sampleSugarModel2 = getModel( "SugarCRM_sample_unit_2" );
	OlapModel sampleSugarXmlaModel2 = getModel( "SugarCRM_xmla_sample" );

	// call getResult on the models and assert they are good
	//assertOlapModelResult( sampleSugarModel2, 5 );

	// server must be running for xmla model to work
	// should move Xmla to remote-tests
	//assertOlapModelResult( sampleSugarXmlaModel1, 5 );

	//assertOlapModelResult( sampleSugarXmlaModel2, 5 );
    }

	private void assertOlapModelResult( OlapModel m, int expectedNumCells ) {
	try {
	    m.initialize();
	    Result r = m.getResult();
	    log.info( "cell count = " + r.getCells().size() );
	    assertNotNull( r );
	    assertTrue( "sample query returned " + r.getCells().size() + 
			" cells, when expecting " + expectedNumCells,
			r.getCells().size() == expectedNumCells );
	} catch (OlapException e) {
	    log.error(e);
	    throw new RuntimeException(e);
	}
    }

    /*
    public void testOlapUnitFind() {
	ResourceLookup[] results 
	    = getRepository().findResource(null, FilterCriteria.createFilter(OlapUnit.class));
	assertTrue("got results: " + results.length + " when expecting 6", 
		   results.length == (DO_PSEUDO_JAPANESE_FOODMART ? 1 : 0) + 6);
    }
    */

    private OlapModel getModel( String viewName ) {
	RepositoryService rep = getRepository();
	OlapUnit unit = (OlapUnit)rep.getResource( null, "/analysis/views/" + viewName );
	return getConnectionService().createOlapModel( null, unit );
    }

    /**
     * property: repository
     */
    private RepositoryService mRepository;
    public RepositoryService getRepository() {
        return mRepository;
    }
    public void setRepository(RepositoryService repository) {
        mRepository = repository;
    }

    /**
     * property: olapConnectionService
     */
    private OlapConnectionService mConnectionService;
    public OlapConnectionService getConnectionService() {
        return mConnectionService;
    }
    public void setConnectionService( OlapConnectionService cs ) {
        mConnectionService = cs;
    }

    
    private final static String FOODMART_REPORT_JRXML = "/reports/jasper/MondrianFoodMartSalesReport2006.jrxml";
    
    protected void createXmlaFoodmartReport(Folder parent) {
	createOlapFoodmartReport(parent,
				 "/analysis/reports/",
				 "FoodmartSalesXmlaReport",
				 "Foodmart XML/A Sales Report",
				 "Report with XML/A Datasource",
				 "/analysis/connections/FoodmartXmlaConnection");
    }

    protected void createMondrianFoodmartReport(Folder parent) {
	createOlapFoodmartReport(parent,
				 "/analysis/reports/",
				 "FoodmartSalesMondrianReport",
				 "Foodmart Mondrian Sales Report",
				 "Report with Mondrian Datasource",
				 "/analysis/connections/Foodmart");
    }

    protected Folder createAnalysisReportsFolder() {
    	Folder reportsFolder = new FolderImpl();
    	reportsFolder.setParentFolder("/analysis");
    	reportsFolder.setName("reports");
    	reportsFolder.setLabel("Analysis Reports");
    	mRepository.saveFolder(executionContext, reportsFolder);
	return reportsFolder;
    }

    protected void createOlapFoodmartReport(Folder parent,
					    String path,
					    String name,
					    String label,
					    String desc,
					    String connectionUri) {
	ReportUnit ru = (ReportUnit) mRepository.newResource(executionContext, ReportUnit.class);
	ru.setParentFolder(parent);
	ru.setName(name);
	ru.setLabel(label);
	ru.setDescription(desc);
	
	FileResource report = (FileResource) mRepository.newResource(executionContext, FileResource.class);
	report.setFileType(FileResource.TYPE_JRXML);
	report.setName("FoodmartSalesReportJRXML");
	report.setLabel("FoodmartSalesReportJRXML");
	report.readData(getClass().getResourceAsStream(FOODMART_REPORT_JRXML));
	
	ru.setMainReport(report);
	
	ru.setDataSourceReference(connectionUri);
	
	mRepository.saveResource(executionContext, ru);
	
	Resource savedResource = mRepository.getResource(executionContext, path + name);
	assertNotNull(savedResource);
	assertTrue(savedResource instanceof ReportUnit);
	ReportUnit savedRU = (ReportUnit) savedResource;
	ResourceReference savedDSRef = savedRU.getDataSource();
	assertNotNull(savedDSRef);
	assertFalse(savedDSRef.isLocal());
	assertEquals(connectionUri, savedDSRef.getReferenceURI());
    }
    
}
