/*
 * Copyright (C) 2005 - 2007 JasperSoft Corporation.  All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased a commercial license agreement from JasperSoft,
 * the following license terms apply:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published by
 * the Free Software Foundation.
 *
 * This program is distributed WITHOUT ANY WARRANTY; and without the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see http://www.gnu.org/licenses/gpl.txt
 * or write to:
 *
 * Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330,
 * Boston, MA  USA  02111-1307
 */
package com.jaspersoft.jasperserver.api.metadata.view.service.impl;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import java.util.Set;

import junit.framework.TestCase;

import org.springframework.context.support.ClassPathXmlApplicationContext;

import com.jaspersoft.jasperserver.api.metadata.common.domain.DataType;
import com.jaspersoft.jasperserver.api.metadata.common.domain.FileResource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Folder;
import com.jaspersoft.jasperserver.api.metadata.common.domain.InputControl;
import com.jaspersoft.jasperserver.api.metadata.common.domain.ListOfValues;
import com.jaspersoft.jasperserver.api.metadata.common.domain.ListOfValuesItem;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Resource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.ResourceLookup;
import com.jaspersoft.jasperserver.api.metadata.common.domain.ResourceReference;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Query;
import com.jaspersoft.jasperserver.api.metadata.common.domain.client.DataTypeImpl;
import com.jaspersoft.jasperserver.api.metadata.common.domain.client.FolderImpl;
import com.jaspersoft.jasperserver.api.metadata.common.domain.client.ListOfValuesItemImpl;
import com.jaspersoft.jasperserver.api.metadata.common.service.RepositoryService;
import com.jaspersoft.jasperserver.api.metadata.jasperreports.domain.BeanReportDataSource;
import com.jaspersoft.jasperserver.api.metadata.jasperreports.domain.JdbcReportDataSource;
import com.jaspersoft.jasperserver.api.metadata.jasperreports.domain.JndiJdbcReportDataSource;
import com.jaspersoft.jasperserver.api.metadata.jasperreports.domain.ReportDataSource;
import com.jaspersoft.jasperserver.api.metadata.jasperreports.domain.ReportUnit;
import com.jaspersoft.jasperserver.api.metadata.view.domain.FilterCriteria;
import com.jaspersoft.jasperserver.api.metadata.view.domain.FilterElementDisjunction;
import com.jaspersoft.jasperserver.api.metadata.view.domain.FilterElementOr;
import com.jaspersoft.jasperserver.util.QueryBean;

/**
 * @author Lucian Chirita (lucianc@users.sourceforge.net)
 * @version $Id: HibernateRepositoryTest.java 10134 2007-09-18 23:39:51Z tony $
 */
public class HibernateRepositoryTest extends TestCase {

    private Properties jdbcProps;
    private RepositoryService repo;
    private ClassPathXmlApplicationContext appContext;

    public HibernateRepositoryTest(String name) {
        super(name);
    }

    protected void setUp() throws Exception {
        loadJdbcProps();
        appContext = new ClassPathXmlApplicationContext(
                new String[] {"hibernateConfig.xml", "viewService.xml"});

        repo = (RepositoryService) appContext.getBean("repoService");
    }

    protected void tearDown() {
        appContext.close();
    }

    protected Properties loadJdbcProps() throws IOException, FileNotFoundException {
        jdbcProps = new Properties();
        String jdbcPropFile = System.getProperty("test.hibernate.jdbc.properties");
        BufferedInputStream is = new BufferedInputStream(new FileInputStream(jdbcPropFile));
        jdbcProps.load(is);
        is.close();
        return jdbcProps;
    }

    // TODO disabled for now
    public void disabled_testDeleteFolder() {
        assertNotNull("Repo not null", repo);

        Folder tmpFolder = new FolderImpl();
        setCommon(tmpFolder, "tmp_foo");
        repo.saveFolder(null, tmpFolder);

        Folder subtmpFolder = new FolderImpl();
        setCommon(subtmpFolder, "sub");
        subtmpFolder.setParentFolder(tmpFolder);
        repo.saveFolder(null, subtmpFolder);

        JndiJdbcReportDataSource ds = (JndiJdbcReportDataSource) repo.newResource(null, JndiJdbcReportDataSource.class);
        setCommon(ds, "dummyDS");
        ds.setJndiName("foo");
        ds.setParentFolder(tmpFolder);
        repo.saveResource(null, ds);

        JndiJdbcReportDataSource subds = (JndiJdbcReportDataSource) repo.newResource(null, JndiJdbcReportDataSource.class);
        setCommon(subds, "subDS");
        subds.setJndiName("subfoo");
        subds.setParentFolder(subtmpFolder);
        repo.saveResource(null, subds);

        Resource dsResource = repo.getResource(null, "/tmp_foo/dummyDS");
        assertNotNull(dsResource);
        assertTrue(dsResource instanceof JndiJdbcReportDataSource);
        ds = (JndiJdbcReportDataSource) dsResource;
        assertEquals("foo", ds.getJndiName());

        Resource subdsResource = repo.getResource(null, "/tmp_foo/sub/subDS");
        assertNotNull(subdsResource);
        assertTrue(subdsResource instanceof JndiJdbcReportDataSource);
        subds = (JndiJdbcReportDataSource) subdsResource;
        assertEquals("subfoo", subds.getJndiName());

        repo.deleteFolder(null, "/tmp_foo");

        List folders = repo.getSubFolders(null, "/");
        for (Iterator it = folders.iterator(); it.hasNext();) {
            Folder folder = (Folder) it.next();
            assertFalse("tmp_foo".equals(folder.getName()));
        }

        dsResource = repo.getResource(null, "/tmp_foo/dummyDS");
        assertNull(dsResource);

        subdsResource = repo.getResource(null, "/tmp_foo/dummyDS");
        assertNull(subdsResource);
    }

    // TODO disabled for now
    public void disabled_testDeleteRollback() {
        assertNotNull("Repo not null", repo);

        Folder tmp1 = new FolderImpl();
        setCommon(tmp1, "tmp1");
        repo.saveFolder(null, tmp1);

        Folder tmp2 = new FolderImpl();
        setCommon(tmp2, "tmp2");
        repo.saveFolder(null, tmp2);

        tmp1 = repo.getFolder(null, "/tmp1");
        assertNotNull(tmp1);
        assertEquals("tmp1", tmp1.getName());

        tmp2 = repo.getFolder(null, "/tmp2");
        assertNotNull(tmp2);
        assertEquals("tmp2", tmp2.getName());

        boolean exception = false;
        try {
            repo.delete(null, null, new String[] { "/tmp1", "/tmp_zzz" });
        } catch (Exception e) {
            exception = true;
        }
        assertTrue(exception);


        tmp1 = repo.getFolder(null, "/tmp1");
        assertNotNull(tmp1);
        assertEquals("tmp1", tmp1.getName());

        repo.delete(null, null, new String[] { "/tmp1", "/tmp2" });

        tmp1 = repo.getFolder(null, "/tmp1");
        assertNull(tmp1);

        tmp2 = repo.getFolder(null, "/tmp2");
        assertNull(tmp2);
    }

    public void testFolderUpdate() {
        assertNotNull("Repo not null", repo);

        Folder folder = new FolderImpl();
        setCommon(folder, "test_update");
        repo.saveFolder(null, folder);

        JndiJdbcReportDataSource ds = (JndiJdbcReportDataSource) repo.newResource(null, JndiJdbcReportDataSource.class);
        setCommon(ds, "fooDS");
        ds.setJndiName("foo");
        ds.setParentFolder(folder);
        repo.saveResource(null, ds);

        folder = repo.getFolder(null, "/test_update");
        assertNotNull(folder);
        assertEquals("test_update", folder.getName());
        assertEquals("test_update_label", folder.getLabel());

        folder.setLabel("updated");
        repo.saveFolder(null, folder);

        folder = repo.getFolder(null, "/test_update");
        assertNotNull(folder);
        assertEquals("test_update", folder.getName());
        assertEquals("updated", folder.getLabel());

        Resource resource = repo.getResource(null, "/test_update/fooDS");
        assertNotNull(resource);
        assertTrue(resource instanceof JndiJdbcReportDataSource);
        ds = (JndiJdbcReportDataSource) resource;
        assertEquals("fooDS", ds.getName());
        assertEquals("foo", ds.getJndiName());

        repo.deleteFolder(null, "/test_update");
        folder = repo.getFolder(null, "/test_update");
        assertNull(folder);
    }

    public void testRepo() {
        assertNotNull("Repo not null", repo);

        write();

        read();

        readFolders();

        update();

        optimisticLocking();

        list();

        resources();

        // TODO disabled to allow build to run
        //deleteRollback();
    }

    private void deleteRollback() {
        boolean exception = false;
        try {
            repo.delete(null, new String[]{"/datasources/JServerJdbcDS"}, null);
        } catch (Exception e) {
            exception = true;
        }
        assertTrue(exception);

        Resource resource = repo.getResource(null, "/datasources/JServerJdbcDS");
        assertNotNull(resource);
        assertTrue(resource instanceof JdbcReportDataSource);
    }

    private void write() {
        Folder dsFolder = new FolderImpl();
        dsFolder.setName("datasources");
        dsFolder.setLabel("Data Sources");
        dsFolder.setDescription("Data Sources used by reports");
        repo.saveFolder(null, dsFolder);

        createJndiDS();
        createJdbcDS();
        createBeanDS();
        createTableModelDS();

        Folder reportsFolder = new FolderImpl();
        reportsFolder.setName("reports");
        reportsFolder.setLabel("Reports");
        reportsFolder.setDescription("Reports");
        repo.saveFolder(null, reportsFolder);

        Folder samplesFolder = new FolderImpl();
        samplesFolder.setName("samples");
        samplesFolder.setLabel("Samples");
        samplesFolder.setDescription("Samples");
        samplesFolder.setParentFolder(reportsFolder);
        repo.saveFolder(null, samplesFolder);

        createAllAccounts(samplesFolder);
        createSalesByMonth(samplesFolder);
        createCustomDSReportFileResource(samplesFolder);
        createCustomDSReport(samplesFolder);
        createTableModelDSReport(samplesFolder);
        createEmployeeAccounts(samplesFolder);
        createEmployees(samplesFolder);
        createParamMany(samplesFolder);

        createImage();

        Folder olapFolder = new FolderImpl();
        olapFolder.setName("analysis");
        olapFolder.setLabel("Analysis Components");
        olapFolder.setDescription("Analysis Components");
        repo.saveFolder(null, olapFolder);

        Folder connectionsFolder = new FolderImpl();
        connectionsFolder.setName("connections");
        connectionsFolder.setLabel("Analysis Connections");
        connectionsFolder.setDescription("Connections used by Analysis");
        connectionsFolder.setParentFolder(olapFolder);
        repo.saveFolder(null, connectionsFolder);

        Folder schemasFolder = new FolderImpl();
        schemasFolder.setName("schemas");
        schemasFolder.setLabel("Analysis Schemas");
        schemasFolder.setDescription("Schemas used by Analysis");
        schemasFolder.setParentFolder(olapFolder);
        repo.saveFolder(null, schemasFolder);

        Folder olapDsFolder = new FolderImpl();
        olapDsFolder.setName("datasources");
        olapDsFolder.setLabel("Analysis Data Sources");
        olapDsFolder.setDescription("Data sources used by Analysis");
        olapDsFolder.setParentFolder(olapFolder);
        repo.saveFolder(null, olapDsFolder);

        Folder olapViewsFolder = new FolderImpl();
        olapViewsFolder.setName("views");
        olapViewsFolder.setLabel("Analysis Views");
        olapViewsFolder.setDescription("Analysis Views");
        olapViewsFolder.setParentFolder(olapFolder);
        repo.saveFolder(null, olapViewsFolder);
    }

    private void createImage() {
        Folder folder = new FolderImpl();
        folder.setName("images");
        folder.setLabel("Images");
        folder.setDescription("Folder containing reusable images");
        repo.saveFolder(null, folder);

        FileResource image = (FileResource) repo.newResource(null, FileResource.class);
        image.setFileType(FileResource.TYPE_IMAGE);
        image.readData(getClass().getResourceAsStream("/images/jasperreports.gif"));
        image.setName("JRLogo");
        image.setLabel("JR logo");
        image.setDescription("JR logo");
        image.setParentFolder(folder);

        repo.saveResource(null, image);
    }

    private void createJndiDS() {
        JndiJdbcReportDataSource datasource = (JndiJdbcReportDataSource) repo.newResource(null, JndiJdbcReportDataSource.class);
        datasource.setName("JServerJNDIDS");
        datasource.setLabel("JServer JNDI data source");
        datasource.setDescription("JServer JNDI data source");
        datasource.setJndiName(jdbcProps.getProperty("test.jndi"));
        datasource.setParentFolder("/datasources");

        repo.saveResource(null, datasource);
    }

    private void createJdbcDS() {
        JdbcReportDataSource datasource = (JdbcReportDataSource) repo.newResource(null, JdbcReportDataSource.class);
        datasource.setName("JServerJdbcDS");
        datasource.setLabel("JServer Jdbc data source");
        datasource.setDescription("JServer Jdbc data source");
        datasource.setParentFolder("/datasources");

        datasource.setDriverClass(jdbcProps.getProperty("test.jdbc.driverClassName"));
        datasource.setConnectionUrl(jdbcProps.getProperty("test.jdbc.url"));
        datasource.setUsername(jdbcProps.getProperty("test.jdbc.username"));
        datasource.setPassword(jdbcProps.getProperty("test.jdbc.password"));

        repo.saveResource(null, datasource);
    }

    private void createBeanDS() {
        BeanReportDataSource datasource = (BeanReportDataSource) repo.newResource(null, BeanReportDataSource.class);
        datasource.setName("CustomDSFromBean");
        datasource.setLabel("Custom data source from a bean");
        datasource.setDescription("A custom data source through a bean");
        datasource.setParentFolder("/datasources");

        datasource.setBeanName("customDataSourceService");

        repo.saveResource(null, datasource);
    }

    private void createTableModelDS() {
        BeanReportDataSource datasource = (BeanReportDataSource) repo.newResource(null, BeanReportDataSource.class);
        datasource.setName("CustomTableModelDS");
        datasource.setLabel("Custom data source from a table model");
        datasource.setDescription("A custom data source through a table model");
        datasource.setParentFolder("/datasources");

        datasource.setBeanName("customDataSourceServiceFactory");
        datasource.setBeanMethod("tableModelDataSource");

        repo.saveResource(null, datasource);
    }

    private void createSalesByMonth(Folder folder) {
        FileResource reportRes = (FileResource) repo.newResource(null, FileResource.class);
        reportRes.setFileType(FileResource.TYPE_JRXML);
        reportRes.setName("SalesByMonthReport");
        reportRes.setLabel("Sales By Month Jasper Report");
        reportRes.setDescription("Sales By Month Jasper Report");

        InputStream jrxml = getClass().getResourceAsStream("/reports/jasper/SalesByMonth.jrxml");
        reportRes.readData(jrxml);

        ReportUnit unit = (ReportUnit) repo.newResource(null, ReportUnit.class);
        unit.setName("SalesByMonth");
        unit.setLabel("Sales By Month Report");
        unit.setDescription("Sales By Month Report");
        unit.setParentFolder(folder);

        unit.setDataSourceReference("/datasources/JServerJNDIDS");
        unit.setMainReport(reportRes);

        FileResource jar = (FileResource) repo.newResource(null, FileResource.class);
        jar.setFileType(FileResource.TYPE_JAR);
        jar.readData(getClass().getResourceAsStream("/jars/scriptlet.jar"));
        setCommon(jar, "Scriptlet");
        unit.addResource(jar);

        FileResource img = (FileResource) repo.newResource(null, FileResource.class);
        img.setFileType(FileResource.TYPE_IMAGE);
        img.readData(getClass().getResourceAsStream("/images/jasperreports.gif"));
        setCommon(img, "Logo");
        unit.addResource(img);

        FileResource subrep = (FileResource) repo.newResource(null, FileResource.class);
        subrep.setFileType(FileResource.TYPE_JRXML);
        subrep.readData(getClass().getResourceAsStream("/reports/jasper/SalesByMonthDetail.jrxml"));
        setCommon(subrep, "SalesByMonthDetail");
        unit.addResource(subrep);

        FileResource resBdl = (FileResource) repo.newResource(null, FileResource.class);
        resBdl.setFileType(FileResource.TYPE_RESOURCE_BUNDLE);
        resBdl.readData(getClass().getResourceAsStream("/resource_bundles/sales.properties"));
        setCommon(resBdl, "sales.properties");
        unit.addResource(resBdl);

        FileResource resBdl_ro = (FileResource) repo.newResource(null, FileResource.class);
        resBdl_ro.setFileType(FileResource.TYPE_RESOURCE_BUNDLE);
        resBdl_ro.readData(getClass().getResourceAsStream("/resource_bundles/sales_ro.properties"));
        setCommon(resBdl_ro, "sales_ro.properties");
        unit.addResource(resBdl_ro);

        InputControl textInputCtrl = (InputControl) repo.newResource(null, InputControl.class);
        setCommon(textInputCtrl, "TextInputControl");
        textInputCtrl.setType(InputControl.TYPE_SINGLE_VALUE);
        textInputCtrl.setMandatory(false);
        textInputCtrl.setReadOnly(false);
        textInputCtrl.setVisible(true);
        //FIXME textInputCtrl.setSize(new Integer(30));
        textInputCtrl.setLabel("Text Input Control");
        textInputCtrl.setName("TextInput");
        DataType dataType = new DataTypeImpl();
        dataType.setName("test");
        dataType.setLabel("test");
        dataType.setType(DataType.TYPE_NUMBER);
        textInputCtrl.setDataType(dataType);
        unit.addInputControl(textInputCtrl);

        InputControl checkboxInputControl = (InputControl) repo.newResource(null, InputControl.class);
        setCommon(checkboxInputControl, "CheckboxInputControl");
        checkboxInputControl.setType(InputControl.TYPE_BOOLEAN);
        checkboxInputControl.setMandatory(true);
        checkboxInputControl.setReadOnly(false);
        checkboxInputControl.setVisible(true);
        checkboxInputControl.setLabel("Checkbox Input Control");
        checkboxInputControl.setName("CheckboxInput");
        unit.addInputControl(checkboxInputControl);

        InputControl listInputControl = (InputControl) repo.newResource(null, InputControl.class);
        setCommon(listInputControl, "ListInputControl");
        listInputControl.setType(InputControl.TYPE_SINGLE_SELECT_LIST_OF_VALUES);
        listInputControl.setMandatory(true);
        listInputControl.setReadOnly(false);
        listInputControl.setVisible(true);
        listInputControl.setLabel("List Input Control");
        listInputControl.setName("ListInput");

        ListOfValues values = (ListOfValues) repo.newResource(null, ListOfValues.class);
        values.setName("List_of_values");
        values.setLabel("List of values label");
        values.setDescription("List of values description");
        ListOfValuesItem item = new ListOfValuesItemImpl();
        item.setLabel("An item");
        item.setValue("1");
        values.addValue(item);
        item = new ListOfValuesItemImpl();
        item.setLabel("Another item");
        item.setValue("2");
        values.addValue(item);
        item = new ListOfValuesItemImpl();
        item.setLabel("Yet another item");
        item.setValue("3");
        values.addValue(item);
        listInputControl.setListOfValues(values);

        dataType = new DataTypeImpl();
        dataType.setName("test");
        dataType.setLabel("test");
        dataType.setType(DataType.TYPE_TEXT);
        listInputControl.setDataType(dataType);

        unit.addInputControl(listInputControl);

        createDateDatatype();

        InputControl dateInputCtrl = (InputControl) repo.newResource(null, InputControl.class);
        setCommon(dateInputCtrl, "DateInput");
        dateInputCtrl.setType(InputControl.TYPE_SINGLE_VALUE);
        dateInputCtrl.setMandatory(false);
        dateInputCtrl.setReadOnly(false);
        dateInputCtrl.setVisible(true);
        dateInputCtrl.setDataTypeReference("/datatypes/date");
        unit.addInputControl(dateInputCtrl);

        InputControl queryInputCtrl = (InputControl) repo.newResource(null, InputControl.class);
        setCommon(queryInputCtrl, "QueryInput");
        queryInputCtrl.setType(InputControl.TYPE_SINGLE_SELECT_QUERY);
        queryInputCtrl.setMandatory(false);
        queryInputCtrl.setReadOnly(false);
        queryInputCtrl.setVisible(true);
        queryInputCtrl.setQueryValueColumn("user_name");
        queryInputCtrl.addQueryVisibleColumn("first_name");
        queryInputCtrl.addQueryVisibleColumn("last_name");
        Query query = (Query) repo.newResource(null, Query.class);
        setCommon(query, "testQuery");
        query.setLanguage("sql");
        query.setSql("select user_name, first_name, last_name from users");
        queryInputCtrl.setQuery(query);
        unit.addInputControl(queryInputCtrl);

        repo.saveResource(null, unit);
    }

    private void createEmployees(Folder folder) {
        FileResource reportRes = (FileResource) repo.newResource(null, FileResource.class);
        reportRes.setFileType(FileResource.TYPE_JRXML);
        setCommon(reportRes, "EmployeesJRXML");

        InputStream jrxml = getClass().getResourceAsStream("/reports/jasper/Employees.jrxml");
        reportRes.readData(jrxml);

        ReportUnit unit = (ReportUnit) repo.newResource(null, ReportUnit.class);
        unit.setName("Employees");
        unit.setLabel("Employee List");
        unit.setDescription("Employee List");
        unit.setParentFolder(folder);

        unit.setDataSourceReference("/datasources/JServerJNDIDS");
        unit.setMainReport(reportRes);

        repo.saveResource(null, unit);
    }

    private void createEmployeeAccounts(Folder folder) {
        FileResource reportRes = (FileResource) repo.newResource(null, FileResource.class);
        reportRes.setFileType(FileResource.TYPE_JRXML);
        setCommon(reportRes, "EmployeeAccountsJRXML");

        InputStream jrxml = getClass().getResourceAsStream("/reports/jasper/EmployeeAccounts.jrxml");
        reportRes.readData(jrxml);

        ReportUnit unit = (ReportUnit) repo.newResource(null, ReportUnit.class);
        unit.setName("EmployeeAccounts");
        unit.setLabel("Employee Accounts");
        unit.setDescription("List of Accounts per Employee");
        unit.setParentFolder(folder);

        unit.setDataSourceReference("/datasources/JServerJNDIDS");
        unit.setMainReport(reportRes);

        InputControl empIC = (InputControl) repo.newResource(null, InputControl.class);
        empIC.setName("EmployeeID");
        empIC.setLabel("Employee");
        empIC.setMandatory(true);

        Query empQuery = (Query) repo.newResource(null, Query.class);
        empQuery.setName("EmployeeQuery");
        empQuery.setLabel("Employee Query");
        empQuery.setLanguage("sql");
        empQuery.setSql("SELECT id, user_name FROM users WHERE employee_status = 'Active'");

        empIC.setType(InputControl.TYPE_SINGLE_SELECT_QUERY);
        empIC.setQuery(empQuery);
        empIC.setQueryValueColumn("id");
        empIC.addQueryVisibleColumn("user_name");

        unit.addInputControl(empIC);

        repo.saveResource(null, unit);
    }

    private void createParamMany(Folder folder) {
        FileResource reportRes = (FileResource) repo.newResource(null, FileResource.class);
        reportRes.setFileType(FileResource.TYPE_JRXML);
        setCommon(reportRes, "ParametersJRXML");

        InputStream jrxml = getClass().getResourceAsStream("/reports/jasper/ParamMany.jrxml");
        reportRes.readData(jrxml);

        ReportUnit unit = (ReportUnit) repo.newResource(null, ReportUnit.class);
        unit.setName("Freight");
        unit.setLabel("Freight Report");
        unit.setDescription("Freight Report with Saved Parameters");
        unit.setParentFolder(folder);

        unit.setDataSourceReference("/datasources/JServerJNDIDS");
        unit.setMainReport(reportRes);

        // Country
        InputControl countryIC = (InputControl) repo.newResource(null, InputControl.class);
        countryIC.setName("Country");
        countryIC.setLabel("Country");
        countryIC.setDescription("Country");
        countryIC.setMandatory(true);
        countryIC.setType(InputControl.TYPE_SINGLE_SELECT_QUERY);

        Query countryQ = (Query) repo.newResource(null, Query.class);
        countryQ.setName("CountryQuery");
        countryQ.setLabel("CountryQuery");
        countryQ.setLanguage(QueryBean.DEFAULT_LANGUAGE);
        countryQ.setDataSourceReference("/datasources/JServerJNDIDS");
        countryQ.setSql("select distinct SHIPCOUNTRY from ORDERS");
        countryIC.setQuery(countryQ);

        countryIC.setQueryValueColumn("SHIPCOUNTRY");
        countryIC.addQueryVisibleColumn("SHIPCOUNTRY");

        unit.addInputControl(countryIC);

        // Request Date
        InputControl requestDateIC = (InputControl) repo.newResource(null, InputControl.class);
        requestDateIC.setName("RequestDate");
        requestDateIC.setLabel("RequestDate");
        requestDateIC.setDescription("RequestDate");
        requestDateIC.setMandatory(false);
        requestDateIC.setType(InputControl.TYPE_SINGLE_VALUE);

        DataType dateDT = (DataType) repo.newResource(null, DataType.class);
        setCommon(dateDT, "Date");
        dateDT.setType(DataType.TYPE_DATE);
        requestDateIC.setDataType(dateDT);

        unit.addInputControl(requestDateIC);

        // Order Id
        InputControl orderIdIC = (InputControl) repo.newResource(null, InputControl.class);
        orderIdIC.setName("OrderId");
        orderIdIC.setLabel("OrderId");
        orderIdIC.setDescription("OrderId");
        orderIdIC.setMandatory(false);
        orderIdIC.setType(InputControl.TYPE_SINGLE_VALUE);

        DataType numberDT = (DataType) repo.newResource(null, DataType.class);
        setCommon(numberDT, "Number");
        numberDT.setType(DataType.TYPE_NUMBER);
        orderIdIC.setDataType(numberDT);

        unit.addInputControl(orderIdIC);

        repo.saveResource(null, unit);
    }

    protected void createDateDatatype() {
        Folder folder = new FolderImpl();
        folder.setName("datatypes");
        folder.setLabel("Input data types");
        repo.saveFolder(null, folder);

        DataType dateDataType = new DataTypeImpl();
        setCommon(dateDataType, "date");
        dateDataType.setType(DataType.TYPE_DATE);
        dateDataType.setParentFolder(folder);
        repo.saveResource(null, dateDataType);
    }

    private void setCommon(Resource res, String id) {
        res.setName(id);
        res.setLabel(id + "_label");
        res.setDescription(id + " description");
    }

    private void createAllAccounts(Folder folder) {
        FileResource reportRes = (FileResource) repo.newResource(null, FileResource.class);
        reportRes.setFileType(FileResource.TYPE_JRXML);
        reportRes.setName("AllAccountsReport");
        reportRes.setLabel("All Accounts Jasper Report");
        reportRes.setDescription("All Accounts Jasper Report");
        reportRes.setParentFolder(folder);

        InputStream jrxml = getClass().getResourceAsStream(
                "/reports/jasper/AllAccounts.jrxml");
        reportRes.readData(jrxml);

        ReportUnit unit = (ReportUnit) repo.newResource(null, ReportUnit.class);
        unit.setName("AllAccounts");
        unit.setLabel("All Accounts Report");
        unit.setDescription("All Accounts Report");
        unit.setParentFolder(folder);

        unit.setDataSourceReference("/datasources/JServerJNDIDS");
        unit.setMainReport(reportRes);

        FileResource res1 = (FileResource) repo.newResource(null, FileResource.class);
        res1.setFileType(FileResource.TYPE_IMAGE);
        res1.readData(getClass().getResourceAsStream("/images/jasperreports.gif"));
        res1.setName("AllAccounts_Res1");
        res1.setLabel("AllAccounts_Res1");
        res1.setDescription("AllAccounts_Res1");
        unit.addResource(res1);

        FileResource res2 = (FileResource) repo.newResource(null, FileResource.class);
        res2.setFileType(FileResource.TYPE_IMAGE);
        res2.readData(getClass().getResourceAsStream("/images/logo.jpg"));
        res2.setName("AllAccounts_Res2");
        res2.setLabel("AllAccounts_Res2");
        res2.setDescription("AllAccounts_Res2");
        unit.addResource(res2);

        repo.saveResource(null, unit);
    }


    private void createCustomDSReportFileResource(Folder folder) {
        FileResource reportRes = (FileResource) repo.newResource(null, FileResource.class);
        reportRes.setFileType(FileResource.TYPE_JRXML);
        reportRes.setName("DataSourceReportTemplate");
        reportRes.setLabel("Report showing Custom Data Source");
        reportRes.setDescription("Report showing use of Custom Data Source via a bean");
        reportRes.setParentFolder(folder);

        InputStream jrxml = getClass().getResourceAsStream(
                "/reports/jasper/DataSourceReport.jrxml");
        reportRes.readData(jrxml);

        repo.saveResource(null, reportRes);
    }

    private void createCustomDSReport(Folder folder) {

        ReportUnit unit = (ReportUnit) repo.newResource(null, ReportUnit.class);
        unit.setName("DataSourceReport");
        unit.setLabel("Report showing Custom Data Source");
        unit.setDescription("Report showing use of Custom Data Source via a bean");
        unit.setParentFolder(folder);

        unit.setMainReportReference("/reports/samples/DataSourceReportTemplate");
        unit.setDataSourceReference("/datasources/CustomDSFromBean");

        repo.saveResource(null, unit);
    }

    private void createTableModelDSReport(Folder folder) {

        ReportUnit unit = (ReportUnit) repo.newResource(null, ReportUnit.class);
        unit.setName("DataSourceTableModel");
        unit.setLabel("Table Model Data Source");
        unit.setDescription("Report showing use of Custom Data Source via table model");
        unit.setParentFolder(folder);

        unit.setMainReportReference("/reports/samples/DataSourceReportTemplate");
        unit.setDataSourceReference("/datasources/CustomTableModelDS");
        unit.setMainReportReference("/reports/samples/DataSourceReportTemplate");

        repo.saveResource(null, unit);
    }

    private void update() {
        ReportUnit unit = (ReportUnit) repo.getResource(null, "/reports/samples/AllAccounts");
        assertEquals("AllAccounts", unit.getName());
        assertEquals("/reports/samples/AllAccounts", unit.getURIString());

        unit.setLabel("Accounts Report");

        FileResource removed = unit.removeResourceLocal("AllAccounts_Res1");
        assertNotNull(removed);

        FileResource res3 = (FileResource) repo.newResource(null, FileResource.class);
        res3.setFileType(FileResource.TYPE_IMAGE);
        res3.readData(getClass().getResourceAsStream("/images/jasperreports.gif"));
        res3.setName("AllAccounts_Res3");
        res3.setLabel("AllAccounts_Res3");
        res3.setDescription("AllAccounts_Res3");
        unit.addResource(res3);

        FileResource res4 = (FileResource) repo.newResource(null, FileResource.class);
        res4.setFileType(FileResource.TYPE_IMAGE);
        res4.setReferenceURI("/images/JRLogo");
        setCommon(res4, "LogoLink");
        unit.addResource(res4);

        repo.saveResource(null, unit);

        unit = (ReportUnit) repo.getResource(null, "/reports/samples/AllAccounts");
        assertTrue(unit.getLabel().startsWith("Accounts "));

        List resources = unit.getResources();
        assertNotNull(resources);
        assertEquals(3, resources.size());
    }

    private void read() {
        ReportUnit unit = (ReportUnit) repo.getResource(null, "/reports/samples/AllAccounts");
        assertNotNull(unit);
        assertEquals(ReportUnit.class.getName(), unit.getResourceType());
        assertEquals("AllAccounts", unit.getName());

        ResourceReference dsRef = unit.getDataSource();
        assertNotNull(dsRef);
        assertFalse(dsRef.isLocal());
        assertEquals("/datasources/JServerJNDIDS", dsRef.getReferenceURI());

        ResourceReference mainReportRef = unit.getMainReport();
        assertNotNull(mainReportRef);
        assertTrue(mainReportRef.isLocal());
        Resource mainReport = mainReportRef.getLocalResource();
        assertTrue(mainReport instanceof FileResource);
        assertEquals("AllAccountsReport", mainReport.getName());

        List resources = unit.getResources();
        assertNotNull(resources);
        assertEquals(2, resources.size());

        FileResource img = (FileResource) repo.getResource(null, "/images/JRLogo");
        assertNotNull(img);
        assertEquals("JRLogo", img.getName());

        datasources();
        parentFolder();
        propertyFilter();
        disjunctionFilter();
        orFilter();
    }

    private void datasources() {
        FilterCriteria criteria = FilterCriteria.createFilter(ReportDataSource.class);
        ResourceLookup[] lookups = repo.findResource(null, criteria);
        assertNotNull(lookups);
        assertTrue(2 <= lookups.length);
        for (int i = 0; i < lookups.length; i++) {
            Resource res = repo.getResource(null, lookups[i].getURIString());
            assertTrue(res instanceof ReportDataSource);
        }
    }

    private void parentFolder() {
        FilterCriteria criteria = FilterCriteria.createFilter();
        criteria.addFilterElement(FilterCriteria.createParentFolderFilter("/datasources"));
        ResourceLookup[] folderResources = repo.findResource(null, criteria);
        assertNotNull(folderResources);
        assertTrue(folderResources.length > 0);
        for (int i = 0; i < folderResources.length; i++) {
            ResourceLookup lookup = folderResources[i];
            assertEquals("/datasources", lookup.getParentFolder());
            if (lookup.getName().equals("JServerJdbcDS")) {
                assertEquals(JdbcReportDataSource.class.getName(), lookup.getResourceType());
            } else if (lookup.getName().equals("JServerJNDIDS")) {
                assertEquals(JndiJdbcReportDataSource.class.getName(), lookup.getResourceType());
            }
        }
    }

    private void propertyFilter() {
        FilterCriteria criteria = FilterCriteria.createFilter(FileResource.class);
        criteria.addFilterElement(FilterCriteria.createPropertyEqualsFilter("fileType", FileResource.TYPE_IMAGE));
        ResourceLookup[] resources = repo.findResource(null, criteria);
        assertNotNull(resources);
        assertTrue(resources.length > 0);
        for (int i = 0; i < resources.length; i++) {
            Resource res = repo.getResource(null, resources[i].getURIString());
            assertTrue(res instanceof FileResource);
            FileResource fileRes = (FileResource) res;
            assertEquals(FileResource.class.getName(), fileRes.getResourceType());
            assertEquals(FileResource.TYPE_IMAGE, fileRes.getFileType());
        }
    }

    private void disjunctionFilter() {
        FilterCriteria criteria = FilterCriteria.createFilter();
        FilterElementDisjunction disjunction = criteria.addDisjunction();
        disjunction.addFilterElement(FilterCriteria.createPropertyLikeFilter("name", "%JdbcDS"));
        disjunction.addNegatedFilterElement(FilterCriteria.createParentFolderFilter("/datasources"));

        ResourceLookup[] resources = repo.findResource(null, criteria);
        assertNotNull(resources);
        assertTrue(resources.length > 0);
        for (int i = 0; i < resources.length; i++) {
            ResourceLookup lookup = resources[i];
            assertTrue(lookup.getName().endsWith("JdbcDS") || !lookup.getParentFolder().equals("/datasources"));
        }
    }

    private void orFilter() {
        FilterCriteria criteria = FilterCriteria.createFilter();
        FilterElementOr or = criteria.addOr();
        or.setLeftHandSide(FilterCriteria.createParentFolderFilter("/images"));
        or.setRightHandSide(FilterCriteria.createParentFolderFilter("/datasources"));

        ResourceLookup[] resources = repo.findResource(null, criteria);
        assertNotNull("Null resources found for or filter", resources);
        assertTrue("No resources found for or filter", resources.length > 0);
        boolean imagesFound = false;
        boolean dataSourcesFound = false;
        for (int i = 0; i < resources.length && (!imagesFound || !dataSourcesFound); i++) {
            ResourceLookup lookup = resources[i];
            if (lookup.getParentFolder().equals("/images")) {
                imagesFound = true;
            } else if (lookup.getParentFolder().equals("/datasources")) {
                dataSourcesFound = true;
            }
        }
        assertTrue("No images found for or filter", imagesFound);
        assertTrue("No data sources found for or filter", dataSourcesFound);
    }

    private void readFolders() {
        List folders = repo.getAllFolders(null);
        assertNotNull(folders);
        assertTrue(5 <= folders.size());

        Iterator it = folders.iterator();
        Folder folder = (Folder) it.next();
        assertEquals("/", folder.getURIString());

        Set folderURIs = new HashSet();
        while (it.hasNext()) {
            folder = (Folder) it.next();
            folderURIs.add(folder.getURIString());
        }
        assertTrue(folderURIs.contains("/datasources"));
        assertTrue(folderURIs.contains("/images"));
        assertTrue(folderURIs.contains("/reports"));
        assertTrue(folderURIs.contains("/reports/samples"));

        List subFolders = repo.getSubFolders(null, "/reports");
        assertNotNull(subFolders);
        assertEquals(1, subFolders.size());
        folder = (Folder) subFolders.get(0);
        assertEquals("/reports/samples", folder.getURIString());
    }

    private void list() {
        ResourceLookup[] units = repo.findResource(null, FilterCriteria.createFilter(ReportUnit.class));
        assertNotNull(units);
        assertTrue(units.length >= 2);
    }

    private void resources() {
        Resource[] resources = repo.findResource(null, null);
        assertNotNull(resources);
        assertTrue(resources.length >= 2);
    }

    private void optimisticLocking() {
        JndiJdbcReportDataSource ds1 = (JndiJdbcReportDataSource) repo.getResource(null, "/datasources/JServerJNDIDS");
        ds1.setLabel(ds1.getLabel() + " Updated 1");

        JndiJdbcReportDataSource ds2 = (JndiJdbcReportDataSource) repo.getResource(null, "/datasources/JServerJNDIDS");
        ds2.setLabel(ds1.getLabel() + " Updated 2");

        repo.saveResource(null, ds1);

        boolean failed = false;
        try {
            repo.saveResource(null, ds2);
        } catch (Exception e) {
            failed = true;
        }
        assertTrue(failed);
    }


    public void testLocalResourceReplace() {
        assertNotNull("Repo not null", repo);

        Folder folder = new FolderImpl();
        setCommon(folder, "testLocalResourceReplace");
        folder.setParentFolder("/");
        repo.saveFolder(null, folder);

        ReportUnit unit = (ReportUnit) repo.newResource(null, ReportUnit.class);
        setCommon(unit, "unit");
        unit.setParentFolder(folder);

        FileResource rep = (FileResource) repo.newResource(null, FileResource.class);
        setCommon(rep, "report");
        rep.readData(getClass().getResourceAsStream("/reports/jasper/SalesByMonth.jrxml"));
        unit.setMainReport(rep);

        JdbcReportDataSource jdbcDS = (JdbcReportDataSource) repo.newResource(null, JdbcReportDataSource.class);
        setCommon(jdbcDS, "ds");
        jdbcDS.setConnectionUrl("jdbc:mysql://localhost:3306/jasperserver");
        jdbcDS.setDriverClass("com.mysql.jdbc.Driver");
        jdbcDS.setUsername("user");
        jdbcDS.setPassword("passwd");
        unit.setDataSource(jdbcDS);

        repo.saveResource(null, unit);

        Resource res = repo.getResource(null, "/testLocalResourceReplace/unit");
        assertNotNull(unit);
        assertTrue(res instanceof ReportUnit);
        unit = (ReportUnit) res;
        ResourceReference dsRef = unit.getDataSource();
        assertNotNull(dsRef);
        assertTrue(dsRef.isLocal());
        Resource ds = dsRef.getLocalResource();
        assertTrue(ds instanceof JdbcReportDataSource);
        assertEquals("ds", ds.getName());

        JndiJdbcReportDataSource jndiDS = (JndiJdbcReportDataSource) repo.newResource(null, JndiJdbcReportDataSource.class);
        setCommon(jndiDS, "ds");
        jndiDS.setJndiName("jdbc/jserver");
        unit.setDataSource(jndiDS);

        repo.saveResource(null, unit);

        res = repo.getResource(null, "/testLocalResourceReplace/unit");
        assertNotNull(unit);
        assertTrue(res instanceof ReportUnit);
        unit = (ReportUnit) res;

        dsRef = unit.getDataSource();
        assertNotNull(dsRef);
        assertTrue(dsRef.isLocal());
        Resource dsLocal = dsRef.getLocalResource();
        assertEquals("ds", dsLocal.getName());
        assertTrue(dsLocal instanceof JndiJdbcReportDataSource);
        jndiDS = (JndiJdbcReportDataSource) dsLocal;
        assertEquals("jdbc/jserver", jndiDS.getJndiName());

        repo.deleteFolder(null, "/testLocalResourceReplace");
    }
}
