/*
 * Copyright (C) 2005 - 2007 JasperSoft Corporation.  All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased a commercial license agreement from JasperSoft,
 * the following license terms apply:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published by
 * the Free Software Foundation.
 *
 * This program is distributed WITHOUT ANY WARRANTY; and without the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see http://www.gnu.org/licenses/gpl.txt
 * or write to:
 *
 * Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330,
 * Boston, MA  USA  02111-1307
 */
package com.jaspersoft.jasperserver.api.metadata.view.service.impl;

import java.io.File;
import java.util.List;
//import java.util.ArrayList;
//import java.util.List;

//import org.hibernate.Session;
import org.acegisecurity.GrantedAuthority;
import org.acegisecurity.userdetails.UserDetails;
import org.acegisecurity.userdetails.UserDetailsService;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;

import com.jaspersoft.jasperserver.api.metadata.user.domain.Role;
import com.jaspersoft.jasperserver.api.metadata.user.domain.User;
import com.jaspersoft.jasperserver.api.metadata.user.service.UserAuthorityService;

import junit.framework.TestCase;
import junit.textui.TestRunner;

/**
 * @author swood
 * @version $Id: UserAuthorityServiceTest.java 10075 2007-09-14 15:31:49Z tony $
 */
public class UserAuthorityServiceTest extends TestCase {
	
	private ClassPathXmlApplicationContext appContext;
	ApplicationContext factory;
	UserAuthorityService userAuthService;
	
	public UserAuthorityServiceTest(String name) {
		super(name);
	}

	public static void main(String[] args) {
		TestRunner.run(UserAuthorityServiceTest.class);
	}

	public void setUp() throws Exception {
		System.out.println("Setup");
		//ClassPathResource resource = new ClassPathResource("userAuthorityService.xml");
		appContext = new ClassPathXmlApplicationContext(
		        new String[] {"hibernateConfig.xml", "userAuthorityService.xml", "viewService.xml"});

		factory = (ApplicationContext) appContext;

		/*
		 * Allow overrides of the jdbcProperties from a file defined in settings.xml
		 */
		String jdbcProperties = System.getProperty("test.hibernate.jdbc.properties");
		assertNotNull(jdbcProperties);
		//System.out.println("jdbcProperties: " + jdbcProperties);
		File cfgFile = new File(jdbcProperties);
		assertTrue(cfgFile.exists());

		userAuthService = (UserAuthorityService) factory.getBean("userAuthorityService");
	}
	
	public void tearDown() {
		System.out.println("Tear down");
		/*
		 *  Leave entries in the database
		 *  		
		 */
		
		// get rid of objects we don't want in samples
		
		userAuthService.deleteUser(null, "newUser");
		
		userAuthService.deleteRole(null, "newRole");
		userAuthService.deleteRole(null, "anotherRole");
		
		// temp
		userAuthService.deleteUser(null, "TestUser");
		userAuthService.deleteRole(null, "TestRole0");
		userAuthService.deleteRole(null, "TestRole1");
		userAuthService.deleteRole(null, "TestRole2");
		userAuthService.deleteRole(null, "TestRole3");
		userAuthService.deleteRole(null, "TestRole4");
		
		appContext.close();
	}

	public Role addRole(User user, String roleName) {
		System.out.println("addRole");
		
		Role role = userAuthService.getRole(null, roleName);
		
		if (role == null) {
			role = userAuthService.newRole(null);
			//System.out.println("role class: " + role.getClass().getName());
			role.setRoleName(roleName);
			role.setExternallyDefined(false);
			
			userAuthService.putRole(null, role);
		}
		
		userAuthService.addRole(null, user, role);
		return role;
	}
	
	private User findOrCreateUser(String username, String password) {
		User workingUser = userAuthService.getUser(null, username);
		if (workingUser == null) {
			workingUser = userAuthService.newUser(null);
			workingUser.setUsername(username);
			workingUser.setPassword((password == null ? username : password));
			workingUser.setFullName(username + " user");
			workingUser.setEnabled(true);
			workingUser.setExternallyDefined(false);
		
			userAuthService.putUser(null, workingUser);
		}

		return workingUser;
	}
	
	public void testNewUser() {
		System.out.println("testNewUser");
		User newUser = findOrCreateUser("newUser", "newPassword");
		// No roles added yet
		assertTrue(newUser.getRoles() == null || newUser.getRoles().isEmpty());
		
		addRole(newUser, "newRole");
		assertTrue(newUser.getRoles() != null || newUser.getRoles().size() == 1);
		
	}
	
	public void testOrdinaryUser() {
		System.out.println("testOrdinaryUser");
		User newUser = findOrCreateUser("joeuser", "joeuser");
		// No roles added yet
		assertTrue(newUser.getRoles() == null || newUser.getRoles().isEmpty());

		addRole(newUser, "ROLE_USER");
		assertTrue(newUser.getRoles() != null || newUser.getRoles().size() == 1);
	}
	
	public void testOrdinaryETLUser() {
		System.out.println("testOrdinaryETLUser");
		User newUser = findOrCreateUser("etluser", "etluser");
		// No roles added yet
		assertTrue(newUser.getRoles() == null || newUser.getRoles().isEmpty());

		addRole(newUser, "ROLE_USER");
		assertTrue(newUser.getRoles() != null || newUser.getRoles().size() == 1);
		
		addRole(newUser, "ROLE_ETL_ADMIN");
		assertTrue(newUser.getRoles() != null || newUser.getRoles().size() == 2);
	}
	
	public void testAdminETLUser() {
		System.out.println("testAdminETLUser");
		User newUser = findOrCreateUser("etladmin", "etladmin");
		// No roles added yet
		assertTrue(newUser.getRoles() == null || newUser.getRoles().isEmpty());

		addRole(newUser, "ROLE_USER");
		assertTrue(newUser.getRoles() != null || newUser.getRoles().size() == 1);

		addRole(newUser, "ROLE_ADMINISTRATOR");
		assertTrue(newUser.getRoles() != null || newUser.getRoles().size() == 2);
		
		addRole(newUser, "ROLE_ETL_ADMIN");
		assertTrue(newUser.getRoles() != null || newUser.getRoles().size() == 3);
		
	}
	
	public void testAdminUser() {
		System.out.println("testAdminUser");
		User newUser = findOrCreateUser("tomcat", null);
		// No roles added yet
		assertTrue(newUser.getRoles() == null || newUser.getRoles().isEmpty());

		addRole(newUser, "ROLE_USER");
		assertTrue(newUser.getRoles() != null || newUser.getRoles().size() == 1);

		addRole(newUser, "ROLE_ADMINISTRATOR");
		assertTrue(newUser.getRoles() != null || newUser.getRoles().size() == 2);
		
	}
	
	public void testJasperadminUser() {
		System.out.println("testJasperadminUser");
		User newUser = findOrCreateUser("jasperadmin", "password");
		// No roles added yet
		assertTrue(newUser.getRoles() == null || newUser.getRoles().isEmpty());

		addRole(newUser, "ROLE_USER");
		assertTrue(newUser.getRoles() != null || newUser.getRoles().size() == 1);

		addRole(newUser, "ROLE_ADMINISTRATOR");
		assertTrue(newUser.getRoles() != null || newUser.getRoles().size() == 2);	
	}	
	
	public void testAnonymousUser() {
		System.out.println("testAnonymousUser");
		User newUser = findOrCreateUser("anonymousUser", "");
		// No roles added yet
		assertTrue(newUser.getRoles() == null || newUser.getRoles().isEmpty());
		
		addRole(newUser, "ROLE_ANONYMOUS");
		assertTrue("newUser.getRoles() wrong size", newUser.getRoles() != null || newUser.getRoles().size() == 1);
		//assertTrue("newRole.getUsers() wrong size", newRole.getUsers() != null || newRole.getUsers().size() == 1);
		
		UserDetailsService userDetailsService = (UserDetailsService) userAuthService;
		
		UserDetails details = userDetailsService.loadUserByUsername(newUser.getUsername());
		
		assertTrue("load by Acegi interface", details != null);
		
		GrantedAuthority[] authorities = details.getAuthorities();
		assertTrue("authorities wrong size", authorities != null || authorities.length == 1);
		assertTrue("right authority", authorities[0].getAuthority().equalsIgnoreCase("ROLE_ANONYMOUS"));
		
	}
	
	public void testUpdateUser() {
		System.out.println("testUpdateUser");
		User newUser = findOrCreateUser("newUser", "");
		assertNotNull(newUser);
		assertTrue(newUser.isEnabled());
		userAuthService.disableUser(null, newUser.getUsername());
		newUser = findOrCreateUser("newUser", "");
		assertTrue("error: user still disabled", !newUser.isEnabled());
		
		addRole(newUser, "anotherRole");
		
		Role r = userAuthService.getRole(null, "anotherRole");
		
		assertTrue("error: 'anotherRole' does not exist", r != null);
		assertTrue("error: newUser does not have anotherRole", newUser.getRoles().contains(r));

		userAuthService.putUser(null, newUser);
		
		newUser.removeRole(r);
		assertTrue("error: newUser still contains anotherRole", !newUser.getRoles().contains(r));

		userAuthService.putUser(null, newUser);
		
		newUser = userAuthService.getUser(null, "newUser");
		assertTrue("error: after retrieval - newUser REALLY still contains anotherRole", !newUser.getRoles().contains(r));
		
	}
		
	public void testGetUsersAndRoles() {
		System.out.println("testGetUsersAndRoles");
		// Depending on the order of test classes that are run, we could have more users and roles
		// than we created here.
		
		List results = userAuthService.getUsers(null, null);
		assertTrue("getUsers right size: expected at least 3, got " + results.size(), results.size() >= 3);
		results = userAuthService.getRoles(null, null);
		assertTrue("getRoles right size: expected at least 4, got " + results.size(), results.size() >= 4);
	}
	
/*	public void testObjectPermissionMaintenance() {
		
	}
*/
}
