
/**
 * TreeSupport is a convenient wrapper for Tree to use it with JasperServer.
 * You can extend it to change parameters and/or look and feel
 * @param {String} treeName unique name for the tree on the page
 * @param {String} providerId Tree Data Provider id
 * @param {String} rootUri uri for the root of this tree 
 *   (usually '/' but could be something like '/area/node' if this tree is supposed 
 *   to show only particular branch from the data structure)
 * @param {Function} treeErrorHandlerFn optional function to be called if error occurs
 */
function TreeSupport(treeName, providerId, rootUri, treeErrorHandlerFn) {

    // instance variables
    this.treeName = treeName;
    this.providerId = providerId;
    this.rootUri = rootUri;
    this.treeErrorHandlerFn = treeErrorHandlerFn;

    this.tree = null;
    this.inInit = true;


    // defaul types
    // used to separate folders from leaves
    // if TreeSupport extended, could be used to assign different icons based on type
    this.Types = {
      folderType: 'com.jaspersoft.jasperserver.api.metadata.common.domain.Folder'
    };

    // default icons
    this.closedGif = 'images/treeimages/folder_closed.gif';
    this.openGif = 'images/treeimages/folder_open.gif';
    this.pageIcon = 'images/treeimages/page16x16.gif';
    this.userIcon = 'images/treeimages/user_16x16.gif';
    this.helpIcon = 'images/treeimages/help_16x16.gif';

    this.waitIcon = 'images/spinner_moz.gif';

    // default ajax related values
    this.ajaxBufferId = 'ajaxbuffer'; // id of a DIV elements that receives server response
    this.nodeTextId = 'treeNodeText'; // id of a text element that contains JSONized tree
    this.urlGetNode = 'flow.html?_flowId=treeFlow&method=getNode'; // server url for 'getNode' method
    this.urlGetChildren = 'flow.html?_flowId=treeFlow&method=getChildren'; // server url for 'getChildren' method

    /////////////////////////////////////
    // methods
    /////////////////////////////////////

    /**
     * Loads tree from server and renders it into given container.
     * Calls user callback at the end.
     * depth controls how many levels of children to prefetch at this load
     * @type Asynchronous method
     * @param {String} containerId
     * @param {Number} depth
     * @param {Function} userCallbackFn
     * @param {Function} errorCallbackFn
     */
    this.showTree = function (containerId, depth, userCallbackFn, errorCallbackFn) {
        var url = this.urlGetNode + '&provider=' + providerId + '&uri=' + rootUri + '&depth=' + depth;
        this._showTree(containerId, url, userCallbackFn, errorCallbackFn);
    }

    /**
     * Loads tree from the server and renders it into given container.
     * Calls user callback at the end.
     * prefetchedListStr controls which tree branches to prefetch at this load
     * @type Asynchronous method
     * @param {String} containerId DOM id for container element where to render a tree
     * @param {String} prefetchedListStr comma separated uris to prefetch (example: '/reports/samples,/adhoc/topics')
     * @param {Function} userCallbackFn optional user function object to call after tree gets loaded and rendered
     * @param {Function} errorCallbackFn optional user error handler function to call if error occured
     */
    this.showTreePrefetchNodes = function (containerId, prefetchedListStr, userCallbackFn, errorCallbackFn) {
        var url = this.urlGetNode + '&provider=' + providerId + '&uri=' + rootUri;
        if (prefetchedListStr) {
            url += '&prefetch=' + prefetchedListStr;
        }
        this._showTree(containerId, url, userCallbackFn, errorCallbackFn);
    }  
    
    this._showTree = function (containerId, url, userCallbackFn, errorCallbackFn) {

        this.inInit = true;

        if (!rootUri) {
            rootUri = '/';
        }

        var callback = function(obj, ci, uc, ec) {
            return function() {
                return obj.showTreeCallback(ci, uc, ec);
            }
        }(this, containerId, userCallbackFn, errorCallbackFn);

        ajaxTargettedUpdate(
            url,
            this.ajaxBufferId,
            null,
            callback
        );

        var div = document.getElementById(containerId);
        div.innerHTML = '<img src="' + this.waitIcon + '">';

    }

    this.showTreeCallback = function (containerId, userCallbackFn, errorCallbackFn) {

        // get JSONized Node
        var div = document.getElementById(this.nodeTextId);
        if (div == null) {
            if (errorCallbackFn) {
                errorCallbackFn();
            }
            return;
        }
        
        var rootObj = window.eval('(' + div.innerHTML + ')');

        // clean AJAX buffer
        div = document.getElementById(this.ajaxBufferId);
        div.innerHTML = '';

        // build the tree
        var root = this.processNode(rootObj);
        this.tree = new Tree(this.treeName, root, this.providerId, 'images/tree', false);
        this.tree.resetStates();

        div = document.getElementById(containerId);
        div.innerHTML = trees[this.treeName].renderTree();

        this.inInit = false;

        if (userCallbackFn) {
            userCallbackFn();
        }
    }

    /**
     * Renders tree into a given container
     * @param {Object} containerId DOM id for a container to render tree to
     */
    this.renderTree = function (containerId) {
        div = document.getElementById(containerId);
        div.innerHTML = trees[this.treeName].renderTree();
    }

    /**
     * internally used method to turn server model into javascript tree model
     * Be advised of the power of 'extra' property of server node object.
     * You can set there pretty much anything and therefore customize you tree behaviour.
     * TreeNode is available in node handlers which you may assign as callback functions to your code
     * @param {Object} metaNode server node
     */
    this.processNode = function (metaNode) {
        var param = {};
        param.id = metaNode.id;
        param.type = metaNode.type;
        param.uri = metaNode.uri;
        param.extra = metaNode.extra;

        var localRoot = new TreeNode(metaNode.label, this.getIconForNode(metaNode), param);
        localRoot.setHandler(this.getHandlerForNode(metaNode));
        localRoot.setHasChilds(this.isFolderNode(metaNode));

        localRoot.addOpenEventListener(
            function(obj) {
                return function(nodeId) {
                    obj.openNodeHandler(nodeId);
                }
            } (this)
        );

        var ch = metaNode.children;
        if (ch != null) {
            var len = ch.length;
            if (len == 0) {
                localRoot.setHasChilds(false);
            } else {
                for (var i = 0; i < len; i++) {
                    var chNodeObj = ch[i];
                    var chTreeNode = this.processNode(chNodeObj);
                    localRoot.addChild(chTreeNode);
                }
            }
            localRoot.setIsLoaded(true);
        }
        return localRoot;
    }


    /**
     * Returns folder icon for folders, page icon for leaves.
     * It is a good candidate to be overwritten to customize icons based on a node type
     * @param {Object} node TreeNode instance
     */
    this.getIconForNode = function (node) {
        if (node.type == this.Types.folderType) {
            return new Array(this.closedGif,this.openGif);
        } else {
            return this.pageIcon;
        }
    }

    this.openNodeHandler = function (nodeId){ 
        var node = nodes[nodeId];
        if (node && !node.isLoaded()) {
            this.getTreeNodeChildren(node, null, this.treeErrorHandlerFn);
        }
    }

    this.getHandlerForNode = function (node) {
        if (node.type == this.Types.folderType) {
            return this.treeFolderHandler;
        } else {
            return this.treeLeafHandler;
        }
    }

    this.isFolderNode = function (node) {
        return (node.type == this.Types.folderType);
    }

    /**
     * Default empty handler for a folder
     * @param {Object} node TreeNode instance clicked
     */
    this.treeFolderHandler = function (node) {
    }

    /**
     * Default empty handler for a leaf
     * @param {Object} node TreeNode instance clicked
     */
    this.treeLeafHandler = function (node) {
    }

    /**
     * Dynamically loads children for the node
     * @param {Object} parentNode TreeNode instance
     * @param {Function} userCallbackFn optional user function
     * @param {Function} errorCallbackFn optional error handler function
     */
    this.getTreeNodeChildren = function (parentNode, userCallbackFn, errorCallbackFn) {

        var uri = parentNode.getParam().uri;

        var callback = function(obj, ni, uc, ec) {
            return function() {
                return obj.getTreeNodeChildrenCallback(ni, uc, ec);
            }
        }(this, parentNode.getID(), userCallbackFn, errorCallbackFn);

        ajaxTargettedUpdate(
            this.urlGetChildren + '&provider=' + this.providerId + '&uri=' + uri,
            this.ajaxBufferId,
            null,
            callback
        );

        if (!this.inInit) {
            //var waitNode = new TreeNode('', this.waitIcon, null);
            //parentNode.addChild(waitNode);
            //parentNode.refreshNode();

            parentNode.setIcon(this.waitIcon);
            var img = document.getElementById('iconimage' + parentNode.getID());
            img.src = this.waitIcon;
        }

    }

    this.getTreeNodeChildrenCallback = function (parentNodeId, userCallbackFn, errorCallbackFn) {

        var div = document.getElementById(this.nodeTextId);
        if (div == null) {
            if (errorCallbackFn) {
                errorCallbackFn();
            }
            return;
        }
        var ns = window.eval('(' + div.innerHTML + ')');
        div = document.getElementById(this.ajaxBufferId);
        div.innerHTML = '';

        var parentNode = nodes[parentNodeId];
        parentNode.resetChilds();
        parentNode.setIcon(this.getIconForNode(parentNode.param));
        var img = document.getElementById('iconimage' + parentNode.getID());
        img.src = parentNode.getIcon();
        var len = ns.length;
        if (len == 0) {
            parentNode.setHasChilds(false);
        } else {
            for (var i = 0; i < len; i++) {
                var node = this.processNode(ns[i]);
                parentNode.addChild(node);
            }
        }

        parentNode.setIsLoaded(true);
        parentNode.refreshNode();

        if (userCallbackFn) {
            userCallbackFn();
        }
    }

    /**
     * Dynamically loads children for a given node.
     * Makes sure that all requested nodes get prefetched.
     * Nodes to be prefetched have to have parentNode as a common (grand*)parent
     * @param {Object} parentNode parent TreeNode
     * @param {String} prefetchedListStr comma separated URIs to be prefetched
     * @param {Function} userCallbackFn optional user callback function
     * @param {Function} errorCallbackFn optional error handler function
     */
    this.getTreeNodeChildrenPrefetched = function (parentNode, prefetchedListStr, userCallbackFn, errorCallbackFn) {

        var uri = parentNode.getParam().uri;

        var url = this.urlGetNode + '&provider=' + this.providerId + '&uri=' + uri;
        if (prefetchedListStr) {
            url += '&prefetch=' + prefetchedListStr;

        }
        var callback = function(obj, ni, uc, ec) {
            return function() {
                return obj.getTreeNodeChildrenPrefetchedCallback(ni, uc, ec);
            }
        }(this, parentNode.getID(), userCallbackFn, errorCallbackFn);

        ajaxTargettedUpdate(
            url,
            this.ajaxBufferId,
            null,
            callback
        );

        if (!this.inInit) {
            //var waitNode = new TreeNode('', this.waitIcon, null);
            //parentNode.addChild(waitNode);
            //parentNode.refreshNode();

            parentNode.setIcon(this.waitIcon);
            var img = document.getElementById('iconimage' + parentNode.getID());
            img.src = this.waitIcon;
        }

    }

    this.getTreeNodeChildrenPrefetchedCallback = function (parentNodeId, userCallbackFn, errorCallbackFn) {

        var div = document.getElementById(this.nodeTextId);
        if (div == null) {
            if (errorCallbackFn) {
                errorCallbackFn();
            }
            return;
        }
        var n = window.eval('(' + div.innerHTML + ')');
        div = document.getElementById(this.ajaxBufferId);
        div.innerHTML = '';

        var parentNode = nodes[parentNodeId];
        parentNode.resetChilds();
        parentNode.setIcon(this.getIconForNode(parentNode.param));
        if (n.children) {
            for (var i = 0; i < n.children.length; i++) {
                var node = this.processNode(n.children[i]);
                parentNode.addChild(node);
            }
        }

        parentNode.setIsLoaded(true);
        parentNode.refreshNode();

        if (userCallbackFn) {
            userCallbackFn();
        }
    }

    /**
     * Expands the tree up to a given node, and then select it
     * @param {String} uriStr node URI
     */
    this.openAndSelectNode = function(uriStr) {
        var fn = function(node) {
            if (node.parent) {
                var tree = trees[node.getTreeId()];
                if (tree && tree.rootNode != node.parent && tree.getState(node.parent.id) == 'closed') {
                    node.parent.handleNode();
                }
            }
            node.selectNode();
        };

        this.processNodePath(uriStr, fn);

        // scroll tree container to make selected node visible
        var tree = this.tree;
        if (tree) {
            var selectedNode = tree.selectedNode;
            if (selectedNode) {
                var rootNodeId = 'node' + tree.rootNode.getID();
                var rootNodeElement = document.getElementById(rootNodeId);
                var container = rootNodeElement.parentNode;
                var selectedNodeId = 'node' + selectedNode;
                var selectedNodeElement = document.getElementById(selectedNodeId);
                if (container) {
                    var ch = container.clientHeight;
                    var cw = container.clientWidth;
                    var cst = container.scrollTop;
                    var csl = container.scrollLeft;
                    var nt = selectedNodeElement.offsetTop;
                    var nl = selectedNodeElement.offsetLeft;
                    var nh = selectedNodeElement.clientHeight;
                    var nw = selectedNodeElement.clientWidth;
                    if (nt > (cst + ch)) { // node is below
                        container.scrollTop = nt - (ch / 2 - nh / 2);
                    } else if ((nt + nh) < cst) { // node is above
                        container.scrollTop = nt - (ch / 2 - nh / 2);
                    }
                    if (nl > (csl + cw)) { // node is out left
                        container.scrollLeft = nl - (cw / 2 - nw / 2);
                    } else if ((nl + nw) < csl) { // node is out right
                        container.scrollTop = nl - (cw / 2 - nw / 2);
                    }
                }
            }
        }
    }

    this.processNodePath = function(uriStr, fnForNode) {

        var path = uriStr.split('/');
        var node = this.tree.rootNode;
        var i;

        for (i = 0; i < path.length; i++) {
            if (!path[i]) {
                continue;
            }
            node = this.findNodeChildByMetaName(node, path[i]);
            if (!node) {
                return;
            }
            fnForNode(node);
        }

    }

    /**
     * Returns TreeNode which is last node in node hierarchical chain for a given uri
     * If returned node corresponds to uriStr, it means no more server requests needed
     * If it corresponds to parent (grand-parent, etc.), the value shows existing root 
     * from which the rest should be requested from server
     * Example: uriStr='/area/subarea/dept/prod1', return is TreeNode with uri '/area/subarea'.
     * It means, we need to load children of 'subarea' and children of 'dept' from server
     * 
     * @param {Object} uriStr
     */
    this.findLastLoadedNode = function(uriStr) {
        var nodeHolder = { node: null };
        var fn = function(holder) {
            return function(node) {
                holder.node = node;
            }
        }(nodeHolder);

        this.processNodePath(uriStr, fn);

        return nodeHolder.node;
    }

    this.findNodeChildByMetaName = function (node, name) {
        if (node.hasChilds()) {
            for (var i = 0; i < node.childs.length; i++) {
                if (node.childs[i].param.id == name) {
                    return node.childs[i];
                }
            }
        }
        return null;
    }


} // TreeSupport