/*
 * Copyright (C) 2006 JasperSoft http://www.jaspersoft.com
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed WITHOUT ANY WARRANTY; and without the 
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see http://www.gnu.org/licenses/gpl.txt 
 * or write to:
 * 
 * Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330,
 * Boston, MA  USA  02111-1307
 */

package com.jaspersoft.jasperserver.ws.axis2.repository;

import java.util.HashMap;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.jaspersoft.jasperserver.api.metadata.common.domain.Resource;
import com.jaspersoft.jasperserver.api.metadata.common.service.RepositoryService;
import com.jaspersoft.jasperserver.api.metadata.jasperreports.domain.ReportUnit;
import com.jaspersoft.jasperserver.api.metadata.xml.domain.impl.Argument;
import com.jaspersoft.jasperserver.api.metadata.xml.domain.impl.OperationResult;
import com.jaspersoft.jasperserver.api.metadata.xml.domain.impl.ResourceDescriptor;
import com.jaspersoft.jasperserver.ws.axis2.RepositoryServiceContext;
import com.jaspersoft.jasperserver.ws.axis2.ResourceHandler;
import com.jaspersoft.jasperserver.ws.axis2.ResultAttachments;
import com.jaspersoft.jasperserver.ws.axis2.ServiceRequest;
import com.jaspersoft.jasperserver.ws.axis2.WSException;

/**
 * @author gtoffoli
 * @version $Id: RepositoryResourceHandler.java 9354 2007-08-06 14:47:57Z lucian $
 */
public abstract class RepositoryResourceHandler implements ResourceHandler {

	private static final Log log = LogFactory
			.getLog(RepositoryResourceHandler.class);

	public ResourceDescriptor describe(Resource resource, Map arguments,
			RepositoryServiceContext serviceContext) throws WSException {
		ResourceDescriptor descriptor = new ResourceDescriptor();
		setCommonAttributes(resource, descriptor);

		doDescribe(resource, descriptor, arguments, serviceContext);
		return descriptor;
	}

	protected void setCommonAttributes(Resource resource,
			ResourceDescriptor descriptor) {
		descriptor.setUriString(resource.getURIString());
		descriptor.setDescription(resource.getDescription());
		descriptor.setLabel(resource.getLabel());
		descriptor.setName(resource.getName());
		descriptor.setResourceType(resource.getResourceType());
		descriptor.setParentFolder(resource.getParentFolder());
		descriptor.setVersion(resource.getVersion());
		descriptor.setCreationDate(resource.getCreationDate());
	}

	protected abstract void doDescribe(Resource resource,
			ResourceDescriptor descriptor, Map arguments,
			RepositoryServiceContext serviceContext) throws WSException;

	public void put(ServiceRequest request) throws WSException {
		ResourceDescriptor descriptor = request.getRequestDescriptor();
		RepositoryServiceContext context = request.getContext();
		RepositoryService repository = context.getRepository();
		OperationResult result = request.getResult();

		Resource parentResource = getModifyingParentResource(request);

		if (descriptor.getIsNew()) {
			if (parentResource == null) {
				Resource resource = toNewResource(descriptor, context);
				repository.saveResource(null, resource);
				result.getResourceDescriptors().add(
						context.createResourceDescriptor(resource
								.getURIString()));
			} else {
				SubResourceHandler parentHandler = (SubResourceHandler) context
						.getHandlerRegistry().getHandler(parentResource);
				Resource subResource = parentHandler.setSubResource(parentResource,
						request);
				if (subResource != null) {
					repository.saveResource(null, parentResource);
					result.getResourceDescriptors().add(
							context.createResourceDescriptor(subResource));
				}
			}
		} else {
			if (parentResource == null) {
				log.debug("Put: modifying " + descriptor.getWsType());

				Class resourceType = getResourceType();
				Resource resource = repository.getResource(null, descriptor
						.getUriString(), resourceType);

				if (resource == null) {
					result.setReturnCode(2);
					result.setMessage(context.getMessage(
							"webservices.error.resourceNotFound", null));
				} else {
					copyToResource(resource, descriptor, context);
					repository.saveResource(null, resource);
					result.getResourceDescriptors().add(
							context.createResourceDescriptor(resource));
				}
			} else {
				// Report unit modification....
				SubResourceHandler parentHandler = (SubResourceHandler) context
						.getHandlerRegistry().getHandler(parentResource);
				Resource subResource = parentHandler.setSubResource(parentResource,
						request);
				if (subResource != null) {
					repository.saveResource(null, parentResource);

					Map options = new HashMap();
					options.put(ReportUnitHandler.OPTION_REPORT_UNIT_CONTENTS,
							Boolean.TRUE);
					result.getResourceDescriptors().add(
							context.createResourceDescriptor(parentResource,
									options));
				}
			}
		}
	}

	protected Resource getModifyingParentResource(ServiceRequest request)
			throws WSException {
		Resource resource = null;
		// for now, only report units can act as parent resources
		String reportUnitUrl = request
				.getRequestArgument(Argument.MODIFY_REPORTUNIT);
		if (reportUnitUrl != null && reportUnitUrl.length() > 0) {
			log.debug("Put: adding/modifying resource in reportUnit "
					+ reportUnitUrl);
			ReportUnit reportUnit = (ReportUnit) request.getContext().getRepository()
					.getResource(null, reportUnitUrl);
			if (reportUnit == null) {
				throw new WSException(
						WSException.REFERENCED_RESOURCE_NOT_FOUND, request
								.getContext().getMessage(
										"webservices.error.reportUnitNotFound",
										null));
			}
			resource = reportUnit;
		}
		return resource;
	}

	protected Resource toNewResource(ResourceDescriptor descriptor,
			RepositoryServiceContext serviceContext) throws WSException {
		Class resourceType = getResourceType();
		Resource resource = serviceContext.getRepository().newResource(null,
				resourceType);
		resource.setParentFolder(descriptor.getParentFolder());
		resource.setVersion(Resource.VERSION_NEW);
		resource.setName(descriptor.getName());
		copyToResource(resource, descriptor, serviceContext);

		return resource;
	}

	public void copyToResource(Resource resource,
			ResourceDescriptor descriptor,
			RepositoryServiceContext serviceContext) throws WSException {
		resource.setLabel(descriptor.getLabel());
		resource.setDescription(descriptor.getDescription());

		updateResource(resource, descriptor, serviceContext);
	}

	protected abstract void updateResource(Resource resource,
			ResourceDescriptor descriptor,
			RepositoryServiceContext serviceContext) throws WSException;

	protected Resource toChildResource(ResourceDescriptor descriptor,
			RepositoryServiceContext context) throws WSException {
		String childType = descriptor.getWsType();
		RepositoryResourceHandler childHandler = (RepositoryResourceHandler) context
				.getHandlerRegistry().getHandler(childType);
		return childHandler.toNewResource(descriptor, context);
	}

	public void getAttachments(Resource resource, Map arguments,
			ResourceDescriptor descriptor,
			ResultAttachments attachments, RepositoryServiceContext serviceContext) {
		// default implementation does nothing
	}

	public void delete(ResourceDescriptor descriptor,
			RepositoryServiceContext serviceContext) throws WSException {
        serviceContext.getRepository().deleteResource(null, descriptor.getUriString());
	}

}
