<?php
# SHDFS(Simple Http Distributed File System)
# Copyright (C) Digital Adventure Inc.
#
# 2010/05/28 hoshino
#
# 必要なpearライブラリ
#  Cache_Lite
#  MIME_Type

	require_once(dirname(__FILE__) . '/shdfs.ini.php') ;
	$conf = shdfs_init_conf() ;

	# 停止サーバ
	$conf['replication_hosts'] = exclusion_stop_servers($conf['replication_hosts']) ;

	# 自動キャッシュガベージコレクション
	if(substr(PHP_OS,0,3) != 'WIN') {
		# WindowsOSの場合はバックグラウンド実行できないのでなにもしない
		if(mt_rand(1,10000) === 1 and $conf['auto_garbage_collection'] === TRUE){
			# 約10000クエストに1回、かつ auto_garbage_collection がTRUEの場合、
			# 古いキャッシュを消すCLIコマンドを実行する
			$php           = $conf['php_cli_path'] ;
			$document_root = $_SERVER['DOCUMENT_ROOT'] ;

			# コマンドをバックグラウンドで起動
			$cmd = sprintf("%s %s delete_cache %s &",$php,__FILE__,$document_root) ;
			shell_exec($cmd) ;
		}
	}

	# キャッシュクリアチェック
	if($argv[1] === 'delete_cache' and strlen($argv[2]) != 0){
		# CUI実行で delete_cache が指定された場合は古いキャッシュを消して終わる
		$cache_path = sprintf("%s/.shdfs/.cache",rtrim($argv[2],'/')) ;
		delete_cache($cache_path,86400) ;

		# コマンドラインから呼び出されてキャッシュクリアしたら停止する
		exit ;
	}

	# 内部リクエストチェック
	$result = check_request_header('X-SHDFS-Internal-Request','yes') ;
	if($result === TRUE){
		# レプリケートサーバからのGETリクエストは .htaccess の
		# mod_rewrite 設定によってこのスクリプトは実行されないはず
		# 実行されてしまった場合はエラーとする(そうしないとリクエストがループしてしまうよ)
		header("HTTP/1.0 404 Not Found") ;
		echo "404 Not Found" ;
		exit ;
	}

	# レプリケートサーバに問い合わせ
	$remote_entitys = search_remote_entity($conf['replication_hosts']) ;
	$entity_number  = (int)$remote_entitys['number'] ; # レプリケートサーバに存在するコピーの数

	# ローカルファイル存在確認
	$local_entity = search_local_entty($_SERVER['DOCUMENT_ROOT']) ;
	$is_local_exist  = $local_entity['is_local_exist'] ;
	$local_filemtime = $local_entity['local_filemtime'] ;
	$local_path      = $local_entity['local_path'] ;

	if($is_local_exist === TRUE){
		# ローカルファイルとリモートファイル比較
		$is_local_old    = FALSE ;
		foreach((array)$remote_entitys['exists'] as $values){
			$last_modified_int = $values['last_modified_int'] ;
			if($last_modified_int > $local_filemtime){
				# 他のレプリケートサーバのファイルよりローカルファイルが古い
				$is_local_old = TRUE ;
				# 古いファイルは消してしまう
				@unlink($local_path) ;
				break ;
			}
		}

		if($is_local_old === FALSE){
			# ローカルディスクにあるファイルが古くなければそれを送信する
			send_local_entity($local_path) ;
			exit ;
		}
	}


	# キャッシュファイル存在確認
	$cache_path = sprintf("%s/.cache",dirname(__FILE__)) ;
	$cache_entity = search_local_entty($cache_path) ;
	$is_cache_exist  = $cache_entity['is_local_exist'] ;
	$cache_filemtime = $cache_entity['local_filemtime'] ;
	$cache_path      = $cache_entity['local_path'] ;

	if($is_cache_exist === TRUE and $entity_number >= $conf['replication_number']){
		# キャッシュファイルとリモートファイル比較
		$is_cache_old    = FALSE ;
		foreach((array)$remote_entitys['exists'] as $values){
			$last_modified_int = $values['last_modified_int'] ;
			if($last_modified_int > $cache_filemtime){
				# 他のレプリケートサーバのファイルよりローカルファイルが古い
				$is_cache_old = TRUE ;
				# 古いファイルは消してしまう
				@unlink($cache_path) ;
				break ;
			}
		}

		if($is_cache_old === FALSE){
			# ローカルディスクにあるキャッシュが古くなければそれを送信する
			send_local_entity($cache_path) ;
			exit ;
		}
	}

	if($entity_number === 0){
		# どのレプリケーションホストにも存在しなかったら404 Not Found
		header("HTTP/1.0 404 Not Found") ;
		echo "404 Not Found" ;
		exit ;
	}

	# リモートから転送する
	send_remote_entity($remote_entitys) ;

exit ;

function get_request_header($key=''){
	# リクエストヘッダ取得(大文字小文字は無視)
	$key   = strtolower($key) ;
	foreach(apache_request_headers() as $hkey => $hvalue){
		$hkey   = strtolower($hkey) ;
		if($hkey === $key){
			return $hvalue ;
		}
	}
	return NULL ;
}

function get_request_range(){
	# リクエストヘッダのうち "Range:" ヘッダのみ解釈する
	# format
	# Range: bytes=100-199
	#
	# 戻り値
	# $array['start'] = 100
	# $array['end']   = 109 or NULL

	$return = array() ;
	$range_header = get_request_header('Range') ;

	preg_match('/(\d*)\-(\d*)/',$range_header,$matches) ;
	$return['start'] = $matches[1] ;
	$return['end']   = $matches[2] ;
	return $return ;
}

function check_request_header($key='',$value=''){
	# 大文字小文字を無視してヘッダ$keyに値$valueがあるか調べる
	$request_headers = array() ;
	$key   = strtolower($key) ;
	$value = strtolower($value) ;
	foreach(apache_request_headers() as $hkey => $hvalue){
		$hkey   = strtolower($hkey) ;
		$hvalue = strtolower($hvalue) ;
		if($hkey === $key and $hvalue === $value){
			return TRUE ;
		}
	}
	return FALSE ;
}


function search_local_entty($base_path=''){
	$return = array() ;

	# ファイルパス組み立て
	$request_uri_path = parse_url($_SERVER['REQUEST_URI'],PHP_URL_PATH) ;
	$local_path       = sprintf("%s%s",$base_path,$request_uri_path) ;
	$local_path_part  = sprintf("%s.part",$local_path) ;

	# ファイルが存在するかチェック
	if(file_exists($local_path) === FALSE){
		# ローカルディスクにリクエストファイルが無い場合はFASLE
		$return['is_local_exist'] = FALSE ;
		return $return ;
	}

	# ゾンビ化確認
	$filesize  = filesize($local_path) ;
	$filemtime = filemtime($local_path) ;
	if($filesize === 0 and time() - $filemtime > 3600){
		# ファイルサイズゼロで残ってしまっていたらゾンビファイルなので削除する
		@unlink($local_path) ;
		@unlink($local_path_part) ;
		# ファイルは無かったこととしてFALSE
		$return['is_local_exist'] = FALSE ;
		return $return ;
	}

	# レプリケート中でないか確認
	if(file_exists($local_path_part) === TRUE){
		# partファイルがある場合はレプリケート途中なのでFALSEを返す
		$return['is_local_exist'] = FALSE ;
		return $return ;
	}

	# ローカルファイルは存在する
	$return['is_local_exist']  = TRUE ;
	$return['local_filemtime'] = $filemtime ;
	$return['local_path']      = $local_path ;

	return $return ;
}

function search_remote_entity($replication_hosts=array()){
	# レプリケーションサーバの中から実ファイルのあるURLをリストする
	# 戻り値
	# $return['exists'][]	array	ファイルが存在するホストの一覧
	# $return['number']	int	レプリケートファイルの数

	global $conf ;
	require_once("Cache/Lite.php") ;
	$options = array(
		'cacheDir' => $conf['cache_lite_dir'],
		'lifeTime' => $conf['cache_lite_lifetime']
	) ;

	$Cache_Lite = new Cache_Lite($options) ;
	$serialize = $Cache_Lite->get($_SERVER['REQUEST_URI']) ;
	if($serialize != FALSE){
		$return = unserialize($serialize) ;
		return $return ;
	}

	$return = array() ;
	$return['number'] = 0 ; # 発見した実ファイル数の初期値

	foreach((array)$replication_hosts as $value){
		list($ip,$port,$server_name) = explode(':',$value) ;

		# レプリケーションサーバごとにHEADリクエスト
		$res = shdfs_head_request($ip,$port,$server_name,$_SERVER['REQUEST_URI']) ;
		if($res === FALSE){
			# 接続失敗した場合はレプリケートホストから一時除外する
			add_stop_server($ip) ;
			continue ;
		}

		# レスポンスからパラメータ取り出し
		$http_response_code        = $res['response_code'] ;
		$http_header_length   = (int)$res['headers']['content-length'] ;
		$http_header_last_modified = $res['headers']['last-modified'] ;

		if($http_response_code != '200'){
			# 無い場合はなにもしない
			continue ;
		}

		if($http_header_length === 0){
			# サイズがゼロの場合はレプリケート中かもしれないので除外
			# ただしコピー数としてはカウント(そうしないとコピーが多くなってしまうよ)
			$return['number'] += 1 ;
			continue ;
		}

		# 最終更新日のUNIXタイム
		$last_modified_int = strtotime($http_header_last_modified) ;
		# 最新ファイルのUNIXタイム
		if($last_modified_int > $newest_modified_int){
			$newest_modified_int = $last_modified_int ;
		}

		$return['exists'][] = array(
			'ip'                 => $ip,
			'port'               => $port,
			'server_name'        => $server_name,
			'last_modified_int'  => $last_modified_int
		) ;
		$return['number'] += 1 ;
	}

	# 古いファイルを持っているレプリケートサーバは除外する
	$delete_keys = array() ;
	foreach((array)$return['exists'] as $key => $value){
		if($value['last_modified_int'] < $newest_modified_int){
			$delete_keys[] = $key ;
		}
	}
	foreach((array)$delete_keys as $key){
		unset($return['exists'][$key]) ;
		$return['number'] -= 1 ;
	}

	$serialize = serialize($return) ;
	$a = $Cache_Lite->save($serialize,$_SERVER['REQUEST_URI']) ;

	return $return ;
}

function send_local_entity($local_path=''){
	# 自分自身のコンテンツファイルを送信

	if(file_exists($local_path) === FALSE){
		# なぜかファイルが無い
		header("HTTP/1.0 500 Server Error") ;
		echo "500 Server Error\n" ;
		exit ;
	}

	require_once 'MIME/Type.php';
	# ヘッダ送信
	$filesize  = filesize($local_path) ;
	$filemtime = filemtime($local_path) ;
	$range = get_request_range() ;
	if($range['start'] >= 1){
		# 部分的GETの場合は 206 Partial Content を返す
		$content_length = $filesize - $range['start'] ;
		header('HTTP/1.1 206 Partial Content') ;
		header(sprintf("Content-Length: %d",$content_length)) ;
		header(sprintf("Content-Range: bytes %d-%d/%d",$range['start'],$filesize -1,$filesize)) ;
	}else{
		$content_length = $filesize ;
		header('HTTP/1.1 200 OK') ;
		header(sprintf("Content-Length: %d",$content_length)) ;
		$range['start'] = 0 ;
	}
	header(sprintf("Date: %s GMT",gmdate('D, d M Y H:i:s', time($filemtime)))) ;
	header(sprintf("Content-Type: %s",MIME_Type::autoDetect($local_path))) ;
	header('X-SHDFS-send-entity: local') ;
	ob_end_flush();
	flush();

	# ファイル送信
	$fp = @fopen($local_path,'r') ;
	$sh = @fopen('php://output','w') ;
	fseek($fp,(int)$range['start']);
	while(feof($fp) === FALSE){
		$buffer = fread($fp,8192) ;
		fwrite($sh,$buffer) ;
	}
	fclose($fp) ;
	fclose($sh) ;

}

function send_remote_entity($remote_entitys=array()){
	# 別のレプリケートから実ファイルをGETしてクライアントに送信する
	global $conf ;

	# コピー元は分散するようにランダムに決める
	$target_key = array_rand((array)$remote_entitys['exists']) ;
	$ip          = $remote_entitys['exists'][$target_key]['ip'] ;
	$port        = $remote_entitys['exists'][$target_key]['port'] ;
	$server_name = $remote_entitys['exists'][$target_key]['server_name'] ;
	$request_uri = $_SERVER['REQUEST_URI'] ;

	# このホストにレプリケートする場合のローカルパス決定
	$document_root        = $_SERVER['DOCUMENT_ROOT'] ;
	$request_uri_path     = parse_url($_SERVER['REQUEST_URI'],PHP_URL_PATH) ;
	$local_file_path      = sprintf("%s%s",$document_root,$request_uri_path) ;
	$local_file_path_part = sprintf("%s.part",$local_file_path) ;
	$cache_file_path      = sprintf("%s/.cache%s",dirname(__FILE__),$request_uri_path) ;
	$cache_file_path_part = sprintf("%s.part",$cache_file_path) ;

	# ローカルコピーを作るか決定
	if($remote_entitys['number'] < $conf['replication_number']){
		# レプリケート数が少ない場合はローカルにコピーを作る
		$is_local_copy = TRUE ;
		$is_cache_copy = FALSE ;
	}else{
		# レプリケート数が十分でもキャッシュを作る
		$is_local_copy = FALSE ;
		$is_cache_copy = TRUE ;
	}

	$disk_usage_rate = disk_usage_rate(dirname(__FILE__)) ;
	if(file_exists($local_file_path_part) === TRUE){$is_local_copy = FALSE ;} # ただしローカルディスクにpartファイルがある場合はローカルコピーしない
	if(file_exists($cache_file_path_part) === TRUE){$is_cache_copy = FALSE ;} # ただしキャッシュディレクトリにpartファイルがある場合はキャッシュコピーしない
	if($disk_usage_rate >= $conf['disk_usage_limit_of_replication']){$is_local_copy = FALSE ;} # ただしディスク使用量が設定値より多い場合はローカルコピーしない
	if($disk_usage_rate >= $conf['disk_usage_limit_of_cache']){$is_cache_copy = FALSE ;} # ただしディスク使用量が設定値より多い場合はローカルコピーしない
	if($conf['disable_cache'] === TRUE){		$is_cache_copy = FALSE ;} # disable_cache が指定されていたらキャッシュしない
	if($conf['disable_replicate'] === TRUE){	$is_local_copy = FALSE ;} # disable_replicateが指定されていたらコピーしない

	$range = get_request_range() ;
	if($range['start'] >= 1){
		# ただし部分的GETリクエストの場合はローカルコピーもキャッシュコピーもしない
		$is_local_copy = FALSE ;
		$is_cache_copy = FALSE ;
	}

	# リモート接続
	$fp = fsockopen($ip,$port,$errno,$errstr,5) ;
	if($fp === FALSE){
		header("HTTP/1.0 500 Server Error") ;
		echo "500 Server Error\n" ;
		exit ;
	}

	# リクエスト送信
	fwrite($fp,sprintf("GET %s HTTP/1.1\r\n",$request_uri)) ;
	fwrite($fp,sprintf("Host: %s\r\n",$server_name)) ;
	fwrite($fp,sprintf("User-Agent: SHDFS Internal Request\r\n")) ;
	fwrite($fp,sprintf("X-SHDFS-Internal-Request: yes\r\n")) ;
	if($range['start'] >= 1){
		fwrite($fp,sprintf("Range: %s\r\n",get_request_header('Range'))) ;
	}
	fwrite($fp,sprintf("Connection: close\r\n")) ;
	fwrite($fp,sprintf("\r\n")) ;

	# レスポンス取得
	$res = array() ;
	for($i=0;$i<100;$i++){
		# headerだけ先に送信してbodyの直前までポインタを進める
		$line = trim(fgets($fp)) ;
		if(strlen($line) === 0){break ;}
		list($key,$value) = explode(':',$line,2) ;
		$key   = strtolower(trim($key)) ;
		$value = trim($value) ;
		$res['headers'][$key] = $value ;
	}
	$last_modified_int = strtotime($res['headers']['last-modified']) ;

	# レスポンスヘッダ送信
	if($range['start'] >= 1){
		# 部分的GETの場合は 206 Partial Content を返す
		header('HTTP/1.1 206 Partial Content') ;
		header(sprintf("Content-Length: %d",$res['headers']['content-length'])) ;
		header(sprintf("Content-Range: %s",$res['headers']['content-range'])) ;
	}else{
		header('HTTP/1.1 200 OK') ;
		header(sprintf("Content-Length: %d",$res['headers']['content-length'])) ;
	}
	header(sprintf("Date: %s",$res['headers']['date'])) ;
	header(sprintf("Content-Type: %s",$res['headers']['content-type'])) ;
	header(sprintf("Connection: %s",$res['headers']['Connection'])) ;

	if($is_local_copy === TRUE){
		# クライアントにファイルを送信しつつこのホストに複製する
		header('X-SHDFS-send-entity: remote/replicate') ;
		ob_end_flush();
		flush();

		$dir = dirname($local_file_path) ;
		if(file_exists($dir) === FALSE){mkdir($dir,0777,TRUE) ;}

		@touch($local_file_path) ;
		$wh = @fopen($local_file_path_part,'w') ;
		$sh = @fopen('php://output','w') ;
		while(feof($fp) === FALSE){
			$buffer = fread($fp,8192) ;
			fwrite($wh,$buffer) ;
			fwrite($sh,$buffer) ;
		}
		fclose($fp) ;
		fclose($sh) ;
		fclose($wh) ;
		rename($local_file_path_part,$local_file_path) ;
		touch($local_file_path,$last_modified_int) ;
	}elseif($is_cache_copy){
		# クライアントにファイルを送信しつつこのホストにキャッシュする
		header('X-SHDFS-send-entity: remote/cache') ;
		ob_end_flush();
		flush();

		$dir = dirname($cache_file_path) ;
		if(file_exists($dir) === FALSE){mkdir($dir,0777,TRUE) ;}

		@touch($cache_file_path) ;
		$wh = @fopen($cache_file_path_part,'w') ;
		$sh = @fopen('php://output','w') ;
		while(feof($fp) === FALSE){
			$buffer = fread($fp,8192) ;
			fwrite($wh,$buffer) ;
			fwrite($sh,$buffer) ;
		}
		fclose($fp) ;
		fclose($sh) ;
		fclose($wh) ;
		rename($cache_file_path_part,$cache_file_path) ;
		touch($cache_file_path,$last_modified_int) ;

		# 後で削除しやすいようにキャッシュ履歴ファイルを更新
		$cache_log_path = sprintf("%s/.data/cache_log",dirname(__FILE__)) ;
		if(filesize($cache_log_path) >= 1048576){
			# キャッシュログが大きくなっていたらリネームする
			$cache_log_path_backup = sprintf("%s.%s",$cache_log_path,time()) ;
			rename($cache_log_path,$cache_log_path_backup) ;
		}
		$lh = fopen($cache_log_path,'a') ;
		$line = sprintf("%s\t%s\n",time(),$cache_file_path) ;
		fwrite($lh,$line) ;
		fclose($lh) ;
	}else{
		# キャッシュもレプリケートも行わずクライアントにファイルを送信する
		header('X-SHDFS-send-entity: remote/throw') ;
		ob_end_flush();
		flush();

		$sh = @fopen('php://output','w') ;
		while(feof($fp) === FALSE){
			$buffer = fread($fp,8192) ;
			fwrite($sh,$buffer) ;
		}
		fclose($fp) ;
		fclose($sh) ;
	}
}

function shdfs_head_request($ip='',$port=80,$server_name='',$request_uri=''){
	# HEAD リクエスト
	# 戻り値
	# $res['response_code']		HTTP応答コード
	# $res['headers'][$key] 	ヘッダ配列(lower case)

	$fp = @fsockopen($ip,$port,$errno,$errstr,5) ;
	if($fp === FALSE){
		return FALSE ;
	}

	# リクエスト送信
	fwrite($fp,sprintf("HEAD %s HTTP/1.1\r\n",$request_uri)) ;
	fwrite($fp,sprintf("Host: %s\r\n",$server_name)) ;
	fwrite($fp,sprintf("User-Agent: SHDFS Internal Request\r\n")) ;
	fwrite($fp,sprintf("X-SHDFS-Internal-Request: yes\r\n")) ;
	fwrite($fp,sprintf("Connection: close\r\n")) ;
	fwrite($fp,sprintf("\r\n")) ;

	# レスポンス取得
	$res = array() ;
	preg_match('/^HTTP\/\d\.\d (\d{3}) .+/',fgets($fp),$matches) ;
	$res['response_code'] = $matches[1] ;
	while (!feof($fp)) {
		$line = trim(fgets($fp)) ;
		list($key,$value) = explode(':',$line,2) ;
		$key   = strtolower(trim($key)) ;
		$value = trim($value) ;
		if(strlen($key) === 0){continue;}
		$res['headers'][$key] = $value ;
	}
	fclose($fp);
	return $res ;
}

function disk_usage_rate($dir='/'){
	$total = disk_total_space($dir) ;
	$free  = disk_free_space($dir) ;

	$usage_rate = (1 - (float)$free / (float)$total) * 100 ;

	return $usage_rate ;
}

function delete_cache($path='',$lifetime_sec=86400){
	if(strlen($path) === 0){
		return FALSE ;
	}

	if(is_dir($path) === FALSE){
		return FALSE ;
	}

	$path = rtrim($path,'/') ;

	foreach(scandir($path) as $target){
		if($target === '.' or $target === '..'){
			continue ;
		}
		$target_path = sprintf("%s/%s",$path,$target) ;
		if(is_dir($target_path) === TRUE){
			delete_cache($target_path,$lifetime_sec) ;
		}
		if(time() - filemtime($target_path) > $lifetime_sec){
			$result = unlink($target_path) ;
		}
	}
}

function add_stop_server($ip=''){
	if(strlen($ip) === 0){
		return FALSE ;
	}

	$stop_server_file = sprintf("%s/.data/stop.%s",dirname(__FILE__),$ip) ;
	@touch($stop_server_file) ;

	return TRUE ;
}

function exclusion_stop_servers($replication_hosts=array()){
	$new_replication_hosts ;
	foreach((array)$replication_hosts as $value){
		list($ip,$port,$server_name) = explode(':',$value) ;
		$stop_server_file = sprintf("%s/.data/stop.%s",dirname(__FILE__),$ip) ;

		if(file_exists($stop_server_file) === FALSE){
			# stopファイルが無ければ生きているはず
			$new_replication_hosts[] = $value ;
		}else{
			$filemtime = filemtime($stop_server_file) ;
			if(time() - $filetime >= 30){
				# ファイルが古ければ生き返っているかもしれない
				@unlink($stop_server_file) ;
				$new_replication_hosts[] = $value ;
			}else{
				# まだ出来立ての場合は死んでるかもしれない
				# (new_replication_hosts には入れない)
			}
		}
	}
	return $new_replication_hosts ;
}

function logw($message='',$logdir=''){
	if(strlen($logdir) === 0){
		$logdir = sprintf("%s/.data",dirname(__FILE__)) ;
	}

	$logfile = sprintf("%s/%s.%s",$logdir,basename($_SERVER['SCRIPT_NAME']),date('Ymd')) ;
	$message = trim($message) ;
	$logline = sprintf("%s %s\n",date('Y-m-d H:i:s'),$message) ;
	$fh = fopen($logfile,'a') ;
	fwrite($fh,$logline) ;
	fclose($fh) ;
}
?>
