(**************************************************************************)
(*  Mana : A kana(romaji)-kanji conversion engine using ChaSen algorithm.    *)
(*  Copyright (C) 2003, 2004, 2005  Yamagata Yoriyuki                     *)
(*                                                                        *)
(*  This program is free software; you can redistribute it and/or modify  *)
(*  it under the terms of the GNU General Public License as published by  *)
(*  the Free Software Foundation; either version 2 of the License, or (at  *)
(*  your option) any later version.                                       *)
(*                                                                        *)
(*  This program is distributed in the hope that it will be useful, but   *)
(*  WITHOUT ANY WARRANTY; without even the implied warranty of            *)
(*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *)
(*  General Public License for more details.                              *)
(**************************************************************************)

(**************)
(* ܥ *)
(**************)

module Vertex = struct

  type t = {pos : int; state : int}

  (* üpos,state=-1ɽȤˤ*)
  let terminal = {pos = -1; state = -1}

  let create ~pos ~state = {pos=pos; state=state}

  let compare v1 v2 =
    let sgn = v1.pos - v2.pos in
    if sgn <> 0 then sgn else
    let sgn = v1.state - v2.state in
    sgn

  let equal v1 v2 = (v1.pos = v2.pos) && (v1.state = v2.state)

      (* äȤϥåȤ٤ *)
  let hash v = v.pos * 31 + v.state

end

module EdgeLabel = struct 

  (* cost = ǼΤΥ + ξ֤ؤϢܥ *)
  type t = {mrph : Mrph.mrph; pos : int; cost : int}

  let create mrph ~pos ~cost = {mrph=mrph; pos=pos; cost=cost}

  let compare v1 v2 =
    let sgn = v1.pos - v2.pos in
    if sgn <> 0 then sgn else
    let sgn = v1.cost - v2.cost in
    if sgn <> 0 then sgn else
    (* IDEA:mrphIDդ롩 *)
    Mrph.compare_mrph v1.mrph v2.mrph

  let equal v1 v2 = (compare v1 v2 = 0)
	
end

module Edge = struct
  type t = {src : Vertex.t; label : EdgeLabel.t; dst : Vertex.t}

  let create src label dst = {src=src; label=label; dst=dst}
  let label e = e.label
  let src e = e.src
  let dst e = e.dst

end

module IntHashtbl = Hashtbl.Make(struct 
  type t = int
  let equal (x : int) (y : int) = (x = y)
  let hash x = x
end)

module Graph = struct

  type node = {vertex : Vertex.t; 
	       mutable edge_out : Edge.t list;
	       mutable edge_in : Edge.t list}

  type t = {
      mutable positions : node IntHashtbl.t array
    }

  module E = Edge

  let length graph = Array.length graph.positions

  let create len =
    let positions = Array.init len (fun _ -> IntHashtbl.create 16) in
    {positions = positions}

  let add_node graph n =
    IntHashtbl.replace 
      graph.positions.(n.vertex.Vertex.pos) 
      n.vertex.Vertex.state n

  let add_vertex graph v =
    let n = {vertex = v; edge_out = []; edge_in = []} in
    add_node graph n

  let mem_vertex graph v =
    IntHashtbl.mem graph.positions.(v.Vertex.pos) v.Vertex.state

  let find_node graph v =
    IntHashtbl.find 
      graph.positions.(v.Vertex.pos) v.Vertex.state 

  let add_edge graph e =
    let src = 
      try find_node graph e.Edge.src with Not_found ->
	let n = {vertex = e.Edge.src; edge_out = []; edge_in = []} in
	add_node graph n;
	n in
    let dst =
      try find_node graph e.Edge.dst with Not_found ->
	let n = {vertex = e.Edge.dst; edge_out = []; edge_in = []} in
	add_node graph n;
	n in
    src.edge_out <- e :: src.edge_out;
    dst.edge_in <- e :: dst.edge_in

  let succ_e graph v =
    match try Some (find_node graph v) with Not_found -> None with
      None -> []
    | Some n -> n.edge_out

  let pred_e graph v =
    match
      try Some (find_node graph v) with Not_found -> None
    with
      None -> []
    | Some n -> n.edge_in

  let iter_pos proc graph pos =
    IntHashtbl.iter (fun _ n -> proc n.vertex) graph.positions.(pos)

end

module Mark = struct

  module VT= Hashtbl.Make(Vertex)
  type 'a t = 'a VT.t

  let create () = VT.create 0

  let lookup m v =
    try
      Some (VT.find m v)
    with Not_found ->
      None

  let mark m v a = VT.replace m v a

end

module type CostType = sig
  type t
  val weight : EdgeLabel.t -> t
  val zero : t
  val add : t -> t -> t
  val compare : t -> t -> int
end

module Viterbi (Cost : CostType) = struct

  type cost = Cost.t

  type marker = Cost.t * (Edge.t option)

  let mark_pos graph m pos =
    let mark_edge cost e =
      let cost = Cost.add (Cost.weight e.Edge.label) cost in
      match Mark.lookup m e.Edge.dst with
        None -> Mark.mark m e.Edge.dst (cost, Some e)
      | Some ((cost', _)) ->
	  if Cost.compare cost cost' < 0 then
	    Mark.mark m e.Edge.dst (cost, Some e) in
    let mark_node n =
      match Mark.lookup m n.Graph.vertex with
	None -> ()
      | Some ((cost, _)) ->
	  List.iter (mark_edge cost) n.Graph.edge_out in
    IntHashtbl.iter (fun _ n -> mark_node n) graph.Graph.positions.(pos)


  let forward_cost graph start =
    let m = Mark.create () in
    Mark.mark m start (Cost.zero, None);
    for i = 0 to Array.length graph.Graph.positions - 1 do
      mark_pos graph m i
    done;
    m
	    
  let rec search_best_path m dst =
    match Mark.lookup m dst with
      None -> raise Not_found
    | Some ((_, None)) -> []
    | Some ((cost, Some e)) ->
	search_best_path m (Edge.src e) @ [(e, cost)]

  let best_path (graph, m) start pos =
    if pos < 0 || pos >= Graph.length graph then raise Not_found else
    let choose_min_cost _ n a =
      match Mark.lookup m n.Graph.vertex with
	None -> a
      | Some ((c, _)) ->
	  match a with
	    None -> Some (c, n.Graph.vertex)
	  | Some ((c0, _)) ->
	      if Cost.compare c0 c <= 0 then a else Some (c, n.Graph.vertex) in
    match IntHashtbl.fold choose_min_cost graph.Graph.positions.(pos) None with
      None -> raise Not_found
    | Some ((_, v)) -> search_best_path m v
 
end




