require 'amrita/tag'

module Amrita
  Version = '1.82'
  @@accelerator_loaded = false
  def Amrita::accelerator_loaded?
    @@accelerator_loaded
  end
end

class Object
  def amrita_sanitize
    to_s.amrita_sanitize
  end

  def amrita_sanitize_as_attribute
    to_s.amrita_sanitize_as_attribute
  end

  def amrita_sanitize_as_url
    to_s.amrita_sanitize_as_url
  end
end

module Amrita
  module AmritaDictionary
  end
end

class Hash
  include Amrita::AmritaDictionary
end

class String #:nodoc:

  def amrita_sanitize
    Amrita::Sanitizer::sanitize_text(self)
  end

  def amrita_sanitize_as_attribute
    Amrita::Sanitizer::sanitize_attribute_value(self)
  end

  def amrita_sanitize_as_url
    Amrita::Sanitizer::sanitize_url(self)
  end

  # to treat Symbol and String equally
  def id2name
    self
  end

  # clone and freeze a String to share it
  def frozen_copy
    if frozen?
      self
    else
      dup.freeze
    end
  end
end

class Symbol #:nodoc:
  # to treat Symbol and String equally
  def intern
    self
  end

  def to_ruby
    ret = to_s
    if ret =~ /^\w*$/
      ':' + ret
    else
      %Q[#{ret.inspect}.intern]
    end
  end
end

module Amrita 
  module AttrCommon
    def clone
      self.class.new(key, value)
    end

    def dup
      Attr.new(key, value)
    end

    def to_ruby
      "Attr.new(#{key_symbol.to_ruby}, #{value.inspect})"
    end

    def inspect
      "a(#{key_symbol.to_ruby}=>#{value.inspect})"
    end
  end

  # represents a +key+ +value+ pair in HTML Element
  class Attr
    include AttrCommon
    attr_reader :value

    def initialize(key, value=nil)
      @key = key.intern
      self.value = value
    end

    # return +key+ as String    
    def key
      @key.id2name
    end

    # return +key+ as Symbol
    def key_symbol
      @key
    end

    def value=(value)
      case value
      when NilClass, FalseClass, TrueClass
        @value = value
      else
        @value = value.to_s
      end
    end 
  end

  module AttrArrayCommon
    include Enumerable

    def clone
      self.class.new(array.collect { |a| a.clone } )
    end

    def dup
      self.class.new(array.collect { |a|  a.clone } )
    end

    def inspect
      to_ruby
    end

    def clear
      array.clear
    end

    def [](index)
      array[index]
    end

    def []=(index, val)
      array[index] = val
      val
    end

    # iterate on each Attr
    def each(&block)
      array.each(&block)
    end

    def size
      array.size
    end

    def attr_by_key(key)
      each do |a| 
        return a if a.key_symbol == key 
      end
      nil
    end

    def value_by_key(key)
      each do |a| 
        return a.value if a.key_symbol == key 
      end
      nil
    end

    def to_hash
      ret = {}
      each do |a|
        ret[a.key.intern] = a.value
      end
      ret
    end

    def keys
      collect { |a| a.key_symbol }
    end

    def to_ruby
      ret = "a(" + array.collect {|a| "#{a.key_symbol.to_ruby}=>#{a.value.inspect}"}.join(", ") + ")"
      case body
      when nil, Null
      when Node
        ret += body.to_ruby
      else
        ret += body.inspect
      end
      ret
    end
  end

  # Array of Attr s.
  # It can hold +body+ part for using as a model data for Node#expand.
  # Amrita#a() method is a shortcut for Attr.new
  class AttrArray
    include AttrArrayCommon
    # If you call a() { ... }, block yields to +body+
    attr_reader :array, :body

    # internal use only, never touch it!
    # 
    # true if this instance is shared by two or more elements
    attr_accessor :shared

    # Don't use AttrArray.new use a() instead
    def initialize(*attrs, &block)
      @array = []
      @shared = false
      self << attrs

      if block_given?
        @body = yield 
      else
        @body = Null
      end
    end

    # add an Attr
    def add(*x)
      x.each do |a|
        case a
        when Hash
          a.each do |k, v|
            self << Attr.new(k, v)
          end
        when Attr
          @array << a
        when Array, AttrArray
          a.each do |aa|
            self << aa
          end
        else
          raise "can't accept #{a}"
        end
      end
      self
    end

    alias :<< :add 
  end


  # Base module for HTML elements
  module Node
    include Enumerable

    # set the +block+ 's result to +body+
    def init_body(&block)
      if block_given?
        @body = to_node(yield)
      else
        @body = Null
      end
    end

    # a Node has NullNode as body before init_body was called.
    def body
      if defined? @body
        @body
      else
        Null
      end
    end

    # test if it has any children
    def no_child?
      body == Null
    end

    # return an Array of child Node or an empty Array if it does not have a body
    def children
      if no_child?
        []
      else
        [ body ]
      end
    end

    def apply_to_children(&block)
      self
    end

    def delete_if(&block)
      if yield(self)
        Null
      else
        self
      end
    end

    # generate a Node object 
    def to_node(n)
      case n
      when nil, false
        Null
      when Node
        n
      when Array
        case n.size()
        when 0
          Null
        when 1
          to_node(n[0])
        else
          NodeArray.new(*n)
        end
      else
        TextElement.new(n.to_s) 
      end
    end
    module_function :to_node

    def amrita_id 
      nil
    end

    def inspect
      to_ruby
    end

    # Node can be added and they become NodeArray
    def +(node)
      NodeArray.new(self, to_node(node))
    end

    # Copy a Node n times and generate NodeArray
    def *(n)
      raise "can't #{self.class} * #{n}(#{n.class})" unless n.kind_of?(Integer)
      a = (0...n).collect { |i| self }
      NodeArray.new(*a)
    end

    # iterate on self and children
    def each_node(&block)
      c = children # save children before yield
      yield(self)
      c.each do |n|
        n.each_node(&block)
      end
    end

    # iterate on child Elements
    def each_element(&block)
      each_node do |node|
        yield(node) if node.kind_of?(Element)
      end
    end
    alias each each_element

    # iterate on child Elements with id.
    # If recursive == false, don't go to children of an Element with id.
    def each_element_with_id(recursive=false, &block)
      children.each do |node|
        node.each_element_with_id(recursive, &block)
      end
    end

    # test if an Element or children has any id
    def has_id_element?
      each_node do |n|
        next unless n.kind_of?(Element)
        return true if n.amrita_id
        return true if n.get_expandable_attrs.size > 0
      end
      false
    end
  end

  # singleton and immutable object
  class NullNode #:nodoc:
    include Node

    private_class_method :new

    # NullNode::new can not be used. Use this instead.
    def NullNode.instance
      new
    end

    def ==(x)
      x.kind_of?(NullNode)
    end

    # Share the only instance because it's a singleton and immutable object.
    def clone
      self
    end

    def +(node)
      node
    end

    def to_ruby
      "Amrita::Null"
    end

    # NullNode has no children
    def children
      []
    end
  end
  Null = NullNode.instance

  # represents HTML element
  module ElementCommon
    include Node
    
    def tag
      tag_symbol.to_s
    end


    def clone(&block)
      Element.new(self, &block)
    end

    def dup(&block)
      Element.new(self, &block)
    end

    def apply_to_children(&block)
      clone { yield(body) }
    end


    # return id=... attribule value. It can be hide by +hide_amrita_id!
    def amrita_id
      if hide_amrita_id
        nil
      else
        self[:id] or self[:ID]
      end
    end


    def tagclass
      self[:class]
    end

    # set attribule.
    def put_attr(a, &block)
      copy_on_write 
      attrs << a
      init_body(&block) if block_given?
      self
    end

    alias :<< :put_attr

    # test if it has attribule for +key+
    def include_attr?(key)
      attrs.attr_by_key(key.intern) != nil
    end

    def get_expandable_attrs #:nodoc:
      ret = {}
      attrs.each do |attr|
        next if attr.key_symbol == :id
        v = attr.value
        next unless v
        if v[0] == ?@ and v[1] != ?{
          ret[attr.key_symbol] = v[1..-1].intern
        else
          a = []
          v.scan(/([^@]*)(@\{(\w+)\})?/) do |str, dummy, w|
            a << str if str.size > 0 
            a << w.intern if w
          end
          ret[attr.key_symbol] = a if a.size() > 0
        end
      end
      ret
    end

    # return attribule value for +key+
    def [](key)
      attrs.value_by_key(key.intern) 
    end

    # set attribule. delete it if +value+ is +nil+
    def []=(key, value)
      copy_on_write 
      key = key.intern 
      a = attrs.attr_by_key(key) 
      if a
        if value
          a.value = value
        else
          delete_attr!(key)
        end
      else
        put_attr(Attr.new(key,value)) if value
      end
      value
    end

    # delete attribute of +key+
    def delete_attr!(key)
      key = key.intern 
      attrs.array.delete_if { |x| x.key_symbol == key }
    end
    
    def to_ruby
      ret = "e(#{tag_symbol.to_ruby}"
      if attrs.size > 0
        ret << ","
        ret << attrs.collect { |a| "#{a.key_symbol.to_ruby}=>#{a.value.inspect}" }.join(",")
      end
      ret << ") "
      ret << "{ #{body.to_ruby} }" if body and not body.kind_of?(NullNode)
      ret
    end

    def each_element_with_id(recursive=false, &block)
      if amrita_id
        yield(self)
        super if recursive
      else
        super
      end
    end

    # set the +text+ to body of this Element.
    def set_text(text)
      init_body { text }
    end

    def simple_span?
      tag_symbol == :span and attrs.size == 1 and attrs.keys == [:id] and
        (no_child? or (body.kind_of?(TextElement)))
    end

    private
    def copy_on_write
      return unless attrs.shared
      set_attrs(attrs.dup)
    end
  end

  class Element
    include ElementCommon

    attr_accessor :tag_symbol
    private :tag_symbol=
    
    # return attributes as AttrArray
    #
    # CAUTION! never edit result of this method. use []= instead.
    # because it may be shared by other Elements.
    attr_reader :attrs

    # CAUTION! internal use only
    attr_reader :hide_amrita_id

    # return body
    attr_reader :body

    # Don't use Element.new. Use Amrita#e instead.
    def initialize(tagname_or_element, *attrs, &block)
      case tagname_or_element
      when Element
        set_tag(tagname_or_element.tag_symbol)
        @attrs = tagname_or_element.attrs
        @attrs.shared = true
        @hide_amrita_id = tagname_or_element.hide_amrita_id
        if block_given?
          init_body(&block)
        else
          @body = tagname_or_element.body.clone
        end
      when Symbol, String
        set_tag(tagname_or_element)
        @attrs = AttrArray.new(*attrs)
        @hide_amrita_id = false
        if block_given?
          init_body(&block)
        else
          @body = Null
        end
      else
        raise "invalid param for Element #{tagname_or_element}"
      end
    end

    def set_tag(t)
      @tag_symbol = t.to_s.intern 
    end

    # hide hid for internal use (expand).
    def hide_amrita_id!
      @hide_amrita_id = true
    end

    def set_attrs(a)
      @attrs = a
    end

    def delete_if(&block)
      if yield(self)
        Null
      else
        clone { body.delete_if(&block) }
      end
    end

    private :set_attrs
  end

  # immutable object
  class TextElement #:nodoc:
    include Node

    def initialize(text=nil)
      case text
      when nil
        @text = ""
      when String
        @text = text.frozen_copy
      when TextElement
        @text = x.to_s
      else
        @text = value.to_s.freeze 
      end
    end

    def clone
      self # immutable object can be shared always
    end

    def ==(x)
      case x
      when String
        to_s == x
      when TextElement
        to_s == x.to_s
      else
        false
      end
    end

    def to_ruby
      @text.inspect
    end

    def to_s
      @text
    end
  end

  # represents an Array of Node. It is a Node also.
  class NodeArray
    include Node
    attr_reader :array
    def initialize(*elements)
      if elements.size() == 1 and elements[0].kind_of?(NodeArray)
        a = elements[0]
        @array = a.array.collect { |n| n.clone }
      else
        @array = elements.collect do |a|
          #raise "can't be a parent of me!" if a.id == self.id # no recusive check because it costs too much
          to_node(a)
        end
      end
    end


    def size()
      @array.size()
    end

    def [](index)
      @array[index]
    end

    def no_child?
      @array.empty?
    end

    def clone
      NodeArray.new(self)
    end

    def children
      @array
    end

    def +(node)
      ret = clone
      ret << node
      ret
    end

    def <<(node)
      raise "can't be a parent of me!" if node == self
      @array << to_node(node)
      self
    end

    def apply_to_children(&block)
      ret =@array.collect do |n|
        yield(n)
      end
      Node::to_node(ret)
    end

    def delete_if(&block)
      array = @array.collect do |x|
        x.delete_if(&block)
      end.find_all do |x|
        x and x != Null
      end
      case array.size
      when 0
        Null
      when 1
        Node::to_node(array[0])
      else
        Node::to_node(array)
      end
    end

    def to_ruby
      "[ " + @array.collect {|e| e.to_ruby}.join(", ") + " ]"
    end
  end

  # represents a special tag like a comment.
  class SpecialElement #:nodoc:
    attr_reader :tag, :body
    attr_reader :fname, :lno
    include Node
    def initialize(tag, body, fname=nil, lno=nil)
      @tag = tag
      @body = body.dup.freeze
      @fname = fname
      @lno = lno
    end

    def clone
      SpecialElement.new(@tag, @body, @fname, @lno)
    end

    def children
      []
    end

    # end tag
    def etag
      case @tag 
      when '%=', '%'
        '%'
      when '!'
        ''
      when '!--'
        '--'
      when '?'
        '?'
      else
        @tag
      end
    end


    def to_ruby
      %Q(special_tag(#{@tag.dump}, #{@body.dump}) )
    end
  end

  class AmritaPragma < SpecialElement
    attr_reader :scope, :pragma_body
    def initialize(scope, pragma_body, fname=nil, lno=nil)
      scope = "local" unless scope
      @scope, @pragma_body = scope.intern, pragma_body
      super('!--', " amrita_pragma(#{scope}): #{pragma_body}", fname, lno)
    end

    def clone
      AmritaPragma.new(scope, pragma_body, @fname, @lno)
    end

    def to_ruby
      %Q[AmritaPragma.new(#{scope.inspect}, #{pragma_body.inspect})]
    end
  end


  # this is a String that amrita dosen't sanitize.
  # If you don't sanitize or sanitize yourself,
  # pass SanitizedString.new(x) as model data.
  class SanitizedString < String
    def SanitizedString::[](s)
      new(s)
    end

    def amrita_sanitize
      self
    end

    def amrita_sanitize_as_attribute
      self
    end

    def amrita_sanitize_as_url
      self
    end

    def to_s
      self
    end
  end
  
  # This module provide methods for avoid XSS vulnerability
  # taken from IPA home page(Japanese)
  # http://www.ipa.go.jp/security/awareness/vendor/programming/a01_02.html
  module Sanitizer
    NAMECHAR = '[-\w\d\.:]'
    NAME = "([\\w:]#{NAMECHAR}*)"
    NOT_REFERENCE = "(?!#{NAME};|&#\\d+;|&#x[0-9a-fA-F]+;)" # borrowed from rexml
    AMP_WITHOUT_REFRENCE = /&#{NOT_REFERENCE}/
    # escape &<>
    def sanitize_text(text)
      return text unless text =~ /[&<>]/
      s = text.dup
      s.gsub!(AMP_WITHOUT_REFRENCE, '&amp;')
      s.gsub!("<", '&lt;')
      s.gsub!(">", '&gt;')
      s
    end

    # escape &<>"'
    def sanitize_attribute_value(text)
      return text unless text =~ /[&<>"']/ #'"
      s = text.dup
      s.gsub!(AMP_WITHOUT_REFRENCE, '&amp;')
      s.gsub!("<", '&lt;')
      s.gsub!(">", '&gt;')
      s.gsub!('"', '&quot;')
      s.gsub!("'", '&#39;')
      s
    end

    DefaultAllowedScheme = {
      'http' => true,
      'https' => true,
      'ftp' => true,
      'mailto' => true,
    }

    #UrlInvalidChar = Regexp.new(%q|[^;/?:@&=+$,A-Za-z0-9\-_.!~*'()%]|)
    UrlInvalidChar = Regexp.new(%q|[^;/?:@&=+$,A-Za-z0-9\-_.!~*'()%#]|) #'

    # +sanitize_url+ accepts only these characters 
    #     --- http://www.ietf.org/rfc/rfc2396.txt ---
    #     uric = reserved | unreserved | escaped
    #     reserved = ";" | "/" | "?" | ":" | "@" | "&" | "=" | "+" | "$" | ","
    #     unreserved = alphanum | mark
    #     mark = "-" | "_" | "." | "!" | "~" | "*" | "'" | "(" | ")"
    #     escaped = "%" hex hex
    # 
    # +sanitize_url+ accepts only schems specified by +allowd_scheme+
    #
    # The default is http: https: ftp: mailt:

    def  sanitize_url(text, allowd_scheme = DefaultAllowedScheme)
      # return nil if text has characters not allowd for URL

      return nil if text =~ UrlInvalidChar

      # return '' if text has an unknown scheme
      # --- http://www.ietf.org/rfc/rfc2396.txt ---
      # scheme = alpha *( alpha | digit | "+" | "-" | "." )

      if text =~ %r|^([A-Za-z][A-Za-z0-9+\-.]*):| 
        return nil unless allowd_scheme[$1]
      end
  
      # escape HTML
      # special = "&" | "<" | ">" | '"' | "'"
      # But I checked  "<" | ">" | '"' before.
      s = text.dup
      #s.gsub!("&", '&amp;')
      s.gsub!("'", '&#39;')

      s
    end

    module_function :sanitize_text, :sanitize_attribute_value, :sanitize_url
  end

  module ElementProcessorCommon
    def generate_element(tag, attrs=nil, &block)
      case attrs
      when nil
        Element.new(tag, &block) 
      when Hash
        a = attrs.collect do |k, v|
          k = k.intern
          k = :__id__ if k == escaped_id or (amrita_id and k == :id)
          k = :id if k == amrita_id
          Attr.new(k, v)
        end
        Element.new(tag, a, &block) 
      else
        raise "invalid parameter #{attrs}"
      end
    end

    def generate_text(text)
      ret = []
      text.scan(/([^@]*)(@\{(\w+)\})?/) do |str, dummy, w|
        ret << TextElement.new(str) if str.size > 0
        ret << generate_element(:span, :id=>w) if w
      end
      Node::to_node(ret)
    end

    def format_attrs(attrarray, taginfo) #:nodoc:
      return nil if attrarray.size == 0
      array = attrarray.collect do |a|
        flag = taginfo ? taginfo.url_attr2?(a.key_symbol) : false
        format_attr(a, flag)
      end.find_all { |x| x }

      case array.size
      when 0
        nil
      when 1
        array[0]
      else
        array.join(" ")
      end
    end

    def format_attr(a, flag) #:nodoc:
      case a.key_symbol
      when :__id__
        a = Attr.new(:id, a.value) 
      when :id
        return nil if delete_id
        a = Attr.new(amrita_id, a.value) if amrita_id
      end
      attrval = a.value
      case attrval 
      when true
        %Q'#{a.key_symbol}="true"'
      when false
        %Q'#{a.key_symbol}="false"'
      when nil
        %Q'#{a.key_symbol}'
      else
        if flag
          attrval = attrval.amrita_sanitize_as_url
        else
          attrval = attrval.amrita_sanitize_as_attribute
        end
        %Q'#{a.key_symbol}="#{attrval}"'
      end
    end

    def taginfo_of_element(element)
      if tagdict
        tagdict.get_tag_info2(element.tag_symbol)
      else
        nil
      end
    end

    def format_single_tag(e) #:nodoc:
      a = format_attrs(e.attrs, taginfo_of_element(e))
      if a
        %Q[<#{e.tag_symbol} #{a} />]
      else
        if e.tag_symbol == :span
          nil
        else
          %Q[<#{e.tag_symbol} />]
        end
      end
    end

    def format_start_tag(e) #:nodoc:
      a = format_attrs(e.attrs, taginfo_of_element(e))
      if a
        %Q[<#{e.tag_symbol} #{a}>]
      else
        if e.tag_symbol == :span
          nil
        else
          %Q[<#{e.tag_symbol}>]
        end
      end
    end

    def format_end_tag(e) #:nodoc:
      a = format_attrs(e.attrs, taginfo_of_element(e))
      if a == nil and e.tag_symbol == :span
        nil
      else
        %Q[</#{e.tag_symbol}>]
      end
    end

    def format_special(s)
      %Q[<#{s.tag} #{s.body} #{s.etag}>]
    end

    def format_node(n)
      case n
      when Null
        ""
      when NodeArray
        ret = n.children.collect do |n|
          format_node(n)
        end.join("")
        ret 
      when Element
        if n.no_child?
          format_single_tag(n)
        else
          format_start_tag(n).to_s + 
            format_node(n.body).to_s +
            format_end_tag(n).to_s
        end
      when AmritaPragma
        if delete_pragma
          ""
        else
          format_special(n)
        end
      when SpecialElement
        format_special(n)
      else
        n.to_s.amrita_sanitize
      end
    end
  end

  class ElementProcessor
    include ElementProcessorCommon
    attr_reader :tagdict, :escaped_id, :amrita_id, :tagdict
    attr_accessor :delete_id, :delete_id_on_copy, :delete_pragma
    def initialize(tagdict=DefaultHtmlTagInfo)
      @tagdict = tagdict
      @delete_id = @delete_id_on_copy = @delete_pragma = true
      @escaped_id = @amrita_id = nil
    end

    def escaped_id=(x)
      if x
        raise "can't set escpaed_id and amrita_id both at once" if amrita_id
        @escaped_id = x.intern
      else
        @escaped_id = nil
      end
    end

    def amrita_id=(x)
      if x
        raise "can't set escpaed_id and amrita_id both at once" if escaped_id
        raise "can't use '__id__' as amrita_id " if x.intern == :__id__
        @escaped_id = :__id__
        @amrita_id = x.intern
      else
        @amrita_id = nil
      end
    end

    def do_copy(&block)
      save = @delete_id
      @delete_id = true if @delete_id_on_copy
      yield
    ensure
      @delete_id = save
    end
  end

  DefaultElementProcessor = ElementProcessor.new

  module ExpandByMember
    include AmritaDictionary
    def [](mid)
      self.__send__(mid.intern)
    end
  end  

  # generate Element object
  #
  
  # [e(:hr)] <hr>
  # [e(:img src="a.png")]  <img src="a.png">
  # [e(:p) { "text" }]  <p>text</p>
  # [e(:span :class=>"fotter") { "bye" } ] <span class="fotter">bye</span>
  
  def e(tagname, attrs=nil, &block)
    DefaultElementProcessor.generate_element(tagname, attrs, &block)
  end
  alias element e
  module_function :e
  module_function :element

  # generate AttrArray object
  def a(*x, &block)
    AttrArray.new(*x, &block)
  end
  alias attr a
  module_function :a
  module_function :attr

  def text(text) #:nodoc:
    TextElement.new(text)
  end
  module_function :text

  def link(href, klass = nil, &block) #:nodoc:
    element("a",&block) << attr(:href, href) 
  end
  module_function :link

  def special_tag(tag, body, fname=nil, lno=nil) #:nodoc:
    SpecialElement.new(tag, body, fname, lno)
  end
  module_function :special_tag

  def Amrita::append_features(klass) #:nodoc:
    super
    def klass::def_tag(tagname, *attrs_p)
      def_tag2(tagname, tagname, *attrs_p)
    end

    def klass::def_tag2(methodname, tagname, *attrs_p)
      methodname = methodname.id2name 
      tagname = tagname.id2name 
      attrs = attrs_p.collect { |a| a.id2name }

      if attrs.size > 0
        param = attrs.collect { |a| "#{a}=nil" }.join(", ")
        param += ",*args,&block"
        method_body = "  e(#{tagname.to_ruby}, "
        method_body += attrs.collect { |a| "a(#{a}, #{a})"}.join(", ")
        method_body += ", *args, &block)"
      else
        param = "*args, &block"
        method_body = "  e(:#{tagname}, *args, &block) "
      end
      a = "def #{methodname}(#{param}) \n#{method_body}\nend\n"
      #print a
      eval a
    end
  end
end
