require 'ftools'
require 'amrita/amulet'

module Amrita
  def inline_c_code(text)
    first_line, *lines = *text.split("\n")
    first_line.gsub(/\|$/, '')
    "/********************************************************************\n" +
    " *         #{caller[0]}\n" +
    " *         #{caller[1]}\n" +
    " ********************************************************************/\n" +
    lines.collect do |l|
      l.sub(first_line, "")
    end.join("\n")
  end

  class Amcc
    include Amrita

    def initialize(dir, fname, modname, srcdir)
      @dir, @fname, @modname, @srcdir = dir, fname, modname, srcdir
      @templates = {}
      @template_clsnames = {}
      @c_sub_seq = 0
      @subroutines = []
      @ids = {}
      @code_cache = {}
    end

    def add_template(path, template, clsname=nil)
      raise "duplicaste path #{path}" if @templates[path] 
      template.setup
      @templates[path] = template
      @template_clsnames[path] = clsname || sprintf("T%03d", @templates.size)
    end

    def id_name(name)
      name = name.intern
      @ids[name] = true
      "id_#{name}"
    end

    def c_sub(subname=nil)
      subname = sprintf("sub%04d", @csseq) unless subname
      @csseq = 0 unless defined? @csseq
      @csubs = [] unless defined? @csubs
      @csseq = @csseq + 1
      @csubs << yield(subname)
    end

    def code_cache(mod, func_key, &block)
      h = @code_cache[mod] || {}
      func_name = h[func_key] 
      unless func_name
        func_name = yield
        h[func_key] = func_name
        @code_cache[mod] = h
      end
      func_name
    end

    def execute_func_name(path, amulet_id=nil)
      clsname = @template_clsnames[path]
      ret = clsname.downcase + "_excecute"
      ret += "_#{amulet_id}" if amulet_id
      ret
    end

    def make_ext
      Dir::mkdir(@dir) unless FileTest::directory? @dir
      File::cp File::join(@srcdir, "amrita_accel.h"), @dir
      File::cp File::join(@srcdir, "amrita_common.c"), @dir
      make_extconf(File::join(@dir, "extconf.rb"), @fname)
      make_ext_src(File::join(@dir, @fname + ".c"), @fname)
    end

    def make_extconf(path, fname)
      File::open(path, "w") do |f|
        f.print <<-END
          require "mkmf"
          create_makefile("#{fname}")
        END
      end
    end

    def make_ext_src(path, fname)
      tmpl_src = @templates.collect do |k, tmpl|
                  tmpl.make_ext_src(self, k)
                  end
      src = ext_src_init_func
      File::open(path, "w") do |f|
        f.puts '#include "amrita_common.c"'
        f.puts ext_src_defs
        f.puts template_defs
        f.puts id_defs
        f.puts @csubs.join("\n")
        f.puts src
      end
    end

    def ext_src_defs
      inline_c_code <<-END
      |
      /********************************************************************/
      static VALUE mAmirta ;
      static VALUE mAmritaAccelerator ; 
      static VALUE cCCompiledTemplate ;
      static VALUE m#{@modname} ;
      /********************************************************************/
      END
    end

    def template_defs
      @templates.collect do |k, tmpl|
        %Q[static VALUE c#{@template_clsnames[k]} ;]
      end.join("\n")
    end

    def id_defs
      @ids.keys.collect do |name|
        "static VALUE id_#{name} ;"
      end.join("\n")
    end

    def id_init
      @ids.keys.collect do |name|
        %Q[    id_#{name} = rb_intern("#{name}");]
      end.join("\n")
    end

    def ext_src_init_func
      id_templates = id_name("Templates")
      inline_c_code <<-END
      |
      /********************************************************************/

      void 
      Init_#{@fname}()
      {
          VALUE templates = rb_hash_new() ;
          VALUE tmp ;
        
          /* puts("init_#{@fname}") ; */
          rb_require("amrita/accel") ;
      #{id_init}
          mAmrita = rb_define_module("Amrita");
          mAmritaAccelerator = rb_define_module("AmritaAccelerator");
          init_static_constants() ;

          cCCompiledTemplate = rb_const_get(mAmrita, rb_intern("CCompiledTemplate")) ;
          m#{@modname} = rb_define_module("#{@modname}");
          rb_const_set(m#{@modname}, #{id_templates}, templates) ;

      #{template_init}    
      }
      /********************************************************************/
      END
    end

    def template_init
      @templates.collect do |k, tmpl|
        cls = @template_clsnames[k]
        inline_c_code <<-END
        |
            c#{cls} = rb_define_class_under(m#{@modname}, "#{cls}", cCCompiledTemplate) ;
            tmp = rb_funcall(c#{cls}, id_new, 0) ;
            rb_funcall(templates , rb_intern("[]="), 2, rb_str_new2("#{k}"),  tmp) ; 
            #{tmpl.define_methods(self, cls, k)}
        END
      end.join("\n")
    end
  end

  class DelayEval
    def c_code(amcc, iset)
      evaluate unless @body
      @body.c_code(amcc, iset)
    end
  end

  class Template
    def make_ext_src(amcc, k)
      setup_compiler
      #compiler.default_pragma.delete_candidate(Attr)
      #compiler.default_pragma.delete_candidate(AttrArray)
      #compiler.default_pragma.delete_candidate(Node)
      compiler.default_pragma.delete_candidate(Amulet)
      setup
      func_name = amcc.execute_func_name(k)
      iset.c_code(amcc, func_name)
      return unless @amulet_seeds
      @amulet_seeds.each do |amid, v|
        c_code_for_amulet(amcc, amid, v, k)
      end
      amcc.c_sub(amcc.execute_func_name(k, "_amulet_specs")) do |funcname|
        make_amulet_specs_code(funcname)
      end
    end

    def make_amulet_specs_code(func_name)
      inline_c_code(<<-END)
      |
      static VALUE
      #{func_name}(self)
      VALUE self ;
      {
          VALUE ret = rb_hash_new() ;
          VALUE tmp, cls, element_code ;
          #{make_amulet_spec_code_body}
          return ret;
      }
      END
    end
 
    def make_amulet_spec_code_body
      @amulet_seeds.collect do |k, v|
        cls = v.amulet_class.to_s.gsub("Amrita::", "")
        element_code = v.element.to_ruby.gsub('"', '\"')
        inline_c_code(<<-END)
        |
          tmp = rb_hash_new() ;
          cls = rb_const_get(mAmrita, rb_intern("#{cls}")) ;
          rb_funcall(tmp, rb_intern("[]="), 2, ID2SYM(rb_intern("amulet_class")), cls) ;
          element_code = rb_str_new2("#{element_code}") ; 
          rb_funcall(tmp, rb_intern("[]="), 2, ID2SYM(rb_intern("element_code")), element_code) ;
          rb_funcall(ret, rb_intern("[]="), 2, ID2SYM(rb_intern("#{k}")), tmp) ;
        END
      end.join("\n")
    end

    def c_code_for_amulet(amcc, k, seed, path)
      func_name = amcc.execute_func_name(path, k)
      seed.iset.c_code(amcc, func_name)
    end
 
    def define_methods(amcc, cls, k)
      ret = %Q[rb_define_method(c#{cls}, "execute", #{amcc.execute_func_name(k)}, 4); \n]
      if defined? @amulet_seeds
        @amulet_seeds.each do |amid, v|
          ret += %Q[rb_define_method(c#{cls}, "execute_amulet_#{amid}", #{amcc.execute_func_name(k, amid)}, 4); \n]
        end
        ret += %Q[rb_define_method(c#{cls}, "amulet_specs", #{amcc.execute_func_name(k, "_amulet_specs")}, 0); \n]
      end
      ret
    end

  end

  class InstructionSet
    include Amrita
    def c_code(amcc, func_name)
      amcc.c_sub(func_name) do |func_name|
        inline_c_code(<<-END)
        |
        static VALUE
        #{func_name}(self, reg, stack, out, vm)
        VALUE self, reg, stack, out, vm ;
        {
          VALUE tmp, tmp2 ;
          VALUE ep = rb_funcall(vm, rb_intern("ep"), 0) ;
          AMRITA_OUTBUF_DEFINE ;

          AMRITA_OUTBUF_INIT(out) ;
          #{@entry_point.c_code(amcc, self)}
          AMRITA_OUTBUF_CLOSE ;
        }
        /********************************************************************/
        END
      end
    end
  end

  module ByteCodeCommon

    module NullInstruction
      def c_code(amcc, iset)
        ""
      end
    end

    module GetDataByKey
      def c_code(amcc, iset)
        inline_c_code <<-END
        |
        AMRITA_STACK_PUSH(stack, reg) ;
        {
			static ID id_index = 0 ;
			static VALUE key_sym = 0 ;
			static ID key_id = 0 ;
			if (key_id == 0) {
				key_id = rb_intern("#{key}") ;
				key_sym = ID2SYM(key_id) ;
			}
			if (id_index == 0) id_index = rb_intern("[]") ;
			switch(TYPE(reg)) {
			case T_HASH:
				if (st_lookup(RHASH(reg)->tbl, key_sym, &reg)) {
				} else {
					reg = Qnil ;
				}
				break ;
			default:
				reg = rb_funcall(reg, id_index, 1, key_sym) ;
			}
		}
        END
      end
    end

    module ExtractData
      def c_code(amcc, iset)
        subname = amcc.code_cache(ExtractData, :element_copy) do
          new_subname = nil
          amcc.c_sub do |new_subname|
            inline_c_code <<-END
            |
            static VALUE #{new_subname}(key_and_value, hashes)
            VALUE key_and_value, hashes ;
            {
              VALUE tmp ;
              static ID id_index = 0 ;
              static ID id_index_assign = 0 ;
              VALUE key = RARRAY(key_and_value)->ptr[0] ;
              VALUE value = RARRAY(key_and_value)->ptr[1] ;
              VALUE new_hash = RARRAY(hashes)->ptr[0] ;
              VALUE src_hash = RARRAY(hashes)->ptr[1] ;
              if (id_index == 0) id_index = rb_intern("[]") ;
              if (id_index_assign == 0) id_index_assign = rb_intern("[]=") ;
              tmp = rb_funcall(src_hash, id_index, 1, value) ;
              rb_funcall(new_hash, id_index_assign, 2, key, tmp) ;
              /* to be continued */
            }
            END
          end
          new_subname
        end

        inline_c_code <<-END
        |
        {
            /* ExtractData */
            static VALUE ex_hash = 0 ;
            if (ex_hash == 0) ex_hash = rb_eval_string("#{keys.inspect}") ;
            tmp = reg ;
            AMRITA_STACK_PUSH(stack, reg) ;
            reg = rb_hash_new() ;
            rb_iterate(rb_each, ex_hash, #{subname}, rb_ary_new3(2, reg, tmp)) ;
        }
        END
      end
    end

    module PopData
      def c_code(amcc, iset)
        "reg = AMRITA_STACK_POP(stack) ;"
      end
    end

    module SwapData
      def c_code(amcc, iset)
        inline_c_code <<-END
        |
        tmp = RARRAY(stack)->ptr[RARRAY(stack)->len-1] ;
        RARRAY(stack)->ptr[RARRAY(stack)->len-1] = reg ;
        reg  = tmp ;
        END
      end
    end

    module PrintBase
      def c_code(amcc, iset)
        return "" unless text
        inline_c_code <<-END
        |
        /* output(out, rb_str_new2("#{text.gsub('"', '\"')}"), 0) ; */
        AMRITA_OUTBUF_PUTCONST("#{text.gsub('"', '\"')}") ;
        END
      end  #'
    end

    module PushElement
      def c_code(amcc, iset)
        code_for_element = ' ' + element.to_ruby
        code_for_element.gsub!(' e\(', 'Amrita::DefaultElementProcessor::generate_element(')
        code_for_element.gsub!("\"", "\\\"")
        inline_c_code <<-END
        |
        AMRITA_STACK_PUSH(stack, reg) ;
        reg = rb_eval_string("#{code_for_element}") ;
        END
      end
    end

    module PrintRegister
      def c_code(amcc, iset)
        inline_c_code <<-END
        |
        switch(TYPE(reg)) {
        case T_STRING:
          AMRITA_OUTBUF_PUTSTR(RSTRING(reg)->ptr, RSTRING(reg)->len) ;
          break ;
        default:
          if (rb_obj_is_kind_of(reg, mNode)) {
              tmp = rb_funcall(ep, rb_intern("format_node"), 1, reg) ;
              /* output(out, tmp, 0) ; */
              AMRITA_OUTBUF_PUTOBJ(tmp, 0) 
          } else {
              /* output(out, reg, id_amrita_sanitize) ; */
              AMRITA_OUTBUF_PUTOBJ(reg, id_amrita_sanitize) 
          }
        }
        END
      end
    end

    module MergeElement
      def element_init_code(varname)
        ret = %Q[#{varname} = rb_funcall(cElement, id_new, 1, ID2SYM(rb_intern("#{element.tag}"))) ;]
        ret += "\n"
        element.attrs.each do |a|
          ret += %Q[tmp = rb_funcall(cAttr, id_new, 2, ID2SYM(rb_intern("#{a.key}")), rb_str_new2("#{a.value}")) ; ]
          ret += "\n"
          ret += %Q[rb_funcall(#{varname}, rb_intern("<<"), 1, tmp) ;]
          ret += "\n"
        end
        ret
      end

      def c_code(amcc, iset)
        subname = amcc.code_cache(MergeElement, :element_sub) do
          new_subname = nil
          amcc.c_sub do |new_subname|
            inline_c_code <<-END
            |
            static VALUE
            #{new_subname}(a, e)
            VALUE e, a ;
            {
              VALUE tmp ;        
              VALUE k ;
              VALUE v ;
              static ID id_index  = 0 ;
              if (id_index == 0) id_index = rb_intern("[]=") ;
              k = RARRAY(a)->ptr[0] ;
              v = RARRAY(a)->ptr[1] ;
              rb_funcall(e, id_index, 2, k, v) ;
              return e ;
            }
            END
          end
          new_subname
        end

        inline_c_code <<-END
        |
        {
          static VALUE e = 0 ;
          if (e == 0) {
              #{element_init_code('e')}
          }
          tmp = rb_funcall(e, id_clone, 0) ;
          rb_iterate(rb_each, reg, #{subname}, tmp) ;
          reg = tmp ;
        }
        END
      end
    end

    module GetDataFromAttrArray
      def c_code(amcc, iset)
        inline_c_code <<-END
        |
        tmp = rb_funcall(reg, rb_intern("body"), 0) ;
        AMRITA_STACK_PUSH(stack, tmp) ;
        reg = rb_funcall(reg, rb_intern("to_hash"), 0) ;
        END
      end
    end

    module SimpleSpan
      def c_code(amcc, iset)
        subname = amcc.code_cache(SimpleSpan, :simplespan_support) do
          new_subname = nil
          amcc.c_sub do |new_subname|
            id_simplespan_execute_func = amcc.id_name("simplespan_execute_func")
            inline_c_code <<-END
            |
            static VALUE #{new_subname}(reg, stack, out, vm, text)
            VALUE reg, stack, out, vm, text ;
            {
                int i ;
                switch(TYPE(reg)) {
                case T_NIL:
                case T_FALSE:
                    break ;
                case T_TRUE:
                    AMRITA_OUTBUF_PUTOBJ(text, 0) ;
                    break ;
                case T_STRING:
                    AMRITA_OUTBUF_PUTOBJ(reg, 0) ;
                    break ;
                case T_ARRAY:
                    for(i = 0 ; i < RARRAY(reg)->len ; i++) {
                        #{new_subname}((RARRAY(reg)->ptr[i]), stack, out, vm, text) ;
                    }
                    break ;
                default:
                    rb_funcall(mAmritaAccelerator, #{id_simplespan_execute_func}, 5, reg, stack, out, vm, text) ;
                    break ;
                }
                return reg;
            }
            END
          end
          new_subname
        end
        t = case text
            when String; text.inspect ;
            when NullNode; '""' ;
            else text.to_s.inspect ;
            end
        inline_c_code <<-END
        |
        {
          static ID id_index = 0 ;
          static VALUE key_sym = 0 ;
          static ID key_id = 0 ;
          static VALUE text = 0 ;

          if (text == 0) text = rb_str_new2(#{t}) ;
          AMRITA_STACK_PUSH(stack, reg) ;

          if (key_id == 0) {
              key_id = rb_intern("#{tag}") ;
              key_sym = ID2SYM(key_id) ;
          }
          if (id_index == 0) id_index = rb_intern("[]") ;
          switch(TYPE(reg)) {
          case T_HASH:
              if (st_lookup(RHASH(reg)->tbl, key_sym, &reg)) {
              } else {
                  reg = Qnil ;
              }
              break ;
          default:
	      reg = rb_funcall(reg, id_index, 1, key_sym) ;
	  }
          #{subname}(reg, stack, out, vm, text) ;
          reg = AMRITA_STACK_POP(stack) ;      
        }
        END
      end
    end

    module GetDataFromAmulet
      def c_code(amcc, iset)
        inline_c_code <<-END
        |
        AMRITA_STACK_PUSH(stack, reg) ;
        AMRITA_STACK_PUSH(stack, rb_funcall(reg, rb_intern("[]"), 1, ID2SYM(rb_intern("amulet_data")))) ;
        reg = rb_funcall(reg, rb_intern("[]"), 1, ID2SYM(rb_intern("amulet_iset"))) ;
        tmp = rb_funcall(vm, rb_intern("iset"), 0) ;
        tmp = rb_funcall(tmp, rb_intern("opt"), 0) ;
        rb_funcall(reg, rb_intern("prepare"), 2, vm, tmp) ;
        END
      end
    end

    module Sequence
      def c_code(amcc, iset)
        ret = self.instructions.collect do |i|
          i.c_code(amcc, iset)
        end.join("\n")
        ret
      end
    end

    module PrintDynamicElement
      def c_code(amcc, iset)
        inline_c_code <<-END
        |
        {
          VALUE e = reg ;      
          /* output(out, rb_funcall(ep, rb_intern("format_start_tag"), 1, e), 0) ; */
          AMRITA_OUTBUF_PUTOBJ(rb_funcall(ep, rb_intern("format_start_tag"), 1, e), 0) ;
          tmp = RARRAY(stack)->ptr[RARRAY(stack)->len-1] ;
          RARRAY(stack)->ptr[RARRAY(stack)->len-1] = reg ;
          reg = tmp ;
          #{body.c_code(amcc, iset)}
          tmp = AMRITA_STACK_POP(stack) ;      
          /* output(out, rb_funcall(ep, rb_intern("format_end_tag"), 1, tmp), 0) ; */
          AMRITA_OUTBUF_PUTOBJ(rb_funcall(ep, rb_intern("format_end_tag"), 1, tmp), 0) ;
        }
        END
      end
    end

    module ExecuteRegister
      def c_code(amcc, iset)
        inline_c_code <<-END
        |
        tmp = AMRITA_STACK_POP(stack) ;
        reg = rb_funcall(reg, id_execute, 4, tmp, stack, out, vm) ;
        END
      end
    end

    module Loop
      def c_code(amcc, iset)
        inline_c_code <<-END
        |
        
        if (TYPE(reg) != T_ARRAY)  tmp = rb_funcall(reg, id_to_a, 0) ;
        else                       tmp = reg ;
        {
          int i ;
          for(i = 0 ; i < RARRAY(tmp)->len ; i++) {
              reg = RARRAY(tmp)->ptr[i] ;
              {
                VALUE tmp ;
                #{body.c_code(amcc, iset)}
              }
          }
        }
        END
      end
    end

    module SelectByType
      def typeselect_type(t)
        if t ==  Amrita::Attr
          %Q[cAttr]
        elsif t ==  Amrita::AttrArray
          %Q[cAttrArray]
        elsif t == Amrita::Node
          %Q[mNode]
        elsif t == Amrita::Amulet
          %Q[mAmulet]
        elsif t == Amrita::AmritaDictionary
          %Q[mAmritaDictionary]
        elsif t == String
          %Q[rb_cString]
        elsif t == Enumerable
          %Q[rb_mEnumerable]
        elsif t == Proc
          %Q[rb_cProc]
        elsif t == Amrita::Node
          %Q[mNode]
        else
          raise "not implemented type #{t}"
        end
      end

      def typeselect_cond(types)
        types.collect do |t|
          if t == TrueClass
            %Q[reg == Qtrue]
          elsif t == NilClass
            %Q[reg == Qnil]
          elsif t == FalseClass
            %Q[reg == Qfalse]
          elsif t == Amrita::NullNode
            %Q[reg == cNull]
          else
            %Q[(rb_obj_is_kind_of(reg, #{typeselect_type(t)}))]
          end
        end.join(" || ")
      end

      def typeselect_code
        i = 0
        conds.collect do |types, instruction|
          i = i + 1
          "if (#{typeselect_cond(types)}) return #{i} ;"
        end.join("\n")
      end

      def body_code1(amcc, iset, no, instruction)
        %Q[case #{no}: #{instruction.c_code(amcc, iset)} break ; \n]
      end

      def body_code(amcc, iset)
        i = 0
        conds.collect do |types, instruction|
          i =  i + 1
          body_code1(amcc, iset, i, instruction)
        end.join("\n")
      end

      def else_code(amcc, iset)
        else_.c_code(amcc, iset)
      end

      def typeno_of(klass)
        i = 0
        conds.each do |types, instruction|
          i =  i + 1
          return i if types.include?(klass)
        end
        -1
      end
      
      def c_code(amcc, iset)
        types_array = conds.collect do |types, instruction|
          types
        end
        typeselect = amcc.code_cache(SelectByType, types_array) do
          new_func_name = nil
          amcc.c_sub do |new_func_name|
            inline_c_code <<-END
            |
            static int #{new_func_name}(reg)
            VALUE reg ;
            {
              #{typeselect_code}    
            }
            END
          end
          new_func_name
        end
        inline_c_code <<-END
        |
        {
            int typeno = -1 ;
            /* DEBUGOBJ(reg) */
            switch(TYPE(reg)) {
            case T_STRING: typeno = #{typeno_of(String)} ; break ;
            case T_TRUE:   typeno = #{typeno_of(TrueClass)} ; break ;
            case T_ARRAY:  typeno = #{typeno_of(Enumerable)} ; break ;
            case T_HASH:   typeno = #{typeno_of(AmritaDictionary)} ; break ;
            case T_FALSE:
            case T_NIL:
                typeno = #{typeno_of(NilClass)} ;
                break ;
            }
            if (typeno == -1) {
                typeno = #{typeselect}(reg) ;
            }
            /* DEBUGINT(typeno) */
            switch(typeno) {
            #{body_code(amcc, iset)}
            default:
            #{else_code(amcc, iset)}
            }
        }
        END
      end
    end
  end

end

__END__
