require 'amrita/node'
require 'amrita/vm'
require 'amrita/amulet'

module Amrita
  module Pragma

    def create_pragma(x)
      case x
      when Class
        create_pragma(x.new)
      when String
        create_pragma(eval(x))
      when Pragma::Pragma
        x
      else
        raise "not pragma class #{x.class}"
      end
    end

    class Pragma
      include Amrita::Pragma
      def Pragma.inherited(subclass)
        def subclass.[] ()
          new
        end
      end

      attr_accessor :compiler
      def initialize
      end

      def inspect
        "#{self.class}[]"
      end
    end

    
    class ScalarData < Pragma
      def convert_element_to_bytecode(element)
        code_for_scalar(element)
      end

      def accept_classes
        [ String, Integer ]
      end
    end

    class ConditionalData < Pragma
      def convert_element_to_bytecode(element)
        code_for_conditional(element)
      end

      def accept_classes
        [ TrueClass, FalseClass, NilClass ]
      end
    end

    class HashData < Pragma
      def convert_element_to_bytecode(element)
        code_for_hash(element)
      end

      def accept_classes
        [ AmritaDictionary ]
      end
    end

    class ProcData < Pragma
      def convert_element_to_bytecode(element)
        code_for_proc(element)
      end

      def accept_classes
        [ Proc ]
      end
    end
    
    class ParentPragma < Pragma
      
      def ParentPragma.inherited(subclass)
        extend Amrita::Pragma
        def subclass.[] (*a)
          new(*a)
        end
      end

      attr_reader :children

      def initialize(*children)
        @children = children.collect do |cp|
          create_pragma(cp)
        end
      end

      def compiler=(comp)
        @compiler = comp
        children.each do |c|
          c.compiler = comp
        end
      end

      def inspect
        "#{self.class}[#{children.collect { |c| c.inspect }.join(',')}]"
      end
    end

    class EnumerableData < ParentPragma
      def convert_element_to_bytecode(element)
        code_for_enumerable(element, children)
      end

      def accept_classes
        [ Enumerable ]
      end
    end

    class AttrData < ParentPragma
      def convert_element_to_bytecode(element)
        code_for_attrarray(element, children)
      end

      def accept_classes
        [ AttrArray ]
      end
    end



    class UncertainData < Pragma
      attr_reader :candidates
      def initialize(*candidates)
        @candidates = candidates
      end

      def clone_without(*types)
        c = @candidates.collect do |t, proc_|
          [t.clone, proc_]
        end
        ret = UncertainData.new(*c)
        types.each do |typ|
          ret.delete_candidate(typ)
        end
        ret
      end

      def convert_element_to_bytecode(element)
        t = ByteCode::SelectByType[]
        t.else_ = delay do
          ByteCode::NullInstruction[]
        end

        @candidates.each do |types, proc_|
          if types.size > 0
            t[*types] = delay do
              proc_.call(element)
            end
          else
            t.else_ = delay do 
              proc_.call(element)
            end
          end
        end
        t
      end

      def add_candidate(*types, &block)
        @candidates << [types, block]
      end

      def delete_candidate(typ)
        new_candidates = @candidates.collect do |types, block|
          if types.size > 0
            types.delete(typ)
            if types.size > 0
              [types, block]
            else
              [nil, nil]
            end
          else
            [types, block]
          end
        end.delete_if do |types, block|
          types == nil
        end
        @candidates = new_candidates
      end

      def accept_classes
        ret = @candidates.collect do |types, proc_|
          types
        end
        ret.flatten.compact
      end
    end
    
      # method for derived classes
    def code_for_hash(element)
      exattrs = element.get_expandable_attrs
      if exattrs.size > 0
        ByteCode::Sequence[
          ByteCode::ExtractData[exattrs],
          ByteCode::MergeElement[element],
          ByteCode::PrintDynamicElement[
            element.body.convert_to_bytecode(compiler),
          ]
        ]
      else
        ByteCode::Sequence[
          ByteCode::PrintStaticText[compiler.ep.format_start_tag(element)],
          element.body.convert_to_bytecode(compiler),
          ByteCode::PrintStaticText[compiler.ep.format_end_tag(element)]
        ]
      end
    end

    def code_for_enumerable(element, children=[])
      ee = element.clone
      ee.hide_amrita_id!
      case children.size
      when 0
        pragma = compiler.default_pragma.clone_without(Enumerable)
        ByteCode::Loop[ pragma.convert_element_to_bytecode(ee) ]
      when 1
        ByteCode::Loop[ 
          children[0].convert_element_to_bytecode(ee)
        ]
      else
        t = ByteCode::SelectByType[]
        children.each do |p|
          t[*p.accept_classes] = t.convert_element_to_bytecode(ee)
        end
        ByteCode::Loop[ t ]
      end        
    end

    def code_for_attrarray(element, children=[])
      ByteCode::Sequence[
        ByteCode::GetDataFromAttrArray[],
        ByteCode::MergeElement[element],
        ByteCode::PrintDynamicElement[
          ByteCode::Sequence[
            code_for_attrarraybody(element, children),
          ]
        ]
      ]
    end

    def code_for_attrarraybody(element, children=[])
      t = ByteCode::SelectByType[]
      t[NilClass, FalseClass, Amrita::NullNode] = element.body.convert_to_bytecode(compiler)
      t.else_ = ByteCode::PrintRegister[]
      t
    end

    def code_for_proc(element)
      e = element.clone 
      e[:id] = nil
      ByteCode::Sequence[
        ByteCode::PushElement[e],
        ByteCode::SwapData[],
        ByteCode::ExecuteRegister[],
        ByteCode::PrintRegister[],
      ]
    end

    def code_for_amulet(element)
      ByteCode::Sequence[
        ByteCode::PrintStaticText[compiler.ep.format_start_tag(element)],
        ByteCode::GetDataFromAmulet[],
        ByteCode::ExecuteRegister[],
        ByteCode::PopData[],
        ByteCode::PrintStaticText[compiler.ep.format_end_tag(element)]
      ]        
    end

    def code_for_scalar(element)
      ByteCode::Sequence[
        ByteCode::PrintStaticText[compiler.ep.format_start_tag(element)],
        ByteCode::PrintRegister[],
        ByteCode::PrintStaticText[compiler.ep.format_end_tag(element)]
      ]
    end

    class AllowNil < UncertainData
      def AllowNil.[](*args)
        new(*args)
      end

      attr_reader :children


      def initialize(child)
        super()
        @child = create_pragma(child)

        add_candidate(*@child.accept_classes) do |element|
          @child.convert_element_to_bytecode(element)
        end
        add_candidate(NilClass, FalseClass) do |element|
          ByteCode::NullInstruction[]
        end
      end

      def compiler=(comp)
        super
        @compiler = comp
        @child.compiler = comp
      end
    end

    def delay(&block)
      DelayEval.new(&block)
    end
  end

  class DelayEval
    def DelayEval.[](*args)
      new(*args)
    end

    def initialize(status=:delayed, body=nil, src=nil, &block)
      @status = status # :delayed/:evaluated/:prepared/:compiled
      @body = body
      @src = src
      @body_proc = block
      @partial_compile = false
      @prepare_param = nil
    end

    def inspect
      %Q[DelayEval(#{@status}) [#{@body.inspect}]]
    end

    def prepare(vm, opt, iset)
      @prepare_param = [vm, opt, iset]
      @partial_compile = opt[:partial_compile]
      evaluate unless opt[:lazy_evaluation]
      if @status == :evaluated
        @body.prepare(vm, opt, iset) 
        @status = :prepared
      end
    end

    def evaluate
      if @status == :delayed
        @body = @body_proc.call 
        if @prepare_param         
          @body.prepare(*@prepare_param) 
          @status = :prepared
        else
          @status = :evaluated
        end
      end
    end

    def execute(register, stack, out, vm)
      evaluate 
      raise "not prepared but status is #{@status}" unless @status == :prepared
      @body.execute(register, stack, out, vm)
    end

    def optimize
      evaluate 
      raise "not prepared" unless @status == :prepared
      @body.optimize
    end

    def ruby_code(vm, iset=nil)
      evaluate 
      if @partial_compile
        @compile_execute = self.method(:compile_execute_do_compile)
        me = iset.new_constant(self)
        %Q[#{me}.compile_execute(register, stack, out, vm)]
      else
        @body.ruby_code(vm, iset)
      end
    end

    def compile_execute(register, stack, out, vm)
      @compile_execute.call(register, stack, out, vm)
    end

    def compile_execute_do_compile(register, stack, out, vm)
      iset = @prepare_param[2]
      @compile_execute = iset.ruby_sub do |method_name|
        @src = @body.ruby_code(vm, iset) unless @src
        @src
      end
      @status = :compiled
      @body = nil
      @compile_execute.call(register, stack, out, vm)
     end

    def dump(out, level=0)
      out << "***:" << ("  " * level) << %Q[DelayEval(#{@status})] << "\n"
      case @status
      when :prepared
        @body.dump(out, level+1) 
      end
    end        

    def _dump(level)
      evaluate 
      Marshal::dump([@status, @body, @src], level)
    end

    def DelayEval._load(s)
      status, body, src = *Marshal::load(s)

      case status
      when :evaluated, :prepared
        status = :evaluated
      when :compiled
      when :delayed
        raise "can't happen"
      else
        raise "can't happen"
      end
      ret = DelayEval.new(status, body, src)
      if status == :compiled
        ret.instance_eval {
          @compile_execute = self.method(:compile_execute_do_compile)
        }
      end
      ret
    end
  end

  class Compiler
    include Pragma
    include ByteCode
    attr_reader :ep, :default_pragma
    attr_accessor :use_simplespan

    def initialize(ep = DefaultElementProcessor)
      @ep = ep
      @current_pragma = nil
      @use_simplespan = true
      make_defaults
    end

    def compiler
      self
    end

    def make_defaults
      @default_pragma = Pragma::UncertainData.new
      @default_pragma.compiler = self
      @default_pragma.add_candidate(AttrArray) do |element|
        code_for_attrarray(element)
      end
      @default_pragma.add_candidate(Amulet) do |element|
        code_for_amulet(element)
      end
      @default_pragma.add_candidate(AmritaDictionary) do |element|
        code_for_hash(element)
      end
      @default_pragma.add_candidate(String, Node) do |element|
        code_for_scalar(element)
      end
      @default_pragma.add_candidate(NilClass, FalseClass) do |element|
        NullInstruction[]
      end
      @default_pragma.add_candidate(TrueClass) do |element|
        PrintStaticNode[element]
      end
      @default_pragma.add_candidate(Enumerable) do |element|
        code_for_enumerable(element)
      end
      @default_pragma.add_candidate(Proc) do |element|
        code_for_proc(element)
      end
      @default_pragma.add_candidate do |element|
        code_for_scalar(element)
      end

    end

    def compile(node)
      InstructionSet.new(node.convert_to_bytecode(self))
    end

    def compile_element(element, pragma=default_pragma)
      bytecode = pragma.convert_element_to_bytecode(element)
      InstructionSet.new(bytecode)
    end

    def process_pragma(pragma_tag)
      case pragma_tag.scope
      when :global
        instance_eval pragma_tag.pragma_body
      when :local
        @current_pragma = create_pragma(pragma_tag.pragma_body)
        @current_pragma.compiler = self
      else
        raise "unknown pragma #{pragma_tag}"
      end
    end

    def get_pragma
      if @current_pragma
        ret = @current_pragma
        @current_pragma = nil
        ret
      else
        @default_pragma
      end
    end

    def no_code_for(*types)
      types.each do |typ|
        default_pragma.delete_candidate(typ)
      end
    end
  end

  module Node
    def convert_to_bytecode(compiler)
      ByteCode::PrintStaticNode[self]
    end
  end

  class AmritaPragma
    def convert_to_bytecode(compiler)
      compiler.process_pragma(self)
      super
    end
  end

  class NodeArray
    def convert_to_bytecode(compiler)
      if has_id_element?
        ret = ByteCode::Sequence.new
        children.each do |n|
          ret << n.convert_to_bytecode(compiler)
        end
        ret
      else
        ByteCode::PrintStaticNode.new(self)
      end
    end
  end

  module ElementCommon
    def convert_to_bytecode(compiler)
      exattrs = get_expandable_attrs

      aid = amrita_id
      if compiler.use_simplespan and simple_span?
        ByteCode::SimpleSpan[amrita_id, body]
      elsif aid
        ret = ByteCode::Sequence[
          ByteCode::GetDataByKey[aid],
          compiler.get_pragma.convert_element_to_bytecode(self),
          ByteCode::PopData[]
        ]
        ret
      elsif exattrs.size > 0
        ByteCode::Sequence[
          ByteCode::ExtractData[exattrs],
          ByteCode::MergeElement[self],
          ByteCode::PrintDynamicElement[
            body.convert_to_bytecode(compiler),
          ]
        ]
      elsif has_id_element?
        ByteCode::Sequence[
          ByteCode::PrintStaticText[compiler.ep.format_start_tag(self)],
          body.convert_to_bytecode(compiler),
          ByteCode::PrintStaticText[compiler.ep.format_end_tag(self)]
        ]
      else
        ByteCode::PrintStaticNode[ self ]
      end
    end
  end

end
