require 'runit/testcase'
require 'amrita/cc'
require 'amrita/testsupport'



class TestCC < RUNIT::TestCase
  include Amrita

  class TestCompiledTemplate < CCompiledTemplate

    def execute(register, stack, out, vm)
      out << "main method"
    end

    def execute_amulet_am1(register, stack, out, vm)
      out << "amulet am1 method"
    end

    def execute_amulet_am2(register, stack, out, vm)
      out << "amulet am2 method"
    end

    def amulet_specs 
      {
        :am1 => { :amulet_class=>DataAmulet, :element_code => "e(:x)" },
        :am2 => { :amulet_class=>DataAmulet, :element_code => "e(:x)"  }
      }
    end

  end

  def test_loading
    t = TestCompiledTemplate.new
    result =""
    t.expand(result, true)
    assert_equal("main method", result)
  end

  def test_make_ext
    savedir  = Dir::pwd

    tmp = "/tmp/testamritacc"
    system "rm -r #{tmp}"
    Dir::mkdir tmp unless FileTest::directory? tmp

    amcc = Amcc.new(tmp, "testamcc", "AmccOut", "../ext")
    t = TemplateText.new '<!-- amrita_pragma: ScalarData --><div id=test></div>'
    amcc.add_template("test1.html", t) 
    amcc.make_ext
    src = File::open(File::join(tmp, "testamcc.c")).read
    Dir::chdir tmp
    system "ruby extconf.rb"
    raise  unless system "make"
    require File::join(tmp, "testamcc.so")
    assert_equal(Module, AmccOut.class)
    t = AmccOut::Templates["test1.html"]
    assert(t.kind_of?(CCompiledTemplate))

    result = ""
    t.expand(result, :test=>'abc')
    assert_equal("<div>abc</div>", result)
  ensure
    Dir::chdir savedir
  end

  def test_make_ext2
    savedir  = Dir::pwd

    tmp = "/tmp/testamritacc"

    system "rm -r #{tmp}"
    Dir::mkdir tmp unless FileTest::directory? tmp

    amcc = Amcc.new(tmp, "testamcc2", "AmccOut2", "../ext")
    t = TemplateText.new '<div id=test><span id=a>qqqq</span></div>'
    amcc.add_template("test1.html", t) 
    amcc.make_ext
    #t.vm.iset.dump(STDOUT)

    src = File::open(File::join(tmp, "testamcc2.c")).read
    #print src
    Dir::chdir tmp
    system "ruby extconf.rb"
    raise  unless system "make"

    require File::join(tmp, "testamcc2.so")
    t = AmccOut2::Templates["test1.html"]

    result = ""
    t.expand(result, :test=>%w(1 2 3))
    assert_equal("<div>1</div><div>2</div><div>3</div>", result)


    result = ""
    t.setup
    t.expand(result, :test=>'abc')
    assert_equal("<div>abc</div>", result)

    result = ""
    t.expand(result, :test=>[1, 2, 3])
    assert_equal("<div>1</div><div>2</div><div>3</div>", result)


    result = ""
    t.expand(result, :test=> { :a=>'xyz' })
    assert_equal("<div>xyz</div>", result)

    result = ""
    t.expand(result, :test=>[ {:a=>'abc'}, { :a=>'xyz' }])
    assert_equal("<div>abc</div><div>xyz</div>", result)

    result = ""
    t.expand(result, :test=>true)
    assert_equal("<div>qqqq</div>", result)

    result = ""
    t.expand(result, :test=>nil)
    assert_equal("", result)

    result = ""
    t.expand(result, :test=>{ :a=>a(:zzz=>'yyy')})
    assert_equal('<div><span zzz="yyy">qqqq</span></div>', result)

    result = ""
    t.expand(result, :test=>{ :a=>a(:zzz=>'yy') {'xxx'}})
    assert_equal('<div><span zzz="yy">xxx</span></div>', result)

    result = ""
    t.expand(result, :test=> proc do |reg, *args|
               reg[:zzz] = "yy"
               reg.set_text('xxx')
               reg
             end)
    assert_equal('<div zzz="yy">xxx</div>', result)

    result = ""
    s1 = Struct.new(:test)
    s2 = Struct.new(:a)
    d = s2.new('666')
    d.extend Amrita::ExpandByMember
    data = s1.new(d)
    t.expand(result, data)
    assert_equal('<div>666</div>', result)

  ensure
    Dir::chdir savedir
  end

  def test_make_ext3
    savedir  = Dir::pwd

    tmp = "/tmp/testamritacc"
    system "rm -r #{tmp}"
    Dir::mkdir tmp unless FileTest::directory? tmp

    tmpl = <<END
<table border="1">
  <tr><th>name</th><th>author</th><th>webpage</tr>
  <tr id=table1>
    <td id="name"></td>
    <td id="author"></td>
    <td><a id="webpage"></a></td>
  </tr>
</table>
END

    data = {
    :table1=>[ 
    { 
      :name=>"Ruby", 
      :author=>"matz" , 
      :webpage=> a(:href=>"http://www.ruby-lang.org/") { "Ruby Home Page" },
    },
    { 
      :name=>"python", 
      :author=>"Guido van Rossum" ,
      :webpage=> a(:href=>"http://www.python.org/") { "Python Language Website" },
    },
    { 
      :name=>"perl", 
      :author=>"Larry Wall" ,
      :webpage=> a(:href=>"http://www.perl.com/") { "Perl.com" },
    },
   ] 
    }
    amcc = Amcc.new(tmp, "testamcc3", "AmccOut3", "../ext")
    t = TemplateText.new tmpl
    amcc.add_template("test1.html", t) 
    amcc.make_ext

    src = File::open(File::join(tmp, "testamcc3.c")).read
    #print src
    Dir::chdir tmp
    system "ruby extconf.rb"
    raise  unless system "make"
    require File::join(tmp, "testamcc3.so")
    t = AmccOut3::Templates["test1.html"]

    result = ""
    t.setup
    t.expand(result, data)
    
    assert_equal('<table border="1">
  <tr><th>name</th><th>author</th><th>webpage</th></tr>
  <tr>
    <td>Ruby</td>
    <td>matz</td>
    <td><a href="http://www.ruby-lang.org/">Ruby Home Page</a></td>
  </tr><tr>
    <td>python</td>
    <td>Guido van Rossum</td>
    <td><a href="http://www.python.org/">Python Language Website</a></td>
  </tr><tr>
    <td>perl</td>
    <td>Larry Wall</td>
    <td><a href="http://www.perl.com/">Perl.com</a></td>
  </tr>
</table>
', result)


    File::open("/dev/null", "w") do |f|
      tt = Time.now
      1000.times { t.expand(f, data) }
      p Time.now - tt
    end
  ensure
    Dir::chdir savedir
  end    


  def test_multi
    savedir  = Dir::pwd

    tmp = "/tmp/testamritacc"
    system "rm -r #{tmp}"
    Dir::mkdir tmp unless FileTest::directory? tmp

    amcc = Amcc.new(tmp, "testmulti", "AmccMulti", "../ext")
    t1 = TemplateText.new '<div id=test></div>'
    amcc.add_template("test1.html", t1) 
    t2 = TemplateText.new 'test2<div id=test></div>test2'
    amcc.add_template("test2.html", t2) 
    t3 = TemplateText.new '<div id=test3>test3</div>'
    amcc.add_template("test3.html", t3) 
    amcc.make_ext
    src = File::open(File::join(tmp, "testmulti.c")).read
    Dir::chdir tmp
    system "ruby extconf.rb"
    raise  unless system "make"
    require File::join(tmp, "testmulti.so")

    t1 = AmccMulti::Templates["test1.html"]
    result = ""
    t1.expand(result, :test=>'abc')
    assert_equal("<div>abc</div>", result)

    t2 = AmccMulti::Templates["test2.html"]
    result = ""
    t2.expand(result, :test=>'xyz')
    assert_equal("test2<div>xyz</div>test2", result)

    t3 = AmccMulti::Templates["test3.html"]
    result = ""
    t3.expand(result, :test3=>true)
    assert_equal("<div>test3</div>", result)
  ensure
    Dir::chdir savedir
  end


  def test_expand_attr
    savedir  = Dir::pwd

    tmp = "/tmp/testamritacc"
    system "rm -r #{tmp}"
    Dir::mkdir tmp unless FileTest::directory? tmp

    amcc = Amcc.new(tmp, "testattr", "AmccOut", "../ext")
    t = TemplateText.new '<span id=test><span xxx="@aaa" yyy="@bbb"><span id="ccc"></span></span></span>'
    amcc.add_template("test1.html", t) 
    amcc.make_ext
    src = File::open(File::join(tmp, "testattr.c")).read
    #puts src
    Dir::chdir tmp
    system "ruby extconf.rb"
    raise  unless system "make"
    require File::join(tmp, "testattr.so")
    assert_equal(Module, AmccOut.class)
    t = AmccOut::Templates["test1.html"]
    assert(t.kind_of?(CCompiledTemplate))

    result = ""
    t.expand(result, :test=>{:aaa=>123, :bbb=>456, :ccc=>nil})
    assert_equal('<span xxx="123" yyy="456"></span>', result)

    result = ""
    t.expand(result, :test=>{:aaa=>'http://www.walru-ruby.org/', :bbb=>'<>&', :ccc=>789})
    assert_equal('<span xxx="http://www.walru-ruby.org/" yyy="&lt;&gt;&amp;">789</span>', result)
  ensure
    Dir::chdir savedir
  end

  def test_amulet
    savedir  = Dir::pwd

    tmp = "/tmp/testamritacc"
    system "rm -r #{tmp}"
    Dir::mkdir tmp unless FileTest::directory? tmp

    amcc = Amcc.new(tmp, "testamulet", "AmccOut", "../ext")

    t = TemplateText.new '<xx><yy id=aaa><zz id=bbb></zz></yy></xx>'
    t.define_amulet(:aaa)
    amcc.add_template("test1.html", t) 

    t = TemplateText.new '<div id=t2></div>'
    t.setup
    amcc.add_template("test2.html", t) 
    #t.vm.iset.dump(STDOUT)

    amcc.make_ext
    src = File::open(File::join(tmp, "testamulet.c")).read
    #puts src
    Dir::chdir tmp
    system "ruby extconf.rb"
    raise  unless system "make"

    require File::join(tmp, "testamulet.so")

    t = AmccOut::Templates["test1.html"]
    t.setup
    assert_equal(CCompiledAmuletSeed, t[:aaa].class)
    data = { :aaa=>t[:aaa][ { :bbb=>'aaa'} ] }
    result = ""
    t.expand(result, data)
    assert_equal('<xx><yy><zz>aaa</zz></yy></xx>', result)

    result = ""
    data = { :aaa=>t[:aaa]['xxx'] }
    t.expand(result, data)
    assert_equal('<xx><yy>xxx</yy></xx>', result)

    t2 = AmccOut::Templates["test2.html"]
    result = ""
    t2.expand(result, :t2=>'aaa')
    assert_equal('<div>aaa</div>', result)


    result = ""
    t2.expand(result, :t2=>t[:aaa][{:bbb=>789}])
    assert_equal('<div><yy><zz>789</zz></yy></div>', result)

  ensure
    Dir::chdir savedir
  end

end

#--- main program ----
if __FILE__ == $0
  require 'runit/cui/testrunner'

  if ARGV.size == 0
    RUNIT::CUI::TestRunner.run(TestCC.suite)
  else
    ARGV.each do |method|
      RUNIT::CUI::TestRunner.run(TestCC.new(method))
    end
  end
end

