/* # skkinput (Simple Kana-Kanji Input)
 *
 * This file is part of skkinput.
 * Copyright (C) 2002
 * Takashi SAKAMOTO (PXG01715@nifty.ne.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "AfxWin.h"
#include "dispatch.h"
#include <assert.h>
#include <stdarg.h>
#include "lmachinep.h"
#include "lispmgrp.h"
#include "TTerminal.h"
#include "TFrame.h"
#include "TNormalFrame.h"
#include "TLispClient.h"

/*	Prototypes
 */
static	Boolean		tLispClient_initialize		(TLispClient*, TLispManager*, TLispMachine*, Widget, Widget) ;
static	Boolean		tLispClient_createMinibuffer(TLispMachine*, TLispEntity**) ;
static	Boolean		tLispClient_createFrame		(TLispMachine*, Widget, Widget, TLispEntity*, TLispEntity*) ;
static	void		tLispClient_postInitialize	(TLispClient*) ;
static	Boolean		tLispClient_tickAll			(void) ;
static	void		tLispClient_register		(TLispClient*) ;
static	void		tLispClient_unregister		(TLispClient*) ;
static	void		tLispClient_updateParent	(TLispMachine*) ;
static	Boolean		tLispClient_parseAndEval	(TLispMachine*, const char*, ...) ;
#if defined (USE_BLOCKHOOK)
static	Boolean		tLispClient_workHook		(XtPointer) ;
#endif

/*	Local Global Variables
 */
static	TLispClient*		slstLispClient	= NULL ;
#if defined (USE_BLOCKHOOK)
static	volatile Boolean	sfRunGC ;
static	volatile int		snNeedToRunGC ;
#endif

/*
 */
void
TLispClient_MainLoop (
	register XtAppContext	appcontext,
	register TLispMachine*	pTopLM,
	register int			(*pGetExitFlagProc)(void))
{
	register Boolean		fBlock		= True ;
	XEvent					xev ;
#if defined (USE_BLOCKHOOK)
	register XtWorkProcId	nID			= 0 ;
	register TLispManager*	pLispMgr	= pTopLM->m_pLispMgr ;
#endif

	assert (pTopLM != NULL) ;
	assert (pGetExitFlagProc != NULL) ;

#if defined (USE_BLOCKHOOK)
	sfRunGC			= False ;
	snNeedToRunGC	= 0 ;
#endif

	while (! (*pGetExitFlagProc)()) {
		if (fBlock) {
			XtAppNextEvent (appcontext, &xev) ;
			AfxDispatchEvent (&xev) ;
		} else {
			while (XtAppPending (appcontext) != 0) {
				XtAppNextEvent (appcontext, &xev) ;
				AfxDispatchEvent (&xev) ;
			}
		}
		fBlock	= tLispClient_tickAll () ;
#if defined (USE_BLOCKHOOK)
		if (snNeedToRunGC > 0 && !sfRunGC) {
			nID	= XtAppAddWorkProc (appcontext, tLispClient_workHook, pLispMgr) ;
			sfRunGC	= True ;
		}
#endif
	}
#if defined (USE_BLOCKHOOK)
	if (sfRunGC) {
		XtRemoveWorkProc (nID) ;
		sfRunGC	= False ;
	}
#endif
	return ;
}

/*
 */
Boolean
TLispClient_ClassInitialize (
	register TLispManager**	ppLispMgr,
	register TLispMachine**	ppLM,
	register const char*	pStrConfigPath,
	register const char*	pStrServerHost,
	register int			nPortNum,
	register Boolean		fCreateLispMgr)
{
	TLispManager*		pLispMgr ;
	TLispMachine*		pLM ;
	TLispEntity*		pEntBuffer ;
	register const char*	pPath ;

	if (fCreateLispMgr) {
		if (TFAILED (TLispMgr_Create (&pLispMgr)))
			return	False ;
	} else {
		if (ppLispMgr == NULL || *ppLispMgr == NULL)
			return	False ;
		pLispMgr	= *ppLispMgr ;
	}
	if (TFAILED (TLispMachine_Create (pLispMgr, NULL, &pLM)))
		return	False ;

	lispMgr_CreateBuffer (pLispMgr, &pEntBuffer) ;
	lispMachine_InsertBuffer (pLM, pEntBuffer) ;
	lispMachineCode_SetCurrentBuffer (pLM, pEntBuffer) ;

	pPath	= (pStrConfigPath == NULL || *pStrConfigPath == '\0')? "." : pStrConfigPath ;
	if (TFAILED (tLispClient_parseAndEval (pLM, "(setq load-path '(\"%s\"))", pPath)) ||
		TFAILED (tLispClient_parseAndEval (pLM, "(load \"init.el\")")))
		return	False ;
	if (pStrServerHost != NULL &&
		*pStrServerHost != '\0' &&
		TFAILED (tLispClient_parseAndEval (pLM, "(setq skk-server-host \"%s\")", pStrServerHost)))
		return	False ;
	if (nPortNum >= 0 &&
		TFAILED (tLispClient_parseAndEval (pLM, "(setq skk-portnum %d)", nPortNum)))
		return	False ;
	
	*ppLispMgr	= pLispMgr ;
	*ppLM		= pLM ;
	return	True ;
}

void
TLispClient_ClassFinalize (
	register TLispMachine*	pLM)
{
	static const char		sstrEND []	= "(mutual-eval skk-jisyo-mutex '(progn (setq this-command 'save-buffers-kill-emacs) (skk-save-jisyo)))" ;
	register TLispManager*	pLispMgr ;
	register TLispEntity*	pEntTarget ;

#if defined (DEBUG)
	fprintf (stderr, "TLispClient_ClassFinalize (%p)\n", pLM) ;
#endif
	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	pEntTarget	= lispMgr_ParseStringA (pLispMgr, sstrEND, NELEMENTS (sstrEND) - 1, NULL) ;
	assert (pEntTarget != NULL) ;
	TLispMachine_Test (pLM, pEntTarget) ;
	return ;
}

void
TLispClient_PreInitialize (
	register TLispClient*	pClient)
{
	assert (pClient != NULL) ;
	pClient->m_pLM			= NULL ;
	pClient->m_pNext		= NULL ;
	return ;
}

Boolean
TLispClient_Initialize (
	register TLispClient*		pClient,
	register TLispClientArg*	pArg)
{
	register TLispManager*	pLispMgr ;
	register TLispMachine*	pParentLM ;
	register TLispMachine*	pLM ;
	TLispEntity*			pEntClient ;
	static const char		sstrClient []	= "im-client" ;

	assert (pClient != NULL) ;
	assert (pArg    != NULL) ;
	pLispMgr	= pArg->m_pLispMgr ;
	pParentLM	= pArg->m_pLM ;
	assert (pLispMgr != NULL) ;

	if (TFAILED (tLispClient_initialize (pClient, pLispMgr, pParentLM, pArg->m_wgFrame, pArg->m_wgMinibufFrame)))
		return	False ;

	/*	im-client ȤƤνɬפǤʤСǽ򽪤롣*/
	if (pArg->m_pvClient == NULL)
		return	True ;

	/*	im-client entity 롣*/
	if (TFAILED (lispMgr_CreateIMClient (pLispMgr, pArg->m_pvClient, pArg->m_pKeyNotify, pArg->m_pTextNotify, pArg->m_pEndNotify, &pEntClient))) {
		return	False ;
	}
	lispEntity_AddRef (pLispMgr, pEntClient) ;

	/*	symbol: im-client  im-client entity  bind 롣*/
	pLM	= pClient->m_pLM ;
	lispMachine_SetCurrentSymbolValueWithNameA (pLM, sstrClient, NELEMENTS (sstrClient) - 1, pEntClient) ;
	lispEntity_Release (pLispMgr, pEntClient) ;

	tLispClient_register (pClient) ;
	return	True ;
}

void
TLispClient_Uninitialize (
	register TLispClient*	pClient)
{
	register TLispMachine*	pLM	= pClient->m_pLM ;

	if (pLM != NULL) {
		register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;

		tLispClient_updateParent (pLM) ;
		TLispMachine_Destroy (pLM) ;
		pClient->m_pLM	= NULL ;
#if defined (DEBUG)
		fprintf (stderr, "Garbage Collecting ... ") ;
		fflush (stderr) ;
#endif
		lispMgr_CollectGarbage (pLispMgr) ;
		lispMgr_CollectGarbage (pLispMgr) ;
		lispMgr_CollectGarbage (pLispMgr) ;
		lispMgr_CollectGarbage (pLispMgr) ;
#if defined (DEBUG)
		fprintf (stderr, "done\n") ;
#endif
		/*	दξήˤҤäѤꤢʤȤޤʤ
		 */
	}
	tLispClient_unregister (pClient) ;
	return ;
}

void
TLispClient_Reinitialize (
	register TLispClient*	pClient)
{
	assert (pClient != NULL) ;
	assert (pClient->m_pLM != NULL) ;
	lispMachine_ScheduleUpdateAllFrame (pClient->m_pLM) ;
	return ;
}

Boolean
TLispClient_Tick (
	register TLispClient*	pClient)
{
	register TLispMachine*	pLM	= pClient->m_pLM ;
	register TLMRESULT		res ;
	register Boolean		fRetval	= True ;

	if (!TLispClient_Modifiedp (pClient))
		return 	True ;

	res	= lispMachine_ExecuteLoop (pLM) ;
	if (res != LMR_DESTROYED) {
#if defined (USE_BLOCKHOOK)
		if (res == LMR_TICK) {
			fRetval	= False ;
		} else {
			TLispClient_SetModificationFlag (pClient, False) ;
		}
		/*	ʤ dirty ʥɤˤ*/
		lispMachine_UpdateAllFrame (pLM) ;
		snNeedToRunGC	++ ;
#else
		if (res == LMR_TICK) {
			fRetval	= False ;
		} else {
			TLispClient_SetModificationFlag (pClient, False) ;
			
			/*	ʤ dirty ʥɤˤ*/
			lispMachine_UpdateAllFrame (pLM) ;
			
			/*lispMachine_UpdateCurrentFrame (pClient->m_pLM) ;*/
			/*  4Entity ⤬줿᤮ȻפΤǡ
			 *	 GC ư٤ʤΤ ͤѰդ٤
			 *	Ȥϻפɡ*/
			lispMgr_CollectGarbage (pLM->m_pLispMgr) ;
		}
#endif
	}
	return	fRetval ;
}

void
TLispClient_Activate (
	register TLispClient*	pClient,
	register Boolean		fActive)
{
	assert (pClient != NULL) ;
	assert (pClient->m_pLM != NULL) ;
	lispMachine_ActivateAllFrame (pClient->m_pLM, fActive) ;
	return ;
}	

Boolean
TLispClient_Modifiedp (
	register TLispClient*	pClient)
{
	return	pClient->m_fModify ;
}

void
TLispClient_SetModificationFlag (
	register TLispClient*	pClient,
	register Boolean		fModificationFlag)
{
	pClient->m_fModify	= fModificationFlag ;
	return ;
}

Boolean
tLispClient_initialize (
	register TLispClient*		pClient,
	register TLispManager*		pLispMgr,
	register TLispMachine*		pMacParent,
	register Widget				wgFrame,
	register Widget				wgMinibufFrame)
{
	register TLispMachine*	pLM ;
	TLispEntity*	pEntBuffer ;
	TLispEntity*	pEntMinibuf ;
	
	assert (pClient  != NULL) ;
	assert (pLispMgr != NULL) ;

	if (TFAILED (TLispMachine_Create (pLispMgr, pMacParent, &pClient->m_pLM))) 
		return	False ;

	pLM		= pClient->m_pLM ;
	/* Buffer 롣*/
	if (TFAILED (lispMgr_CreateBuffer (pLispMgr, &pEntBuffer)))
		return	False ;
	lispMachine_InsertBuffer (pClient->m_pLM, pEntBuffer) ;
	lispMachineCode_SetCurrentBuffer (pClient->m_pLM, pEntBuffer) ;

	if (TFAILED (tLispClient_createMinibuffer (pLM, &pEntMinibuf))) {
		lispMachine_RemoveBuffer (pLM, pEntBuffer) ;
		return	False ;
	}
	if (TFAILED (tLispClient_createFrame (pLM, wgFrame, wgMinibufFrame, pEntBuffer, pEntMinibuf))) {
		lispMachine_RemoveBuffer (pLM, pEntBuffer) ;
		lispMachine_RemoveBuffer (pLM, pEntMinibuf) ;
		return	False ;
	}
	tLispClient_postInitialize (pClient) ;
	return	True ;
}

Boolean
tLispClient_createMinibuffer (
	register TLispMachine*	pLM,
	register TLispEntity**	ppEntMinibuf)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*			pEntNil ;
	TLispEntity*			pEntMinibuf ;
	TLispEntity*			pEntModeline ;

	assert (pLM != NULL) ;
	assert (ppEntMinibuf != NULL) ;

	/*	Minibuffer 롣*/
	if (TFAILED (lispMgr_CreateBuffer (pLispMgr, &pEntMinibuf)))
		return	False ;
	/*	Minibuffer  mode-line-format  nil ˤƤ*/
	lispMachine_InsertBuffer (pLM, pEntMinibuf) ;
	lispMgr_CreateNil (pLispMgr, &pEntNil) ;
	pEntModeline	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_MODE_LINE_FORMAT) ;
	lispBuffer_MakeSymbolValue (pLispMgr, pEntMinibuf, pEntModeline) ;
	lispBuffer_SetSymbolValue (pLispMgr, pEntMinibuf, pEntModeline, pEntNil) ;
	*ppEntMinibuf	= pEntMinibuf ;
	return	True ;
}

Boolean
tLispClient_createFrame (
	register TLispMachine*	pLM,
	register Widget			wgFrame,
	register Widget			wgMinibufFrame,
	register TLispEntity*	pEntBuffer,
	register TLispEntity*	pEntMinibuf)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntFrame ;
	TLispEntity*	pEntWindow ;
	TLispEntity*	pEntMinibufFrame ;
	TLispEntity*	pEntMinibufWindow ;

	assert (pLM         != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (wgFrame     != NULL) ;
	assert (pEntBuffer  != NULL) ;
	assert (pEntMinibuf != NULL) ;

	/*	Minibuffer Ѥ frame ̤ѰդƤʤ顢 frame 
	 *	롣
	 */
	if (wgMinibufFrame != NULL) {
		if (TFAILED (lispMgr_MakeFrame (pLispMgr, wgMinibufFrame, NULL, pEntMinibuf, NULL, &pEntMinibufFrame)))
			return	False ;
		if (TFAILED (lispFrame_GetTopWindow (pEntMinibufFrame, &pEntMinibufWindow)))
			return	False ;
		assert (pEntMinibufFrame  != NULL) ;
		assert (pEntMinibufWindow != NULL) ;
		//lispWindow_SetMinibufWindow (pEntMinibufWindow, True) ;
		lispMachine_InsertFrame (pLM, pEntMinibufFrame) ;
		XtVaSetValues (wgMinibufFrame, XtNlispFrameObject, pEntMinibufFrame, NULL) ;
	} else {
		pEntMinibufFrame	= NULL ;
		pEntMinibufWindow	= NULL ;
	}
	
	/* Frame 롣*/
	if (TFAILED (lispMgr_MakeFrame (pLispMgr, wgFrame, pEntBuffer, pEntMinibuf, pEntMinibufWindow, &pEntFrame))) {
		if (pEntMinibufFrame != NULL)
			lispMachine_RemoveFrame (pLM, pEntMinibufFrame) ;
		return	False ;
	}
	XtVaSetValues (wgFrame, XtNlispFrameObject, pEntFrame, NULL) ;

	lispMachine_InsertFrame (pLM, pEntFrame) ;
	lispMachineCode_SetCurrentFrame (pLM, pEntFrame) ;
	lispFrame_GetTopWindow (pEntFrame, &pEntWindow) ;
	lispMachineCode_SetCurrentWindow (pLM, pEntWindow) ;
	return	True ;
}

void
tLispClient_postInitialize (
	register TLispClient*	pClient)
{
	static const char*	apGlobalSymbols [] = {
		"unread-command-events",
		"this-command",
		"last-command",
		"last-command-event",
		"unread-command-events",
	} ;
	static const char	sstrSkkStart []	= "(load \"skk-startup.el\")" ;
	register TLispMachine*	pLM ;
	register TLispManager*	pLispMgr ;
	register const char**	ppString ;
	register int			nString ;
	TLispEntity*			pEntNil ;
	register TLispEntity*	pEntTarget ;
	register int			i ;

	pLM			= pClient->m_pLM ;
	pLispMgr	= pLM->m_pLispMgr ;
	lispMgr_CreateNil (pLispMgr, &pEntNil) ;

	/*	Machine  VARIABLE 롣*/
	ppString	= apGlobalSymbols ;
	for (i = 0 ; i < NELEMENTS (apGlobalSymbols) ; i ++) {
		nString	= strlen (*ppString) ;
		lispMachine_SetCurrentSymbolValueWithNameA (pLM, *ppString, nString, pEntNil) ;
		ppString	++ ;
	}

	pEntTarget	= lispMgr_ParseStringA (pLispMgr, sstrSkkStart, NELEMENTS (sstrSkkStart) - 1, NULL) ;
	TLispMachine_Test (pLM, pEntTarget) ;

	pClient->m_pLM->m_pState	= &lispMachineState_WindowProc ;
	pClient->m_fModify			= True ;	/* ͤ True */
	return ;
}	

void
tLispClient_register (
	register TLispClient*	pClient)
{
	assert (pClient != NULL) ;
	
	pClient->m_pNext	= slstLispClient ;
	slstLispClient		= pClient ;
	return ;
}

void
tLispClient_unregister (
	register TLispClient*	pClient)
{
	register TLispClient*	pPrevNode ;
	register TLispClient*	pNode ;

	assert (pClient != NULL) ;
	pNode		= slstLispClient ;
	pPrevNode	= NULL ;
	while (pNode != NULL) {
		if (pNode == pClient) {
			if (pPrevNode != NULL) {
				pPrevNode->m_pNext	= pNode->m_pNext ;
			} else {
				slstLispClient		= pNode->m_pNext ;
			}
			return ;
		}
		pPrevNode	= pNode ;
		pNode		= pNode->m_pNext ;
	}
	return ;
}

Boolean
tLispClient_tickAll (void)
{
	register Boolean		fBlock, fRetval ;
	register TLispClient*	pNode ;

	fBlock	= True ;
	pNode	= slstLispClient ;
	while (pNode != NULL) {
		fRetval	= TLispClient_Tick (pNode) ;
		fBlock	&= fRetval ;
		pNode	= pNode->m_pNext ;
	}
	return	fBlock ;
}

/*	दξήˤҤäѤꤢʤȤޤʤ
 */
void
tLispClient_updateParent (
	register TLispMachine*	pLM)
{
	register TLispMachine*	pTopLM ;
	register TLispManager*	pLispMgr ;
	register const char*	pString ;
	register int			nString ;
	register int			i ;
	TLispEntity*			pEntTarget ;
	TLispEntity*			pEntSrc ;
	TLispEntity*			pEntDest ;
	TLispEntity*			pEntNewValue ;

	/*	ɬפʤΤϼơ
	 *		j-count-touroku			-> û
	 *		j-count-kakutei			-> û
	 *		j-jisyo-buffer-modified	-> ñ㥳ԡ
	 *	˻äƹԤʤȤʤ
	 */
	static struct {
		const char*		m_pName ;
		int				m_nName ;
		Boolean			m_fPlus ;
	}	rUpdateEntity [] = {
		{	"j-count-touroku",			0,	True, },
		{	"j-count-kakutei",			0,	True, },
		{	"j-jisyo-buffer-modified",	0,	False, },
	} ;

	assert (pLM != NULL) ;
	pTopLM		= pLM->m_pMacParent ;
	if (pTopLM == NULL)
		return ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	for (i = 0 ; i < NELEMENTS (rUpdateEntity) ; i ++) {
		pString	= rUpdateEntity [i].m_pName ;
		assert (pString != NULL) ;
		if (rUpdateEntity [i].m_nName <= 0) {
			nString	= rUpdateEntity [i].m_nName = strlen (pString) ;
		} else {
			nString	= rUpdateEntity [i].m_nName ;
		}
		if (TFAILED (lispMgr_InternSymbolA (pLispMgr, pString, nString, &pEntTarget)))
			continue ;
		lispEntity_AddRef (pLispMgr, pEntTarget) ;

		if (TFAILED (lispMachine_GetCurrentSymbolValue (pLM, pEntTarget, &pEntSrc)))
			goto	skip ;
		if (rUpdateEntity [i].m_fPlus) {
			long	lDest, lSrc ;

			if (TFAILED (lispMachine_GetCurrentSymbolValue (pTopLM, pEntTarget, &pEntDest)) ||
				TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntDest, &lDest)))
				lDest	= 0 ;
			if (TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntSrc, &lSrc)) || lSrc == 0)
				goto	skip ;
			lDest	+= lSrc ;
			if (TFAILED (lispMgr_CreateInteger (pLispMgr, lDest, &pEntNewValue)))
				break ;
		} else {
			pEntNewValue	= pEntSrc ;
		}
		lispMachine_SetCurrentSymbolValue (pTopLM, pEntTarget, pEntNewValue) ;
	  skip:
		lispEntity_Release (pLispMgr, pEntTarget) ;
	}
	return ;
}

Boolean
tLispClient_parseAndEval (
	register TLispMachine*	pLM,
	register const char*	pFormat,
	...)
{
	register TLispManager*	pLispMgr ;
	register TLispEntity*	pEntTarget ;
	char	strBuffer [512] ;
	va_list	ap ;

	assert (pLM     != NULL) ;
	assert (pFormat != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	va_start (ap, pFormat) ;
	vsnprintf (strBuffer, NELEMENTS (strBuffer) - 1, pFormat, ap) ;
	va_end (ap) ;
	strBuffer [NELEMENTS (strBuffer) - 1]	= '\0' ;
#if defined (DEBUG) || 0
	fprintf (stderr, "%s\n", strBuffer) ;
#endif

	pEntTarget	= lispMgr_ParseStringA (pLispMgr, strBuffer, strlen (strBuffer), NULL) ;
	if (pEntTarget == NULL)
		return	False ;
	TLispMachine_Test (pLM, pEntTarget) ;
	lispMgr_CollectGarbage (pLispMgr) ;
	return	True ;
}

#if defined (USE_BLOCKHOOK)

Boolean
tLispClient_workHook (
	register XtPointer	client_data)
{
#if defined (debug) || 0
	fprintf (stderr, "Garbage collecting ... %d\n", snNeedToRunGC) ;
#endif
	lispMgr_CollectGarbage ((TLispManager *)client_data) ;
	snNeedToRunGC	-- ;
	if (snNeedToRunGC <= 0) {
		sfRunGC	= 0 ;
		return	True ;
	}
	return	False ;
}

#endif
	
