/* # skkinput (Simple Kana-Kanji Input)
 *
 * This file is part of skkinput.
 * Copyright (C) 2002
 * Takashi SAKAMOTO (PXG01715@nifty.ne.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "local.h"
#include <stdio.h>
#include <assert.h>
#include "lispmgrp.h"
#include "cstring.h"
#include "kanji.h"

#define	lispEntity_GetBufferPtr(ptr)	((TLispBuffer *)((TLispEntity *)(ptr) + 1))

static	Boolean			lispBuffer_initialize	(TLispManager*, TLispEntity*) ;
static	Boolean			lispBuffer_uninitialize (TLispManager*, TLispEntity*) ;

static inline	TLispEntity*
lispBuffer_getMarker (
	register TLispEntity*	pEntBuffer,
	register int			nMarker)
{
	register TLispBuffer*	pBuffer ;
	
	assert (pEntBuffer != NULL) ;
	assert (MARKER_BTOP <= nMarker && nMarker < NUM_BUFFER_MUST_MARKERS) ;

	pBuffer			= lispEntity_GetBufferPtr (pEntBuffer) ;
	return	pBuffer->m_apMustMarkers [nMarker] ;
}

static inline	Boolean
lispBuffer_getMarkerPosition (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntBuffer,
	register int			nMarker,
	register int*			pnRetval)
{
	TLispEntity*	pMarkerBuffer ;
	TLispEntity*	pMarker ;
	
	assert (pLispMgr      != NULL) ;
	assert (pEntBuffer != NULL) ;
	assert (MARKER_BTOP <= nMarker && nMarker < NUM_BUFFER_MUST_MARKERS) ;
	assert (pnRetval      != NULL) ;

	pMarker			= lispBuffer_getMarker (pEntBuffer, nMarker) ;
	if (pMarker == NULL)
		return	False ;
	if (TSUCCEEDED (lispMarker_GetBufferPosition (pLispMgr, pMarker, &pMarkerBuffer, pnRetval)) &&
		pMarkerBuffer == pEntBuffer) 
		return	True ;
	return	False ;
}

/*	buffer entity 롣
 */
Boolean
lispMgr_CreateBuffer (
	register TLispManager*			pLispMgr,
	register TLispEntity** const	ppEntityRet)
{
	TLispEntity*			pEntity ;
	register TLispBuffer*	pBuffer ;
	register int			i ;
	TLispEntity**			ppEntSymbol ;
	int						nEntSymbol ;
	TLispEntity*			pEntEmpty ;

	assert (pLispMgr    != NULL) ;
	assert (ppEntityRet != NULL) ;

#if defined (DEBUG_LV99)
	fprintf (stderr, "Garbage collecting ...") ;
	fflush (stderr) ;
	lispMgr_CollectGarbage (pLispMgr) ;
	fprintf (stderr, "done.\n") ;
#endif
	/*	ƥƥȤɬפʥΰݤ롣*/
	if (TFAILED (lispMgr_AllocateEntity (pLispMgr, sizeof (TLispBuffer), &pEntity))) 
		return	False ;

	/*	ƥƥȤƽ롣*/
	pEntity->m_iType	= LISPENTITY_BUFFER ;
	pBuffer				= lispEntity_GetBufferPtr (pEntity) ;

	/*	ХåեΥܥϤʤ*/
	for (i = 0 ; i < SIZE_LISP_BIND_TABLE ; i ++) {
		pBuffer->m_apVariableTable [i]	= NULL ;
	}
	/*	󥰥ꥹȤˤʤäƤ롣*/
	pBuffer->m_pNextBuffer		= pEntity ;
	pBuffer->m_pPrevBuffer		= pEntity ;
	pBuffer->m_pEntKeymap		= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_NIL) ;
	lispEntity_AddRef (pLispMgr, pBuffer->m_pEntKeymap) ;
	pBuffer->m_pEntName			= NULL ;
	pBuffer->m_fModified		= False ;
	pBuffer->m_nTick			= 0 ;
	
	/*	ХåեˤϰڤΥޡĤʤƤʤ*/
	pBuffer->m_lstMarkers		= NULL ;
	for (i = 0 ; i < NUM_BUFFER_MUST_MARKERS ; i ++) 
		pBuffer->m_apMustMarkers [i]	= NULL ;

	/*	ƥƥꥹȤˤĤʤ*/
	(void) lispMgr_RegisterMisc (pLispMgr, pEntity) ;

	/*	Хåե롣*/
	if (TFAILED (lispBuffer_initialize (pLispMgr, pEntity))) {
		lispMgr_DestroyEntity (pLispMgr, pEntity) ;
		return	False ;
	}

	/*	ǥեȤΥѿ롣ͤϤʤ*/
	(void) lispMgr_GetLocalSymbols (pLispMgr, &ppEntSymbol, &nEntSymbol) ;
	(void) lispMgr_CreateEmpty (pLispMgr, &pEntEmpty) ;
	while (nEntSymbol -- > 0) {
#if defined (DEBUG)
		fprintf (stderr, "Symbol(") ;
		lispEntity_Print (pLispMgr, *ppEntSymbol) ;
		fprintf (stderr, ") <-- nil\n") ;
#endif
		lispBuffer_MakeSymbolValue (pLispMgr, pEntity, *ppEntSymbol) ;
		lispBuffer_SetSymbolValue  (pLispMgr, pEntity, *ppEntSymbol, pEntEmpty) ;
		ppEntSymbol	++ ;
	}

	*ppEntityRet	= pEntity ;
	return	True ;
}

void
lispMgr_DestroyBuffer (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntBuffer)
{
	register TLispBuffer*	pBuffer ;

	assert (pEntBuffer     != NULL) ;
#if defined (DEBUG)
	fprintf (stderr, "lispMgr_DestroyBuffer (%p, %p)\n", pLispMgr, pEntBuffer) ;
#endif
	pBuffer	= lispEntity_GetBufferPtr (pEntBuffer) ;
	if (pBuffer->m_pEntKeymap != NULL)
		lispEntity_Release (pLispMgr, pBuffer->m_pEntKeymap) ;
	pBuffer->m_pEntKeymap	= NULL ;

	/*	ޡƤɬפ롣*/
	lispBuffer_uninitialize (pLispMgr, pEntBuffer) ;
	return ;
}

/*
 *[ǽ]
 *	Хåե֤᤹ʤ
 *	ХåեԽƤʸƾõХåե
 *	ѿƲޡ֤᤹
 */
Boolean
lispBuffer_Clear (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntBuffer)
{
	if (TFAILED (lispBuffer_uninitialize (pLispMgr, pEntBuffer)) ||
		TFAILED (lispBuffer_initialize   (pLispMgr, pEntBuffer)))
		return	False ;
	return	True ;
}

Boolean
lispBuffer_SetNext (
	register TLispEntity* pEntBuffer,
	register TLispEntity* pEntNextBuffer)
{
	register TLispBuffer*	pBuffer ;

	assert (pEntBuffer != NULL) ;
	assert (pEntBuffer->m_iType == LISPENTITY_BUFFER) ;
	assert ((pEntNextBuffer != NULL &&
			 pEntNextBuffer->m_iType == LISPENTITY_BUFFER) ||
			(pEntNextBuffer == NULL)) ;

	pBuffer	= lispEntity_GetBufferPtr (pEntBuffer) ;
	pBuffer->m_pNextBuffer	= pEntNextBuffer ;
	return	True ;
}

Boolean
lispBuffer_GetNext (
	register TLispEntity*			pEntBuffer,
	register TLispEntity** const	ppReturn)
{
	register TLispBuffer*	pBuffer ;

	assert (pEntBuffer != NULL) ;
	assert (pEntBuffer->m_iType == LISPENTITY_BUFFER) ;
	assert (ppReturn != NULL) ;

	pBuffer		= lispEntity_GetBufferPtr (pEntBuffer) ;
	*ppReturn	= pBuffer->m_pNextBuffer ;
	return	True ;
}

Boolean
lispBuffer_SetPrevious (
	register TLispEntity*	pEntBuffer,
	register TLispEntity*	pEntPrevBuffer)
{
	register TLispBuffer*	pBuffer ;

	assert (pEntBuffer != NULL) ;
	assert (pEntBuffer->m_iType == LISPENTITY_BUFFER) ;
	assert ((pEntPrevBuffer != NULL &&
			 pEntPrevBuffer->m_iType == LISPENTITY_BUFFER) ||
			(pEntPrevBuffer == NULL)) ;

	pBuffer	= lispEntity_GetBufferPtr (pEntBuffer) ;
	pBuffer->m_pPrevBuffer	= pEntPrevBuffer ;
	return	True ;
}

Boolean
lispBuffer_GetPrevious (
	register TLispEntity*			pEntBuffer,
	register TLispEntity** const	ppReturn)
{
	register TLispBuffer*	pBuffer ;

	assert (pEntBuffer != NULL) ;
	assert (pEntBuffer->m_iType == LISPENTITY_BUFFER) ;
	assert (ppReturn != NULL) ;

	pBuffer	= lispEntity_GetBufferPtr (pEntBuffer) ;
	*ppReturn	= pBuffer->m_pPrevBuffer ;
	return	True ;
}

Boolean
lispBuffer_MakeSymbolValue (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntBuffer,
	register TLispEntity*	pSymbol)
{
	register TLispBuffer*	pBuffer ;

	assert (pLispMgr != NULL) ;
	assert (pEntBuffer  != NULL) ;
	assert (pSymbol  != NULL) ;
	
	assert (pEntBuffer->m_iType == LISPENTITY_BUFFER) ;
	pBuffer	= lispEntity_GetBufferPtr (pEntBuffer) ;

	return	lispBindTable_MakeEntry (pLispMgr, pBuffer->m_apVariableTable, NELEMENTS (pBuffer->m_apVariableTable), pSymbol, NULL) ;
}

/*
 *	˥ܥ櫓ˤϤʤġ
 */
Boolean
lispBuffer_SetSymbolValue (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntBuffer,
	register TLispEntity*	pSymbol,
	register TLispEntity*	pValue)
{
	register TLispBuffer*	pBuffer ;

	assert (pLispMgr   != NULL) ;
	assert (pEntBuffer != NULL) ;
	assert (pSymbol    != NULL) ;
	/*assert (hValue  != NULL) ;*/

	assert (pEntBuffer->m_iType == LISPENTITY_BUFFER) ;
	pBuffer	= lispEntity_GetBufferPtr (pEntBuffer) ;

	return	lispBindTable_SetEntryValue (pLispMgr, pBuffer->m_apVariableTable, NELEMENTS (pBuffer->m_apVariableTable), pSymbol, pValue) ;
}

Boolean
lispBuffer_SetSymbolValueWithName (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntBuffer,
	register const Char*	pName,
	register const int		nName,
	register TLispEntity*	pValue)
{
	register TLispBuffer*	pBuffer ;
	TLispEntity*	pEntSymbol ;

	assert (pLispMgr   != NULL) ;
	assert (pEntBuffer != NULL) ;
	assert (pName != NULL && nName > 0) ;
	/*assert (hValue  != NULL) ;*/

	if (TFAILED (lispMgr_InternSymbol (pLispMgr, pName, nName, &pEntSymbol)))
		return	False ;

	assert (pEntBuffer->m_iType == LISPENTITY_BUFFER) ;
	pBuffer	= lispEntity_GetBufferPtr (pEntBuffer) ;

	return	lispBindTable_SetEntryValue (pLispMgr, pBuffer->m_apVariableTable, NELEMENTS (pBuffer->m_apVariableTable), pEntSymbol, pValue) ;
}

Boolean
lispBuffer_GetSymbolValue (
	register TLispManager*			pLispMgr,
	register TLispEntity*			pEntBuffer,
	register TLispEntity*			pSymbol,
	register TLispEntity** const	ppReturn)
{
	register TLispBuffer*	pBuffer ;

	assert (pLispMgr   != NULL) ;
	assert (pEntBuffer != NULL) ;
	assert (pSymbol    != NULL) ;
	/*assert (hValue  != NULL) ;*/

	assert (pEntBuffer->m_iType == LISPENTITY_BUFFER) ;
	pBuffer	= lispEntity_GetBufferPtr (pEntBuffer) ;
	return	lispBindTable_GetEntryValue (pLispMgr, pBuffer->m_apVariableTable, NELEMENTS (pBuffer->m_apVariableTable), pSymbol, ppReturn) ;
}

Boolean
lispBuffer_GetSymbolValueWithName (
	register TLispManager*			pLispMgr,
	register TLispEntity*			pEntBuffer,
	register const Char*			pName,
	register const int				nName,
	register TLispEntity** const	ppReturn)
{
	register TLispBuffer*	pBuffer ;
	TLispEntity*	pEntSymbol ;

	assert (pLispMgr   != NULL) ;
	assert (pEntBuffer != NULL) ;
	assert (pName != NULL && nName > 0) ;
	/*assert (hValue  != NULL) ;*/

	assert (pEntBuffer->m_iType == LISPENTITY_BUFFER) ;

	if (TFAILED (lispMgr_InternSymbol (pLispMgr, pName, nName, &pEntSymbol)))
		return	False ;

	pBuffer	= lispEntity_GetBufferPtr (pEntBuffer) ;
	return	lispBindTable_GetEntryValue (pLispMgr, pBuffer->m_apVariableTable, NELEMENTS (pBuffer->m_apVariableTable), pEntSymbol, ppReturn) ;
}


/*
 *	ޡХåեδ֤ͳϡХåե˴
 *	ΥХåեؤ³ޡ¸ߤΤϤޤ顣
 *	ΥХåեؤƤޡ˴褦ˤʤȤʤ
 *	ΤǤϤʤȡ
 *	ȡΥХåե°ƤޡȤǰưʤ
 *	Ȥʤ顣
 */
Boolean
lispBuffer_AddMarker (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntBuffer,
	register TLispEntity*	pEntMarker)
{
	TLispEntity*	pEntPrevBuffer ;
	register TLispBuffer*	pBuffer ;
	
	assert (pLispMgr      != NULL) ;
	assert (pEntBuffer != NULL) ;
	assert (pEntMarker != NULL) ;
	assert (pEntBuffer->m_iType == LISPENTITY_BUFFER) ;

	/*	ޡȤƤȴФ*/
	if (TFAILED (lispMarker_GetBufferPosition (pLispMgr, pEntMarker, &pEntPrevBuffer, NULL))) {
#if defined (DEBUG)
		fprintf (stderr, "[buffer] This marker has no buffer.\n") ;
#endif
		return	False ;
	}

	/*	ˤΥХåեˤĤʤƤˤʤ*/
	if (pEntPrevBuffer == pEntBuffer) {
#if defined (DEBUG)
		fprintf (stderr, "[buffer] This marker has already been inserted.\n") ;
#endif
		return	True ;
	}
	if (pEntPrevBuffer != NULL) 
		lispBuffer_RemoveMarker (pLispMgr, pEntMarker) ;
	/*	ޡΥХåեȰ֤ѹ롣*/
	lispMarker_SetBufferPosition (pLispMgr, pEntMarker, pEntBuffer, 1) ;
	
	pBuffer	= lispEntity_GetBufferPtr (pEntBuffer) ;
	/*	ХåեΥޡ󥯤Ĥʤľ*/
	if (pBuffer->m_lstMarkers == NULL) {
		pBuffer->m_lstMarkers	= pEntMarker ;
		lispMarker_SetNext (pLispMgr, pEntMarker, NULL) ;
	} else {
		lispMarker_SetPrevious (pLispMgr, pBuffer->m_lstMarkers, pEntMarker) ;
		lispMarker_SetNext (pLispMgr, pEntMarker, pBuffer->m_lstMarkers) ;
		pBuffer->m_lstMarkers	= pEntMarker ;
	}
	lispMarker_SetPrevious (pLispMgr, pEntMarker, NULL) ;
	return	True ;
}

Boolean
lispBuffer_RemoveMarker (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntMarker)
{
	TLispEntity*	pEntBuffer ;
	TLispEntity*	pEntPrevMarker ;
	TLispEntity*	pEntNextMarker ;
	register TLispBuffer*	pBuffer ;
	
	assert (pLispMgr      != NULL) ;
	assert (pEntMarker != NULL) ;

	/*	ޡȤƤȴФ*/
	if (TFAILED (lispMarker_GetBufferPosition (pLispMgr, pEntMarker, &pEntBuffer, NULL)) ||
		pEntBuffer == NULL)
		return	False ;
	
	/*	ޡΥХåեȰ֤ѹ롣*/
	lispMarker_SetBufferPosition (pLispMgr, pEntMarker, NULL, 0) ;
	
	pBuffer	= lispEntity_GetBufferPtr (pEntBuffer) ;
	/*	ХåեΥޡ󥯤Ĥʤľ*/
	lispMarker_GetPrevious (pLispMgr, pEntMarker, &pEntPrevMarker) ;
	lispMarker_GetNext (pLispMgr, pEntMarker, &pEntNextMarker) ;
	
	if (pEntPrevMarker != NULL) {
		lispMarker_SetNext (pLispMgr, pEntPrevMarker, pEntNextMarker) ;
	} else {
		pBuffer->m_lstMarkers	= pEntNextMarker ;
	}
	if (pEntNextMarker != NULL) {
		lispMarker_SetPrevious (pLispMgr, pEntNextMarker, pEntPrevMarker) ;
	}
	return	True ;
}

Boolean
lispBuffer_SetKeymap (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntBuffer,
	register TLispEntity*	pEntKeymap)
{
	register TLispBuffer*	pBuffer ;
	assert (pLispMgr       != NULL) ;
	assert (pEntBuffer     != NULL) ;
	assert (pEntKeymap     != NULL) ;
	lispEntity_AddRef (pLispMgr, pEntKeymap) ;
	pBuffer	= lispEntity_GetBufferPtr (pEntBuffer) ;
	if (pBuffer->m_pEntKeymap != NULL)
		lispEntity_Release (pLispMgr, pBuffer->m_pEntKeymap) ;
	pBuffer->m_pEntKeymap	= pEntKeymap ;
	return	True ;
}

Boolean
lispBuffer_GetKeymap (
	register TLispEntity*	pEntBuffer,
	register TLispEntity**	ppEntKeymapRet)
{
	register TLispBuffer*	pBuffer ;
	assert (pEntBuffer     != NULL) ;
	assert (ppEntKeymapRet != NULL) ;
	pBuffer	= lispEntity_GetBufferPtr (pEntBuffer) ;
	*ppEntKeymapRet	= pBuffer->m_pEntKeymap ;
	return	True ;
}

Boolean
lispBuffer_SetPrompt (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntBuffer,
	register const Char*	pString,
	register int			nString)
{
	register TLispBuffer*	pBuffer ;
	int						nBufferTop, nBufferEnd ;
	TBufStringMarker		mk ;

	assert (pLispMgr   != NULL) ;
	assert (pEntBuffer != NULL) ;
	assert (pString    != NULL || nString == 0) ;

	if (nString <= 0)
		return	True ;

	lispBuffer_PointBufferTop (pLispMgr, pEntBuffer, &nBufferTop) ; 
	lispBuffer_PointBufferEnd (pLispMgr, pEntBuffer, &nBufferEnd) ;

	pBuffer	= lispEntity_GetBufferPtr (pEntBuffer) ;
	if (TFAILED (TBufString_SolveMarker (&pBuffer->m_Text, nBufferTop - 1, &mk))) {
#if defined (DEBUG)
		fprintf (stderr, "lispBuffer_SetPrompt () : SolveMarker error!\n") ;
#endif
		return	False ;
	}
	if (TFAILED (TBufStringMarker_InsertString (&mk, pString, nString))) {
#if defined (DEBUG)
		fprintf (stderr, "lispBuffer_SetPrompt () : InsertString error!\n") ;
#endif
		return	False ;
	}
	/*	ư롣*/
	lispBuffer_MoveMarkerWithInsertion (pLispMgr, pEntBuffer, nBufferTop - 1, nString) ;
	pBuffer->m_nTick	++ ;
	return	True ;
}

/*
 *	ޡΥΰ֤֤
 */
Boolean
lispBuffer_Point (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntBuffer,
	register int*			pnPoint)
{
	return	lispBuffer_getMarkerPosition (pLispMgr, pEntBuffer, MARKER_POINT, pnPoint) ;
}

Boolean
lispBuffer_PointBufferEnd (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntBuffer,
	register int*			pnPoint)
{
	return	lispBuffer_getMarkerPosition (pLispMgr, pEntBuffer, MARKER_BLAST, pnPoint) ;
}

/*
 *	ԽǽΰƬ֤֤
 */
Boolean
lispBuffer_PointMax (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntBuffer,
	register int*			pnPoint)
{
	return	lispBuffer_getMarkerPosition (pLispMgr, pEntBuffer, MARKER_ELAST, pnPoint) ;
}

Boolean
lispBuffer_PointBufferTop (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntBuffer,
	register int*			pnPoint)
{
	return	lispBuffer_getMarkerPosition (pLispMgr, pEntBuffer, MARKER_BTOP, pnPoint) ;
}

/*
 *	Խǽΰκǽ֤֤
 */
Boolean
lispBuffer_PointMin (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntBuffer,
	register int*			pnPoint)
{
	return	lispBuffer_getMarkerPosition (pLispMgr, pEntBuffer, MARKER_ETOP, pnPoint) ;
}

Boolean
lispBuffer_PointMarker (
	register TLispManager*			pLispMgr,
	register TLispEntity*			pEntBuffer,
	register TLispEntity** const	ppMarkerReturn)
{
	assert (pLispMgr       != NULL) ;
	assert (pEntBuffer  != NULL) ;
	assert (ppMarkerReturn != NULL) ;
	*ppMarkerReturn	= lispBuffer_getMarker (pEntBuffer, MARKER_POINT) ;
	return	True ;
}

/*
 *	ޡΰ֤֤
 */
Boolean
lispBuffer_MarkMarker (
	register TLispManager*			pLispMgr,
	register TLispEntity*			pEntBuffer,
	register TLispEntity** const	ppMarkerReturn)
{
	assert (pLispMgr       != NULL) ;
	assert (pEntBuffer  != NULL) ;
	assert (ppMarkerReturn != NULL) ;
	*ppMarkerReturn	= lispBuffer_getMarker (pEntBuffer, MARKER_MARK) ;
	return	True ;
}

Boolean
lispBuffer_SetName (
	register TLispManager*		pLispMgr,
	register TLispEntity*		pEntBuffer,
	register TLispEntity*		pEntName)
{
	register TLispBuffer*	pBuffer ;
	assert (pLispMgr   != NULL) ;
	assert (pEntBuffer != NULL) ;
	assert (pEntBuffer->m_iType == LISPENTITY_BUFFER) ;

	pBuffer	= lispEntity_GetBufferPtr (pEntBuffer) ;
	if (pBuffer->m_pEntName != NULL) 
		lispEntity_Release (pLispMgr, pBuffer->m_pEntName) ;
	pBuffer->m_pEntName	= pEntName ;
	if (pEntName != NULL)
		lispEntity_AddRef (pLispMgr, pEntName) ;
	return	True ;
}

Boolean
lispBuffer_GetName (
	register TLispManager*		pLispMgr,
	register TLispEntity*		pEntBuffer,
	register TLispEntity**		ppEntReturn)
{
	register TLispBuffer*	pBuffer ;
	assert (pLispMgr   != NULL) ;
	assert (pEntBuffer != NULL) ;
	assert (pEntBuffer->m_iType == LISPENTITY_BUFFER) ;

	pBuffer			= lispEntity_GetBufferPtr (pEntBuffer) ;
	*ppEntReturn	= pBuffer->m_pEntName ;
	return	True ;
}

/*
 *	Хåե/ԽƤƥȤȴФΥޡ롣
 *	ɥˤϥޡɬפʤΤǡ
 */
Boolean
lispBuffer_GetString (
	register TLispManager*		pLispMgr,
	register TLispEntity*		pEntBuffer,
	register TBufStringMarker*	pMarker,
	register int* const			pnLength)
{
	register TLispBuffer*	pBuffer ;
	int				nTop, nLast ;

	assert (pLispMgr      != NULL) ;
	assert (pEntBuffer != NULL) ;
	assert (pMarker       != NULL) ;
	assert (pnLength      != NULL) ;
	assert (pEntBuffer->m_iType == LISPENTITY_BUFFER) ;
	
	/*	ХåեƬǸޤǤʸȤȴФnarrowing 
	 *	̤դ뤳ȡ*/
	if (TFAILED (lispBuffer_getMarkerPosition (pLispMgr, pEntBuffer, MARKER_BLAST, &nLast)) ||
		TFAILED (lispBuffer_getMarkerPosition (pLispMgr, pEntBuffer, MARKER_BTOP,  &nTop)))
		return	False ;
	assert (nTop > 0 && nLast > 0) ;
	assert (nTop <= nLast) ;
	
	pBuffer		= lispEntity_GetBufferPtr (pEntBuffer) ;
	if (TFAILED (TBufString_SolveMarker (&pBuffer->m_Text, nTop - 1, pMarker)))
		return	False ;
	*pnLength	= nLast - nTop ;
	return	True ;
}

/*
 *	ХåեԽǽǤʤꥢ˴ޤޤʸ(ץץȤȤ)
 *	ޤȡʸ֤Narrowing αƶʤ
 *()
 *	ХåեκǸ˲äĤƤˤϺμǤбǤ
 *	ȡʸ򥫥ȤƤʤΤǡ
 */
Boolean
lispBuffer_GetFullString (
	register TLispManager*		pLispMgr,
	register TLispEntity*		pEntBuffer,
	register TBufStringMarker*	pMarker,
	register int* const			pnLength)
{
	register TLispBuffer*	pBuffer ;
	int				nLast ;

	assert (pLispMgr      != NULL) ;
	assert (pEntBuffer != NULL) ;
	assert (pMarker       != NULL) ;
	assert (pnLength      != NULL) ;
	assert (pEntBuffer->m_iType == LISPENTITY_BUFFER) ;
	
	/*	ХåեƬǸޤǤʸȤȴФnarrowing 
	 *	̤դ뤳ȡ*/
	if (TFAILED (lispBuffer_getMarkerPosition (pLispMgr, pEntBuffer, MARKER_BLAST, &nLast)))
		return	False ;
	pBuffer		= lispEntity_GetBufferPtr (pEntBuffer) ;
	if (TFAILED (TBufString_SolveMarker (&pBuffer->m_Text, 0, pMarker)))
		return	False ;
	*pnLength	= nLast - 1 ;	/* marker  1  - 1 롣*/
	return	True ;
}

/*
 *	ХåեԽǽΰ nStartPos  nEndPos ˸ꤹ롣
 *	ХåեΤȤȤ礭
 */
Boolean
lispBuffer_Narrow (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntBuffer,
	register int			nStartPos,
	register int			nEndPos)
{
	register TLispBuffer*	pBuffer ;
	int				nBufferTop, nBufferLast ;
	
	assert (pLispMgr      != NULL) ;
	assert (pEntBuffer != NULL) ;
	assert (pEntBuffer->m_iType == LISPENTITY_BUFFER) ;
	
	/*	narrowing ΰγϰ֤Ͻλ֤꾮äߤ*/
	if (nStartPos > nEndPos)
		return	False ;

	/*	ХåեϰϤ롣*/
	if (TFAILED (lispBuffer_getMarkerPosition (pLispMgr, pEntBuffer, MARKER_BTOP,  &nBufferTop)) ||
		TFAILED (lispBuffer_getMarkerPosition (pLispMgr, pEntBuffer, MARKER_BLAST, &nBufferLast)))
		return	False ;
	/*	narrowing ϰϤХåեγä饨顼ˤ롣*/
	if (nStartPos < nBufferTop || nEndPos > nBufferLast)
		return	False ;
	
	/*	Խǽΰ򤢤魯ޡư롣*/
	pBuffer		= lispEntity_GetBufferPtr (pEntBuffer) ;
	if (TFAILED (lispMarker_SetBufferPosition (pLispMgr, pBuffer->m_apMustMarkers [MARKER_ETOP],  pEntBuffer, nStartPos)) ||
		TFAILED (lispMarker_SetBufferPosition (pLispMgr, pBuffer->m_apMustMarkers [MARKER_ELAST], pEntBuffer, nEndPos)))
		return	False ;
	pBuffer->m_nTick	++ ;
	return	True ;
}

Boolean
lispBuffer_Widen (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntBuffer)
{
	register TLispBuffer*	pBuffer ;
	int				nBufferTop, nBufferLast ;
	
	assert (pLispMgr      != NULL) ;
	assert (pEntBuffer != NULL) ;
	assert (pEntBuffer->m_iType == LISPENTITY_BUFFER) ;
	
	/*	ХåեϰϤ롣*/
	if (TFAILED (lispBuffer_getMarkerPosition (pLispMgr, pEntBuffer, MARKER_BTOP,  &nBufferTop)) ||
		TFAILED (lispBuffer_getMarkerPosition (pLispMgr, pEntBuffer, MARKER_BLAST, &nBufferLast)))
		return	False ;

	/*	Խǽΰ򤢤魯ޡХåեƬȺǸ˰ư롣*/
	pBuffer		= lispEntity_GetBufferPtr (pEntBuffer) ;
	if (TFAILED (lispMarker_SetBufferPosition (pLispMgr, pBuffer->m_apMustMarkers [MARKER_ETOP],  pEntBuffer, nBufferTop)) ||
		TFAILED (lispMarker_SetBufferPosition (pLispMgr, pBuffer->m_apMustMarkers [MARKER_ELAST], pEntBuffer, nBufferLast)))
		return	False ;
	pBuffer->m_nTick	++ ;
	return	True ;
}

Boolean
lispBuffer_InsertFileContents (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntBuffer,
	register int			nPosition,
	register KFILE*			pFile,
	register int*			pnReadCount)
{
	register TLispBuffer*	pBuffer ;
	int					nEditLast, nEditTop, nBufferLast ;
	TBufStringMarker	mk ;
	register Char		cc ;
	register int		nReadCount	= 0 ;

	assert (pLispMgr   != NULL) ;
	assert (pEntBuffer != NULL) ;
	assert (pEntBuffer->m_iType == LISPENTITY_BUFFER) ;
	assert (pFile      != NULL) ;

	if (TFAILED (lispBuffer_getMarkerPosition (pLispMgr, pEntBuffer, MARKER_ETOP,  &nEditTop)) ||
		TFAILED (lispBuffer_getMarkerPosition (pLispMgr, pEntBuffer, MARKER_ELAST, &nEditLast)) ||
		TFAILED (lispBuffer_getMarkerPosition (pLispMgr, pEntBuffer, MARKER_BLAST, &nBufferLast)) ||
		nEditTop > nPosition || nPosition > nEditLast) {
#if defined (DEBUG)
		fprintf (stderr, "[buffer] Bad Marker Position.\n") ;
#endif
		return	False ;
	}

	assert (nEditTop  <= nEditLast) ;
	assert (nEditLast <= nBufferLast) ;

	pBuffer		= lispEntity_GetBufferPtr (pEntBuffer) ;
	if (TFAILED (TBufString_SolveMarker (&pBuffer->m_Text, nPosition - 1, &mk)))
		return	False ;

	while (cc = KFile_Getc (pFile), cc != EOF) {
		if (TFAILED (TBufStringMarker_InsertChar (&mk, cc, 1)))
			break ;
		nReadCount	++ ;
	}

	/*	ޡưɬפ롣*/
	lispBuffer_MoveMarkerWithInsertion (pLispMgr, pEntBuffer, nPosition, nReadCount) ;
	pBuffer->m_fModified	= True ;
	pBuffer->m_nTick	++ ;

	if (pnReadCount != NULL)
		*pnReadCount			= nReadCount ;
	return	(cc == EOF)? True : False ;
}

/*
 *	Хåեʸ nCount 롣
 */
Boolean
lispBuffer_InsertChar (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntBuffer,
	register int			nPosition,
	register const Char		cc,
	register int			nCount)
{
	register TLispBuffer*	pBuffer ;
	int				nEditLast, nEditTop, nBufferLast ;
	TBufStringMarker	mk ;
	
	assert (pLispMgr      != NULL) ;
	assert (pEntBuffer != NULL) ;
	assert (pEntBuffer->m_iType == LISPENTITY_BUFFER) ;

	if (nCount <= 0)
		return	True ;
	
	if (TFAILED (lispBuffer_getMarkerPosition (pLispMgr, pEntBuffer, MARKER_ETOP,  &nEditTop)) ||
		TFAILED (lispBuffer_getMarkerPosition (pLispMgr, pEntBuffer, MARKER_ELAST, &nEditLast)) ||
		TFAILED (lispBuffer_getMarkerPosition (pLispMgr, pEntBuffer, MARKER_BLAST, &nBufferLast))) {
#if defined (DEBUG)
		fprintf (stderr, "[buffer] Bad Marker Position.\n") ;
#endif
		return	False ;
	}
	
	assert (nEditTop  <= nEditLast) ;
	assert (nEditLast <= nBufferLast) ;
	
	pBuffer		= lispEntity_GetBufferPtr (pEntBuffer) ;
	if (nEditTop <= nPosition && nPosition <= nEditLast) {
		if (TFAILED (TBufString_SolveMarker (&pBuffer->m_Text, nPosition - 1, &mk)))
			return	False ;
		if (TFAILED (TBufStringMarker_InsertChar (&mk, cc, nCount)))
			return	False ;

		/*	ޡưɬפ롣*/
		lispBuffer_MoveMarkerWithInsertion (pLispMgr, pEntBuffer, nPosition, nCount) ;
		pBuffer->m_fModified	= True ;
		pBuffer->m_nTick	++ ;
	}
	return	True ;
}

/*
 *	 delete-char ϥХåե narrow 
 */
Boolean
lispBuffer_DeleteChar (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntBuffer,
	register int			nPosition,
	register int			nCount)
{
	register TLispBuffer*	pBuffer ;
	int						nEditTop, nEditLast ;
	TBufStringMarker	mk ;
	
	assert (pLispMgr      != NULL) ;
	assert (pEntBuffer != NULL) ;
	assert (pEntBuffer->m_iType == LISPENTITY_BUFFER) ;
	
	if (TFAILED (lispBuffer_getMarkerPosition (pLispMgr, pEntBuffer, MARKER_ETOP,  &nEditTop)) ||
		TFAILED (lispBuffer_getMarkerPosition (pLispMgr, pEntBuffer, MARKER_ELAST, &nEditLast)))
		return	False ;
	
	assert (nEditTop  <= nEditLast) ;

	if (nEditTop > nPosition || (nPosition + nCount) > nEditLast)
		return	False ;
	if (nCount <= 0)
		return	True ;

	pBuffer	= lispEntity_GetBufferPtr (pEntBuffer) ;
	if (TFAILED (TBufString_SolveMarker (&pBuffer->m_Text, nPosition - 1, &mk)))
		return	False ;
	if (TFAILED (TBufStringMarker_DeleteChar (&mk, nCount)))
		return	False ;
	/*	ޡưɬפ롣*/
	lispBuffer_MoveMarkerWithDeletion (pLispMgr, pEntBuffer, nPosition, nCount) ;
	pBuffer->m_fModified	= True ;
	pBuffer->m_nTick	++ ;
	return	True ;
}

Boolean
lispBuffer_InsertString (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntBuffer,
	register int			nPosition,
	register const Char*	pString,
	register int			nLength)
{
	register TLispBuffer*	pBuffer ;
	int						nEditLast, nEditTop, nBufferLast ;
	TBufStringMarker		mk ;
	
	assert (pLispMgr   != NULL) ;
	assert (pEntBuffer != NULL) ;

	if (nLength <= 0)
		return	True ;

	assert (pString    != NULL) ;
	
	if (TFAILED (lispBuffer_getMarkerPosition (pLispMgr, pEntBuffer, MARKER_ETOP,  &nEditTop)) ||
		TFAILED (lispBuffer_getMarkerPosition (pLispMgr, pEntBuffer, MARKER_ELAST, &nEditLast)) ||
		TFAILED (lispBuffer_getMarkerPosition (pLispMgr, pEntBuffer, MARKER_BLAST, &nBufferLast)))
		return	False ;
	
	assert (nEditTop  <= nEditLast) ;
	assert (nEditLast <= nBufferLast) ;
	
#if defined (DEBUG)
	lispBuffer_Print (pLispMgr, pEntBuffer) ;
#endif
	pBuffer		= lispEntity_GetBufferPtr (pEntBuffer) ;
	if (nEditTop <= nPosition && nPosition <= nEditLast) {
		if (TFAILED (TBufString_SolveMarker (&pBuffer->m_Text, nPosition - 1, &mk)))
			return	False ;
		if (TFAILED (TBufStringMarker_InsertString (&mk, pString, nLength)))
			return	False ;

		/*	ޡưɬפ롣*/
		lispBuffer_MoveMarkerWithInsertion (pLispMgr, pEntBuffer, nPosition, nLength) ;
		pBuffer->m_fModified	= True ;
		pBuffer->m_nTick	++ ;
	}
#if defined (DEBUG)
	lispBuffer_Print (pLispMgr, pEntBuffer) ;
#endif
	return	True ;
}

/*	buffer  modified flag 򻲾Ȥ롣
 */
Boolean
lispBuffer_GetModifiedp (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntBuffer)
{
	register TLispBuffer*	pBuffer ;
	
	assert (pLispMgr != NULL) ;
	assert (pEntBuffer != NULL) ;

	pBuffer		= lispEntity_GetBufferPtr (pEntBuffer) ;
	return	pBuffer->m_fModified ;
}

/*	buffer  modified flag ꤹ롣modified flag ľܤϽǤʤ
 *	δؿͳ뤳ȡ
 */
void
lispBuffer_SetModifiedp (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntBuffer,
	register Boolean			fModified)
{
	register TLispBuffer*	pBuffer ;
	
	assert (pLispMgr != NULL) ;
	assert (pEntBuffer != NULL) ;

	pBuffer		= lispEntity_GetBufferPtr (pEntBuffer) ;
	pBuffer->m_fModified	= fModified ;
	pBuffer->m_nTick	++ ;
	return ;
}

/*
 *	ХåեƥȤ nPosition  nCount ʸ
 *	ޡư魯롣
 *	ºݤȤϤǤϲͤʤ
 */
Boolean
lispBuffer_MoveMarkerWithInsertion (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntBuffer,
	register int			nPosition,
	register int			nCount)
{
	register TLispBuffer*	pBuffer ;
	register TLispEntity*	pEntMarker ;
	TLispEntity*			pEntNextMarker ;
#if defined (DEBUG_LV99)
	register int			i = 0 ;
#endif

	pBuffer		= lispEntity_GetBufferPtr (pEntBuffer) ;
	pEntMarker	= pBuffer->m_lstMarkers ;
	
#if defined (DEBUG_LV99)
	fprintf (stderr, "move-marker-with-insertion (%d, %d)\n", nPosition, nCount) ;
#endif
	while (pEntMarker != NULL) {
#if defined (DEBUG_LV99)
		fprintf (stderr, "(%d) ", i ++) ;
#endif
		lispMarker_MoveWithInsertion (pLispMgr, pEntMarker, nPosition, nCount) ;
		lispMarker_GetNext (pLispMgr, pEntMarker, &pEntNextMarker) ;
		pEntMarker	= pEntNextMarker ;
	}
	return	True ;
}

/*
 *	ХåեƥȤ nPosition  nCount ʸ
 *	ޡư魯롣
 *	ºݤȤϤǤϲͤʤ
 */
Boolean
lispBuffer_MoveMarkerWithDeletion (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntBuffer,
	register int			nPosition,
	register int			nCount)
{
	TLispEntity*	pEntMarker ;
	TLispEntity*	pEntNextMarker ;
	TLispBuffer*	pBuffer ;

	pBuffer			= lispEntity_GetBufferPtr (pEntBuffer) ;
	pEntMarker	= pBuffer->m_lstMarkers ;
	
	/*	ХåեˤĤʤƤޡ֤˰ưġ
	 *	ˡǤϥޡοȯȤȤäƤ®٤롣
	 *	ޤȤʤޡϥͽʤΤǡġ*/
	while (pEntMarker != NULL) {
		lispMarker_MoveWithDeletion (pLispMgr, pEntMarker, nPosition, nCount) ;
		lispMarker_GetNext (pLispMgr, pEntMarker, &pEntNextMarker) ;
		pEntMarker	= pEntNextMarker ;
	}
	return	True ;
}

void
lispBuffer_Desync (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntBuffer)
{
	register TLispBuffer*	pBuffer ;
	
	assert (pLispMgr   != NULL) ;
	assert (pEntBuffer != NULL) ;

	pBuffer		= lispEntity_GetBufferPtr (pEntBuffer) ;
	pBuffer->m_nTick	++ ;
	return ;
}

/*
 *	ǥХåѤδؿ
 *	ХåեȤɽ롣
 */
Boolean
lispBuffer_Print (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntBuffer)
{
	TLispEntity*		pEntMarker ;
	TLispEntity*		pEntNextMarker ;
	TLispBuffer*		pBuffer ;
	KANJISTATEMACHINE	ksm ;
	int					n, i, nLength ;
	char				achBuf [16] ;
	static char*		apMarkerName []	= {
		"BTOP", "ETOP", "POINT", "ELAST", "BLAST", "MARK",
	} ;
	TBufStringMarker	mk ;
	Char				cc ;

	assert (pLispMgr   != NULL) ;
	assert (pEntBuffer != NULL) ;
	fprintf (stderr, "buffer(%lx) ---\n", (unsigned long)pEntBuffer) ;
	pBuffer		= lispEntity_GetBufferPtr (pEntBuffer) ;
	pEntMarker	= pBuffer->m_lstMarkers ;
	for (i = 0 ; i < NUM_BUFFER_MUST_MARKERS ; i ++) {
		fprintf (stderr, "(%5s) ", apMarkerName [i]) ;
		if (pBuffer->m_apMustMarkers [i] == NULL) {
			fprintf (stderr, "no-exist\n") ;
		} else {
			lispMarker_Print (pLispMgr, pBuffer->m_apMustMarkers [i]) ;
			fprintf (stderr, "\n") ;
		}
	}
	i			= 0 ;
	while (pEntMarker != NULL) {
		fprintf (stderr, "(%d) ", i ++) ;
		lispMarker_Print (pLispMgr, pEntMarker) ;
		fprintf (stderr, "\n") ;
		lispMarker_GetNext (pLispMgr, pEntMarker, &pEntNextMarker) ;
		pEntMarker	= pEntNextMarker ;
	}

	if (TFAILED (TBufString_SolveMarker (&pBuffer->m_Text, 0, &mk)))
		return	False ;

	lispBuffer_PointBufferEnd (pLispMgr, pEntBuffer, &nLength) ;
	nLength		-- ;
	fprintf (stderr, "buffer-string(%d) = \"", nLength) ;
	InitializeKanjiFiniteStateMachine (&ksm, KCODING_SYSTEM_SHIFTJIS) ;
	while (nLength > 0) {
		cc	= TBufStringMarker_GetChar (&mk) ;
		n	= RtransferKanjiFiniteStateMachine (&ksm, cc, achBuf) ;
		achBuf [n]	= '\0' ;
		fprintf (stderr, "%s", achBuf) ;
		TBufStringMarker_Forward (&mk, 1) ;
		nLength	-- ;
	}
	fprintf (stderr, "\"\n") ;
	return	True ;
}

/*	private functions */
/*
 *	Хåե˥ޡ(ɬ¸ߤ)դꤹ롣
 */
Boolean
lispBuffer_initialize (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntBuffer)
{
	register TLispBuffer*	pBuffer ;
	Boolean			afInsertionType [NUM_BUFFER_MUST_MARKERS - 1]	= {
		False,	/* buffer top */
		False,	/* edit   top */
		True,	/* point */
		True,	/* edit   end */
		True,	/* buffer end */
	} ;
	TLispEntity*	pEntMarker ;
	register int	i ;
	
	assert (pLispMgr      != NULL) ;
	assert (pEntBuffer != NULL) ;
	
	pBuffer			= lispEntity_GetBufferPtr (pEntBuffer) ;

	if (TFAILED (TBufString_Initialize (&pBuffer->m_Text)))
		return	False ;

	/*	Хåեɬ¸ߤޡ롣*/
	for (i = 0 ; i < NUM_BUFFER_MUST_MARKERS - 1 ; i ++) {
		if (TFAILED (lispMgr_CreateMarker (pLispMgr, &pEntMarker)))
			return	False ;
		lispEntity_AddRef (pLispMgr, pEntMarker) ;
		lispMarker_SetInsertionType (pLispMgr, pEntMarker, afInsertionType [i]) ;
		/*	ޡХåեδɲá*/
		lispBuffer_AddMarker (pLispMgr, pEntBuffer, pEntMarker) ;
		pBuffer->m_apMustMarkers [i]	= pEntMarker ;
	}
	/*	Хåեؤʤޡ롣*/
	if (TFAILED (lispMgr_CreateMarker (pLispMgr, &pEntMarker)))
		return	False ;
	lispEntity_AddRef (pLispMgr, pEntMarker) ;
	lispMarker_SetInsertionType (pLispMgr, pEntMarker, False) ;
	pBuffer->m_apMustMarkers [i]	= pEntMarker ;
	pBuffer->m_nTick	++ ;
	return	True ;
}

Boolean
lispBuffer_uninitialize (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntBuffer)
{
	register TLispBuffer*	pBuffer ;
	register TLispEntity*	pEntMarker ;
	register TLispEntity**	ppEntMustMarker ;
	register int			i ;

	pBuffer	= lispEntity_GetBufferPtr (pEntBuffer) ;

	/*	Хåեѿ롣*/
	lispBindTable_Destroy (pLispMgr, pBuffer->m_apVariableTable, SIZE_LISP_BIND_TABLE) ;
	pEntMarker	= pBuffer->m_lstMarkers ;
	/*	ɬܥޡʳä*/
	while (pEntMarker != NULL) {
		TLispEntity*	pEntNextMarker ;
		lispMarker_GetNext (pLispMgr, pEntMarker, &pEntNextMarker) ;
		lispBuffer_RemoveMarker (pLispMgr, pEntMarker) ;
		pEntMarker	= pEntNextMarker ;
	}
	/*	ɬܥޡ롣*/
	ppEntMustMarker	= pBuffer->m_apMustMarkers ;
	for (i = 0 ; i < NUM_BUFFER_MUST_MARKERS ; i ++) {
		if (*ppEntMustMarker != NULL) {
			lispEntity_Release (pLispMgr, *ppEntMustMarker) ;
			*ppEntMustMarker = NULL ;
		}
		ppEntMustMarker	++ ;
	}
	TBufString_Uninitialize (&pBuffer->m_Text) ;
	pBuffer->m_nTick	++ ;
	return	True ;
}


