/* # skkinput (Simple Kana-Kanji Input)
 *
 * This file is part of skkinput.
 * Copyright (C) 2002
 * Takashi SAKAMOTO (PXG01715@nifty.ne.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "local.h"
#include <stdio.h>
#include <assert.h>
#include "lmachinep.h"

static	TLMRESULT	lispMachineState_evalLambdaArgFin		(TLispMachine*) ;
static	TLMRESULT	lispMachineState_evalArgStep1			(TLispMachine*) ;
static	TLMRESULT	lispMachineState_evalArgStep2			(TLispMachine*) ;
static	TLMRESULT	lispMachineState_evalArgStep3			(TLispMachine*) ;
static	TLMRESULT	lispMachineState_mapcarListApply		(TLispMachine*) ;
static	TLMRESULT	lispMachineState_mapcarListPostApply	(TLispMachine*) ;
static	TLMRESULT	lispMachineState_mapcarVectorApply		(TLispMachine*) ;
static	TLMRESULT	lispMachineState_mapcarVectorPostApply	(TLispMachine*) ;
static	TLMRESULT	lispMachineState_mapcarStringApply		(TLispMachine*) ;
static	TLMRESULT	lispMachineState_mapcarStringPostApply	(TLispMachine*) ;
static	TLMRESULT	lispMachineState_mapcarPostApplyCommon	(TLispMachine*, TLMRESULT (*)(TLispMachine*)) ;
static	TLMRESULT	lispMachineState_mapcarFinalize			(TLispMachine*) ;
static	TLMRESULT	lispMachineState_runHooksStep			(TLispMachine*) ;
static	TLMRESULT	lispMachineState_runHooksStep2			(TLispMachine*) ;
static	TLMRESULT	lispMachineState_runHooksStep3			(TLispMachine*) ;
static	TLMRESULT	lispMachineState_runHooksFinalize		(TLispMachine*) ;

static	TLMRESULT	lispMachine_evalSubr					(TLispMachine*, TLispEntity*, TLispEntity*) ;
static	TLMRESULT	lispMachine_evalLambdaOrMacro			(TLispMachine*, TLispEntity*, TLispEntity*) ;

/*
 *	(eval FORM)
 *
 *	FORM ɾơ֤ͤ
 */
TLMRESULT
lispMachineState_Eval (
	register TLispMachine* pLM)
{
	TLispEntity*	pArglist ;
	TLispEntity*	pArg ;

	assert (pLM != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar (pLM->m_pLispMgr, pArglist, &pArg))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMachineCode_Evaln (pLM, pArg, &lispMachineState_ReturnOnly) ;
	return	LMR_CONTINUE ;
}

/*
 *	ǽΰؿȤƸƤӽФĤϰȤƤδؿϤ
 *	ؿ֤֤ͤ롣㤨С
 *	(funcall 'cons 'x 'y)  (x . y) ֤
 */
TLMRESULT
lispMachineState_Funcall (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*		pTarget ;
	TLispEntity*		pCar ;
	TLispEntity*		pFunc ;
	TLispEntity*		pArglist ;
	LMCMDINFO const*	pProcInfo ;
	int					nArg ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pTarget) ;
#if defined (DEBUG)
	fprintf (stderr, "funcall = ") ;
	lispEntity_Print (pLispMgr, pTarget) ;
	fprintf (stderr, "\n") ;
#endif
	if (TFAILED (lispEntity_GetCar (pLispMgr, pTarget, &pCar))) 
		goto	error ;
	if (TFAILED (lispMachine_GetSymbolFunctionValue (pLM, pCar, &pFunc))) 
		goto	error ;

	if (TSUCCEEDED (lispEntity_Symbolp (pLispMgr, pFunc))) {
		/*	symbol-fucntion  symbol ʤСˤɤɬפ롣
		 *	롼פˤʤäƤǽΤǾ¤ɬס*/
		if (TFAILED (lispMachine_GetFinalSymbolFunctionValue (pLM, pFunc, &pFunc))) 
			goto	error ;
	}
	if (TSUCCEEDED (lispEntity_Consp (pLispMgr, pFunc))) {
		TLispEntity*	pCdr ;
		assert (pFunc != NULL) ;
		
		if (TFAILED (lispEntity_GetCdr (pLispMgr, pTarget, &pCdr))) 
			goto	error ;
		if (TFAILED (lispMgr_CreateConscell (pLispMgr, pFunc, pCdr, &pTarget)) ||
			pTarget == NULL) 
			return	LMR_ERROR ;

		/*	defun ˤäƤؿξˤϡƤ
		 *	 (lambda) 򿷤ɾоݤѹ롣*/
		lispMachineCode_SetLReg   (pLM, LM_LREG_ACC, pTarget) ;
		if (TFAILED (lispMachine_GetSymbolFunctionValue (pLM, pFunc, &pFunc))) 
			goto	error ;
	}
	if (TFAILED (lispSubr_GetProc (pLispMgr, pFunc, &pProcInfo)) ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pTarget, &pArglist)) ||
		TFAILED (lispMachine_CheckArgument (pLM, pArglist, pProcInfo, &nArg))) 
		goto	error ;
	switch (pProcInfo->m_iArgtype) {
	case	LISPCMD_ARGTYPE_CDR:
	case	LISPCMD_ARGTYPE_MACRO:
		goto	error ;

	case	LISPCMD_ARGTYPE_LAMBDA:
	case	LISPCMD_ARGTYPE_SPECIAL:
		lispMachineCode_SetLReg   (pLM, LM_LREG_ACC, pTarget) ;
		break ;
	default:
		lispMachineCode_SetLReg   (pLM, LM_LREG_ACC, pArglist) ;
		lispMachineCode_SetInteractive (pLM, False) ;
		break ;
	}
	lispMachineCode_SetState (pLM, pProcInfo->m_pProc) ;
	return	LMR_CONTINUE ;
	
  error:
	lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

/*
 *	FUNCTION ĤΰǤäƸƤӽФǸΰ
 *	ϥꥹȤǤ롣FUNCTION ֤ͤapply ֤ͤǤ롣
 *	㤨С(apply '+ 1 2 '(3 4))  10 ֤
 */
TLMRESULT
lispMachineState_Apply (
	register TLispMachine*	pLM)
{
	TLispManager*	pLispMgr ;
	TLispEntity*	pList ;
	TLispEntity*	pTopofList ;
	TLispEntity*	pPrevList ;
	TLispEntity*	pLastArg ;
	
	assert (pLM != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pList) ;
	pTopofList	= pList ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
#if defined (DEBUG)
	fprintf (stderr, "apply => ") ;
	lispEntity_Print (pLispMgr, pTopofList) ;
	fprintf (stderr, "\n") ;
#endif
	
	pPrevList	= NULL ;
	for ( ; ; ) {
		TLispEntity*	pNextList ;
		if (TFAILED (lispEntity_GetCdr (pLispMgr, pList, &pNextList)) ||
			TSUCCEEDED (lispEntity_Nullp (pLispMgr, pNextList)))
			break ;
		pPrevList	= pList ;
		pList		= pNextList ;
	}
	if (TFAILED (lispEntity_GetCar (pLispMgr, pList, &pLastArg)) ||
		TFAILED (lispEntity_Listp  (pLispMgr, pLastArg))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (pPrevList != NULL) {
		lispEntity_SetCdr (pLispMgr, pPrevList, pLastArg) ;
	} else {
		pTopofList	= pLastArg ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pTopofList) ;
	lispMachineCode_SetState (pLM, &lispMachineState_Funcall) ;
#if defined (DEBUG)
	fprintf (stderr, "=> ") ;
	lispEntity_Print (pLispMgr, pTopofList) ;
	fprintf (stderr, "\n") ;
#endif
	return	LMR_CONTINUE ;
}

/*
 *	(run-hooks &rest HOOKS)
 *
 *	HOOKS  hook 줾¹Ԥ롣Major mode function 
 *	Ѥ롣Τ줾ϥܥ롢hook ѿǤ롣
 *	ܥϻꤷ֤˽롣⤷ hook ܥ뤬 non-nil
 *	ͤäƤСͤ function Ǥ뤫 function Υꥹ
 *	Ǥ롣
 *	⤷ͤ function ǤСʤ˸ƤФ롣⤷ꥹȤ
 *	СǤ֤˰̵˸ƤФ롣
 */
TLMRESULT
lispMachineState_RunHooks (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_1) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_2) ;
	lispMachineCode_SetLReg  (pLM, LM_LREG_1, pEntArglist) ;
	lispMachineCode_SetState (pLM, &lispMachineState_runHooksStep) ;
	return	LMR_CONTINUE ;
}

/*
 *	(fboundp SYMBOL)
 */
TLMRESULT
lispMachineState_Fboundp (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pEntArglist ;
	TLispEntity*			pEntFunc ;
	TLispEntity*			pEntSymbol ;
	TLispEntity*			pEntRetval ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntSymbol)) ||
		TFAILED (lispEntity_Symbolp (pLispMgr, pEntSymbol))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	/*	桼ؿˤʤĴ٤롣*/
	if (TSUCCEEDED (lispMachine_GetSymbolFunctionValue (pLM, pEntSymbol, &pEntFunc)) &&
		TFAILED (lispEntity_Voidp (pLispMgr, pEntFunc))) {
		lispMgr_CreateT   (pLispMgr, &pEntRetval) ;
	} else {
		lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*
 *	(symbol-function SYMBOL)
 *
 *	SYMBOL δؿ֤⤷ void ʤ Error ˤʤ롣
 */
TLMRESULT
lispMachineState_SymbolFunction (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pEntArglist ;
	TLispEntity*			pEntFunc ;
	TLispEntity*			pEntSymbol ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntSymbol)) ||
		TFAILED (lispEntity_Symbolp (pLispMgr, pEntSymbol))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	/*	桼ؿˤʤĴ٤롣*/
	if (TFAILED (lispMachine_GetSymbolFunctionValue (pLM, pEntSymbol, &pEntFunc)) ||
		TSUCCEEDED (lispEntity_Voidp (pLispMgr, pEntFunc))) {
		lispMachineCode_SetError (pLM) ;
	} else {
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntFunc) ;
	}
	return	LMR_RETURN ;
}

/*
 *	(mapcar FUNCTION SEQUENCE)
 *
 *	SEQUENCE  element ơ FUNCTION  apply result Υꥹ
 *	롣result  SEQUENCE ƱĹΥꥹȤˤʤ롣SEQUENCE 
 *	ꥹȡ٥ȥޤ bool-vector ޤ string Ǥ롣
 */
TLMRESULT
lispMachineState_Mapcar (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pFunction ;
	TLispEntity*	pSequence ;
	TLispEntity*	pFunclist ;
	TLispEntity*	pFunctail ;
	TLispEntity*	pNil ;
	int			iType ;
	
	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pFunction)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pArglist, &pSequence)) ||
		TFAILED (lispEntity_GetType (pLispMgr, pSequence, &iType))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	switch (iType) {
	case	LISPENTITY_CONSCELL:
		lispMachineCode_SetState (pLM, &lispMachineState_mapcarListApply) ;
		break ;
	case	LISPENTITY_VECTOR:
		lispMachineCode_SetState (pLM, &lispMachineState_mapcarVectorApply) ;
		break ;
	case	LISPENTITY_STRING:
		lispMachineCode_SetState (pLM, &lispMachineState_mapcarStringApply) ;
		break ;
	case	LISPENTITY_SYMBOL:
		if (TSUCCEEDED (lispEntity_Symbolp (pLispMgr, pSequence))) {
			lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pSequence) ;
			return	LMR_RETURN ;
		}
	case	LISPENTITY_BOOLVECTOR:
	default:
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	
	lispMachineCode_PushLReg (pLM, LM_LREG_1) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_2) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_3) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_4) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_5) ;
	
	lispMgr_CreateNil      (pLispMgr, &pNil) ;
	if (TFAILED (lispMgr_CreateConscell (pLispMgr, pNil, pNil, &pFunctail)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg  (pLM, LM_LREG_2, pFunctail) ;
	if (TFAILED (lispMgr_CreateConscell (pLispMgr, pFunction, pFunctail, &pFunclist)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg  (pLM, LM_LREG_1, pFunclist) ;
	lispMachineCode_SetLReg  (pLM, LM_LREG_3, pSequence) ;
	lispMachineCode_SetLReg  (pLM, LM_LREG_4, pNil) ;
	lispMachineCode_SetLReg  (pLM, LM_LREG_5, pNil) ;
	
	lispMachineCode_PushVReg (pLM, LM_VREG_1) ;
	lispMachineCode_SetVRegI (pLM, LM_VREG_1, 0) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_Evaluate (
	register TLispMachine* pLM)
{
	TLispEntity*		pTarget ;
	int			iType ;

	assert (pLM != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pTarget) ;
	assert (pTarget != NULL) ;
	assert (pLM->m_pLispMgr != NULL) ;

	if (TFAILED (lispEntity_GetType (pLM->m_pLispMgr, pTarget, &iType))) {
		lispMachineCode_SetError (pLM) ;
		/*	 State äƤΤǡCONTINUE Ǥ롣*/
		return	LMR_CONTINUE ;
	}

	switch (iType) {
	case	LISPENTITY_SYMBOL:
		lispMachineCode_PushState (pLM, pLM->m_pState) ;
		lispMachineCode_SetState  (pLM, &lispMachineState_EvalSymbol) ;
		return	LMR_CONTINUE ;

	case	LISPENTITY_CONSCELL:
		lispMachineCode_PushState (pLM, pLM->m_pState) ;
		lispMachineCode_SetState  (pLM, &lispMachineState_EvalCons) ;
		return	LMR_CONTINUE ;

	case	LISPENTITY_INTEGER:
	case	LISPENTITY_FLOAT:
	case	LISPENTITY_VECTOR:
	case	LISPENTITY_MARKER:
	case	LISPENTITY_BUFFER:
	case	LISPENTITY_STRING:
	default:
		return	LMR_CONTINUE ;
	}
}

TLMRESULT
lispMachineState_EvalSymbol (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pTarget ;
	TLispEntity*	pReturn ;

	assert (pLM      != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pTarget) ;

	if (TFAILED (lispMgr_SymbolRequireEvalp (pLispMgr, pTarget)))
		return	LMR_RETURN ;

	if (TFAILED (lispMachine_GetCurrentSymbolValue (pLM, pTarget, &pReturn)) ||
		TSUCCEEDED (lispEntity_Voidp (pLispMgr, pReturn))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pReturn) ;
	return	LMR_RETURN ;
}

/*
 *	EVAL 褦ȤƤоݤ ACC äƤΤȹͤ롣
 *	 ACC ֤ͤˤʤΤǡ̵¸ɬפϤʤɡ
 *	ɾƤʳ EVAL ɬפˤʤ뤫ɤƱ⡣
 */
TLMRESULT
lispMachineState_EvalCons (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pEntTarget ;
	TLispEntity*			pEntCar ;
	TLispEntity*			pEntFunc ;

	pLispMgr	= pLM->m_pLispMgr ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntTarget) ;

	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntTarget, &pEntCar)) ||
		TFAILED (lispMachine_GetSymbolFunctionValue (pLM, pEntCar, &pEntFunc))) {
#if defined (DEBUG) || 1
		fprintf (stderr, "Symbol's function value is void: ") ;
		lispEntity_Print (pLispMgr, pEntCar) ;
		fprintf (stderr, "\n") ;
#endif
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}

	if (TSUCCEEDED (lispEntity_Symbolp (pLispMgr, pEntFunc))) {
		/*	symbol-fucntion  symbol ʤСˤɤɬפ롣
		 *	롼פˤʤäƤǽΤǾ¤ɬס*/
		if (TFAILED (lispMachine_GetFinalSymbolFunctionValue (pLM, pEntFunc, &pEntFunc))) {
#if defined (DEBUG) || 1
			fprintf (stderr, "Symbol's function value is void: ") ;
			lispEntity_Print (pLispMgr, pEntFunc) ;
			fprintf (stderr, "\n") ;
#endif
			lispMachineCode_SetError (pLM) ;
			return	LMR_RETURN ;
		}
	}
	if (TSUCCEEDED (lispEntity_Consp (pLispMgr, pEntFunc))) {
		return	lispMachine_evalLambdaOrMacro (pLM, pEntTarget, pEntFunc) ;
	}
	return	lispMachine_evalSubr (pLM, pEntTarget, pEntFunc) ;
}

TLMRESULT
lispMachineState_ReturnOnly (
	register TLispMachine*	pLM)
{
	return	LMR_RETURN ;
}

/*	eval-cons private functions */
TLMRESULT
lispMachine_evalSubr (
	register TLispMachine*	pLM,
	register TLispEntity*	pEntTarget,
	register TLispEntity*	pEntFuncValue)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*			pEntArglist ;
	LMCMDINFO const*		pProcInfo	= NULL ;
	int						nArg ;

	/*	ؿ ``'' õФΥå򤹤롣*/
	if (TFAILED (lispSubr_GetProc (pLispMgr, pEntFuncValue, &pProcInfo)) ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pEntTarget, &pEntArglist)) ||
		TFAILED (lispMachine_CheckArgument (pLM, pEntArglist, pProcInfo, &nArg))) {
#if defined (DEBUG) || 1
		fprintf (stderr, "Wrong number of arguments: ") ;
		lispEntity_Print (pLispMgr, pEntFuncValue) ;
		fprintf (stderr, " %d\n", nArg) ;
#endif
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	switch (pProcInfo->m_iArgtype) {
		Boolean	fOrgInteractive ;
	case	LISPCMD_ARGTYPE_CDR:
		pEntTarget	= pEntArglist ;
	case	LISPCMD_ARGTYPE_SPECIAL:
	case	LISPCMD_ARGTYPE_MACRO:
		lispMachineCode_SetLReg   (pLM, LM_LREG_ACC, pEntTarget) ;
		/*	ɾüǤꡢ֤ɾ줿ꡢɾʤä
		 *	ꤹ롣*/
		lispMachineCode_SetState (pLM, pProcInfo->m_pProc) ;
		break ;

	case	LISPCMD_ARGTYPE_LAMBDA:
		lispMachineCode_SetLReg   (pLM, LM_LREG_ACC, pEntTarget) ;
		lispMachineCode_PushLReg  (pLM, LM_LREG_ACC) ;
		/*	ξˤϡ򤢤餫ɾƤȤɬפˤʤ롣*/
		lispMachineCode_SetLReg   (pLM, LM_LREG_ACC, pEntArglist) ;
		assert (pEntArglist != NULL) ;
		
		/*	ɾϿƤ interactive ºݤ˴ؿɾä
		 *	顢interactive  False ˤʤ롣*/
		lispMachineCode_GetInteractive (pLM, &fOrgInteractive) ;
		lispMachineCode_SetInteractive (pLM, False) ;
		lispMachineCode_PushState (pLM, pProcInfo->m_pProc) ;
		lispMachineCode_UnsetInteractive (pLM, fOrgInteractive) ;
		lispMachineCode_PushState (pLM, &lispMachineState_evalLambdaArgFin) ;
		lispMachineCode_SetState (pLM, &lispMachineState_evalArgStep1) ;
		break ;

	default:
		/*	ξˤϡ򤢤餫ɾƤȤɬפˤʤ롣*/
		lispMachineCode_SetLReg   (pLM, LM_LREG_ACC, pEntArglist) ;
		assert (pEntArglist != NULL) ;
		
		/*	ɾϿƤ interactive ºݤ˴ؿɾä
		 *	顢interactive  False ˤʤ롣*/
		lispMachineCode_GetInteractive (pLM, &fOrgInteractive) ;
		lispMachineCode_SetInteractive (pLM, False) ;
		lispMachineCode_PushState (pLM, pProcInfo->m_pProc) ;
		lispMachineCode_UnsetInteractive (pLM, fOrgInteractive) ;
		lispMachineCode_SetState (pLM, &lispMachineState_evalArgStep1) ;
		break ;
	}
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachine_evalLambdaOrMacro (
	register TLispMachine*	pLM,
	register TLispEntity*	pEntTarget,
	register TLispEntity*	pEntFuncValue)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*		pEntCdr ;
	TLispEntity*		pEntForm ;
	TLispEntity*		pEntSubr ;
	Boolean				fOrgInteractive ;
	LMCMDINFO const*	pProcInfo	= NULL ;
		
	if (TFAILED (lispEntity_GetCdr (pLispMgr, pEntTarget, &pEntCdr)))
		goto	error ;
	if (TFAILED (lispMgr_CreateConscell (pLispMgr, pEntFuncValue, pEntCdr, &pEntForm)) ||
		pEntForm == NULL) 
		return	LMR_ERROR ;
	
	/*	defun ˤäƤؿξˤϡƤ
	 *	 (lambda) 򿷤ɾоݤѹ롣*/
	lispMachineCode_SetLReg   (pLM, LM_LREG_ACC, pEntForm) ;
	if (TFAILED (lispMachine_GetSymbolFunctionValue (pLM, pEntFuncValue, &pEntSubr)) ||
		TFAILED (lispSubr_GetProc (pLispMgr, pEntSubr, &pProcInfo)))
		goto	error ;
#if defined (DEBUG)
	fprintf (stderr, "eval form = ") ;
	lispEntity_Print (pLispMgr, pEntForm) ;
	fprintf (stderr, "\n") ;
#endif
	if (pProcInfo->m_iArgtype != LISPCMD_ARGTYPE_LAMBDA) {
		lispMachineCode_SetState (pLM, pProcInfo->m_pProc) ;
	} else {
		lispMachineCode_PushLReg  (pLM, LM_LREG_ACC) ;
		/*	ξˤϡ򤢤餫ɾƤȤɬפˤʤ롣*/
		lispMachineCode_SetLReg   (pLM, LM_LREG_ACC, pEntCdr) ;
		assert (pEntCdr != NULL) ;
		
		/*	ɾϿƤ interactive ºݤ˴ؿɾä
		 *	顢interactive  False ˤʤ롣*/
		lispMachineCode_GetInteractive (pLM, &fOrgInteractive) ;
		lispMachineCode_SetInteractive (pLM, False) ;
		lispMachineCode_PushState (pLM, pProcInfo->m_pProc) ;
		lispMachineCode_UnsetInteractive (pLM, fOrgInteractive) ;
		lispMachineCode_PushState (pLM, &lispMachineState_evalLambdaArgFin) ;
		lispMachineCode_SetState (pLM, &lispMachineState_evalArgStep1) ;
	}
	return	LMR_CONTINUE ;

  error:
	lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_evalLambdaArgFin (
	register TLispMachine*	pLM)
{
	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) {
		lispMachineCode_PopLReg (pLM, LM_LREG_ACC) ;
	} else {
		TLispEntity*	pEntArglist ;
		TLispEntity*	pEntLambdaForm ;
		lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
		lispMachineCode_PopLReg (pLM, LM_LREG_ACC) ;
		lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntLambdaForm) ;
		lispEntity_SetCdr (pLM->m_pLispMgr, pEntLambdaForm, pEntArglist) ;
	}
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_evalArgStep1 (
	register TLispMachine*	pLM)
{
	TLispEntity*	pNil ;

	assert (pLM != NULL) ;

	lispMgr_CreateNil (pLM->m_pLispMgr, &pNil) ;
	assert (pLM->m_apLREGS [LM_LREG_ACC] != NULL) ;

	lispMachineCode_PushLReg (pLM, LM_LREG_1) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_2) ;
	lispMachineCode_SetLReg  (pLM, LM_LREG_1, pNil) ;
	lispMachineCode_SetLReg  (pLM, LM_LREG_2, pNil) ;
	lispMachineCode_SetState (pLM, &lispMachineState_evalArgStep2) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_evalArgStep2 (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pCAR ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	assert (pArglist != NULL) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pArglist))) {
		lispMachineCode_MoveLReg (pLM, LM_LREG_ACC, LM_LREG_1) ;
		/*	 PUSH Ƥ LREG1/2 ᤹
		 *	ARGLIST Ѥ¦ϥ쥸˲ʤ(ACCʳ)
		 *	ꤷƤ롣
		 *	ޤΤʤƤ*/
		lispMachineCode_PopLReg  (pLM, LM_LREG_2) ;
		lispMachineCode_PopLReg  (pLM, LM_LREG_1) ;
		return	LMR_RETURN ;
	}

	if (TFAILED (lispEntity_GetCar (pLispMgr, pArglist, &pCAR)) ||
		pCAR == NULL) {
		lispMachineCode_SetError (pLM) ;
		/*	 PUSH Ƥ LREG1/2 ᤹*/
		lispMachineCode_PopLReg (pLM, LM_LREG_2) ;
		lispMachineCode_PopLReg (pLM, LM_LREG_1) ;
		return	LMR_RETURN ;
	}
#if defined (DEBUG_LV99)
	lispMachine_ShowRegisterValue (pLM) ;
#endif
	lispMachineCode_PushLReg (pLM, LM_LREG_ACC) ;

	/*	EVAL  register ˲ʤƶ ACC ΤߤȤ롣*/
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pCAR) ;
#if defined (DEBUG_LV99)
	fprintf (stderr, "Eval-Arg: ") ;
	lispEntity_Print (pLM->m_pLispMgr, pCAR) ;
	fprintf (stderr, "\n") ;
#endif
	lispMachineCode_Evaln (pLM, pCAR, &lispMachineState_evalArgStep3) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_evalArgStep3 (
	register TLispMachine*	pLM)
{
	TLispManager*	pLispMgr ;
	TLispEntity*	pEvaledArglistTop ;
	TLispEntity*	pEvaledArglistLast ;
	TLispEntity*	pValue ;
	TLispEntity*	pNil ;
	TLispEntity*	pNewLast ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

#if defined (DEBUG_LV99)
	fprintf (stderr, "state = eval-arg-step3\n") ;
	lispMachine_ShowRegisterValue (pLM) ;
#endif
	/*	EVAL ǥ顼ȯСߤ롣*/
	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) {
		lispMachineCode_PopLReg (pLM, LM_LREG_ACC) ;
		/*	 PUSH Ƥ LREG1/2 ᤹*/
		lispMachineCode_PopLReg (pLM, LM_LREG_2) ;
		lispMachineCode_PopLReg (pLM, LM_LREG_1) ;
		return	LMR_RETURN ;
	}

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pValue) ;
	lispMgr_CreateNil (pLispMgr, &pNil) ;
	if (TFAILED (lispMgr_CreateConscell (pLispMgr, pValue, pNil, &pNewLast)) ||
		pNewLast == NULL) 
		return	LMR_ERROR ;

	lispMachineCode_GetLReg (pLM, LM_LREG_1,   &pEvaledArglistTop) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_2,   &pEvaledArglistLast) ;
	if (TFAILED (lispEntity_Nullp (pLispMgr, pEvaledArglistTop))) {
		lispEntity_SetCdr (pLispMgr, pEvaledArglistLast, pNewLast) ;
	} else {
		lispMachineCode_SetLReg (pLM, LM_LREG_1, pNewLast) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_2, pNewLast) ;
	lispMachineCode_PopLReg (pLM, LM_LREG_ACC) ;
	
	if (TFAILED (lispMachineCode_Cdr (pLM, LM_LREG_ACC, LM_LREG_ACC))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	} else {
		lispMachineCode_SetState (pLM, &lispMachineState_evalArgStep2) ;
		return	LMR_CONTINUE ;
	}
}

/*	run-hooks private functions */
TLMRESULT
lispMachineState_runHooksStep (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntHooks ;
	TLispEntity*	pEntHook ;
	TLispEntity*	pEntHookValue ;
	TLispEntity*	pEntNil ;
	TLispEntity*	pEntTemp ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_1, &pEntHooks) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntHooks))) 
		goto	finalize ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntHooks, &pEntHook)) ||
		TFAILED (lispEntity_Symbolp (pLispMgr, pEntHook))) 
		goto	error ;

	if (TFAILED (lispMachine_GetCurrentSymbolValue (pLM, pEntHook, &pEntHookValue)) ||
		TSUCCEEDED (lispEntity_Voidp (pLispMgr, pEntHookValue))) 
		goto	finalize ;

	lispMachineCode_SetState (pLM, &lispMachineState_runHooksStep2) ;
	if (TSUCCEEDED (lispEntity_Listp (pLispMgr, pEntHookValue))) {
		lispMachineCode_SetLReg (pLM, LM_LREG_2, pEntHookValue) ;
		goto	skip ;
	}
	if (TFAILED (lispMgr_CreateNil (pLispMgr, &pEntNil)) ||
		TFAILED (lispMgr_CreateConscell (pLispMgr, pEntHookValue, pEntNil, &pEntTemp)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_2, pEntTemp) ;

 skip:
	lispEntity_GetCdr (pLispMgr, pEntHooks, &pEntHooks) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_1, pEntHooks) ;
	return	LMR_CONTINUE ;

 error:
	lispMachineCode_SetError (pLM) ;
 finalize:
	lispMachineCode_SetState (pLM, &lispMachineState_runHooksFinalize) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_runHooksStep2 (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntHooks ;
	TLispEntity*	pEntCar ;
	TLispEntity*	pEntNil ;
	TLispEntity*	pEntTemp ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_2, &pEntHooks) ;
	assert (pEntHooks != NULL) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntHooks))) 
		goto	finalize ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntHooks, &pEntCar))) {
		lispMachineCode_SetError (pLM) ;
		goto	finalize ;
	}
	if (TSUCCEEDED (lispEntity_Tp (pLispMgr, pEntCar))) {
		lispMachineCode_SetState (pLM, &lispMachineState_runHooksStep3) ;
		return	LMR_CONTINUE ;
	}
	if (TFAILED (lispMgr_CreateNil (pLispMgr, &pEntNil)) ||
		TFAILED (lispMgr_CreateConscell (pLispMgr, pEntCar, pEntNil, &pEntTemp)))
		return	LMR_ERROR ;

	lispMachineCode_PushState (pLM, &lispMachineState_runHooksStep3) ;
	lispMachineCode_SetLReg   (pLM, LM_LREG_ACC, pEntTemp) ;
	lispMachineCode_SetState (pLM, &lispMachineState_EvalCons) ;
	return	LMR_CONTINUE ;

 finalize:
	lispMachineCode_SetState (pLM, &lispMachineState_runHooksFinalize) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_runHooksStep3 (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntHooks ;
	TLispEntity*	pEntCdr ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) {
		lispMachineCode_SetState (pLM, &lispMachineState_runHooksFinalize) ;
		return	LMR_CONTINUE ;
	}
	lispMachineCode_GetLReg (pLM, LM_LREG_2, &pEntHooks) ;
	assert (pEntHooks != NULL) ;
	if (TFAILED (lispEntity_GetCdr (pLispMgr, pEntHooks, &pEntCdr))) {
		lispMachineCode_SetError (pLM) ;
		lispMachineCode_SetState (pLM, &lispMachineState_runHooksFinalize) ;
	} else {
		lispMachineCode_SetLReg (pLM, LM_LREG_2, pEntCdr) ;
		lispMachineCode_SetState (pLM, &lispMachineState_runHooksStep2) ;
	}
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_runHooksFinalize (
	register TLispMachine*	pLM)
{
	lispMachineCode_MoveLReg (pLM, LM_LREG_ACC, LM_LREG_1) ;
	lispMachineCode_PopLReg  (pLM, LM_LREG_2) ;
	lispMachineCode_PopLReg  (pLM, LM_LREG_1) ;
	return	LMR_RETURN ;
}

/*	mapcar private functions */
TLMRESULT
lispMachineState_mapcarListApply (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pFunctail ;
	TLispEntity*	pFunclist ;
	TLispEntity*	pSequence ;
	TLispEntity*	pEntCar ;
	
	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_3, &pSequence) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pSequence))) {
		lispMachineCode_SetState (pLM, &lispMachineState_mapcarFinalize) ;
		return	LMR_CONTINUE ;
	}
	if (TFAILED (lispEntity_GetCar (pLispMgr, pSequence, &pEntCar))) {
		lispMachineCode_SetError (pLM) ;
		lispMachineCode_SetState (pLM, &lispMachineState_mapcarFinalize) ;
		return	LMR_CONTINUE ;
	}
	lispMachineCode_GetLReg   (pLM, LM_LREG_2,   &pFunctail) ;
	lispEntity_SetCar         (pLispMgr, pFunctail, pEntCar) ;
	lispMachineCode_GetLReg   (pLM, LM_LREG_1,   &pFunclist) ;
	
	lispMachineCode_PushState (pLM, &lispMachineState_mapcarListPostApply) ;
	lispMachineCode_Cdr       (pLM, LM_LREG_3,   LM_LREG_3) ;
	lispMachineCode_SetLReg   (pLM, LM_LREG_ACC, pFunclist) ;
	lispMachineCode_SetState (pLM, &lispMachineState_Funcall) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_mapcarListPostApply (
	register TLispMachine*	pLM)
{
	return	lispMachineState_mapcarPostApplyCommon (pLM, &lispMachineState_mapcarListApply) ;
}

TLMRESULT
lispMachineState_mapcarVectorApply (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pSequence ;
	TLispEntity*	pElement	= NULL ;
	TLispEntity*	pFunctail ;
	TLispEntity*	pFunclist ;
	long		lIndex ;
	
	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	
	lispMachineCode_GetLReg  (pLM, LM_LREG_3, &pSequence) ;
	lispMachineCode_GetVRegI (pLM, LM_VREG_1, &lIndex) ;
#if defined (DEBUG)
	fprintf (stderr, "Sequence = ") ;
	lispEntity_Print (pLispMgr, pSequence) ;
	fprintf (stderr, "\nIndex = %ld\n", lIndex) ;
#endif
	if (TFAILED (lispEntity_GetVectorElement (pLispMgr, pSequence, lIndex, &pElement)) ||
		pElement == NULL) {
		/*	pSequence  VECTOR ǤΤϳǧƤġ*/
		lispMachineCode_SetState (pLM, &lispMachineState_mapcarFinalize) ;
		return	LMR_CONTINUE ;
	}
	lIndex	++ ;
	lispMachineCode_SetVRegI  (pLM, LM_VREG_1,   lIndex) ;
	
	lispMachineCode_GetLReg   (pLM, LM_LREG_2,   &pFunctail) ;
	lispEntity_SetCar         (pLispMgr, pFunctail, pElement) ;
	lispMachineCode_GetLReg   (pLM, LM_LREG_1,   &pFunclist) ;
	
	lispMachineCode_PushState (pLM, &lispMachineState_mapcarVectorPostApply) ;
	lispMachineCode_SetLReg   (pLM, LM_LREG_ACC, pFunclist) ;
	lispMachineCode_SetState (pLM, &lispMachineState_Funcall) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_mapcarVectorPostApply (
	register TLispMachine*	pLM)
{
	return	lispMachineState_mapcarPostApplyCommon (pLM, &lispMachineState_mapcarVectorApply) ;
}

TLMRESULT
lispMachineState_mapcarStringApply (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pSequence ;
	TLispEntity*	pElement ;
	TLispEntity*	pFunctail ;
	TLispEntity*	pFunclist ;
	long			lIndex ;
	Char			cc ;
	
	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	
	lispMachineCode_GetLReg  (pLM, LM_LREG_3, &pSequence) ;
	lispMachineCode_GetVRegI (pLM, LM_VREG_1, &lIndex) ;
	if (TFAILED (lispEntity_GetStringElement (pLispMgr, pSequence, lIndex, &cc))) {
		/*	pSequence  VECTOR ǤΤϳǧƤġ*/
		lispMachineCode_SetState (pLM, &lispMachineState_mapcarFinalize) ;
		return	LMR_CONTINUE ;
	}
	if (TFAILED (lispMgr_CreateInteger (pLispMgr, cc, &pElement)))
		return	LMR_ERROR ;

	lIndex	++ ;
	lispMachineCode_SetVRegI  (pLM, LM_VREG_1,   lIndex) ;
	
	lispMachineCode_GetLReg   (pLM, LM_LREG_2,   &pFunctail) ;
	lispEntity_SetCar         (pLispMgr, pFunctail, pElement) ;
	lispMachineCode_GetLReg   (pLM, LM_LREG_1,   &pFunclist) ;
	
	lispMachineCode_PushState (pLM, &lispMachineState_mapcarStringPostApply) ;
	lispMachineCode_SetLReg   (pLM, LM_LREG_ACC, pFunclist) ;
	lispMachineCode_SetState (pLM, &lispMachineState_Funcall) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_mapcarStringPostApply (
	register TLispMachine*	pLM)
{
	return	lispMachineState_mapcarPostApplyCommon (pLM, &lispMachineState_mapcarStringApply) ;
}

TLMRESULT
lispMachineState_mapcarPostApplyCommon (
	register TLispMachine*	pLM,
	register TLMRESULT		(*pNextState)(TLispMachine*))
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pTail ;
	TLispEntity*	pResult ;
	TLispEntity*	pNewTail ;
	TLispEntity*	pNil ;
	
	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) {
		lispMachineCode_SetState (pLM, &lispMachineState_mapcarFinalize) ;
		return	LMR_CONTINUE ;
	}
	lispMachineCode_GetLReg (pLM, LM_LREG_5,   &pTail) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pResult) ;
	
	lispMgr_CreateNil      (pLispMgr, &pNil) ;
	lispMgr_CreateConscell (pLispMgr, pResult, pNil, &pNewTail) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pTail))) {
		lispMachineCode_SetLReg (pLM, LM_LREG_4, pNewTail) ;
	} else {
		lispEntity_SetCdr (pLispMgr, pTail, pNewTail) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_5, pNewTail) ;
	lispMachineCode_SetState (pLM, pNextState) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_mapcarFinalize (
	register TLispMachine*	pLM)
{
	lispMachineCode_MoveLReg (pLM, LM_LREG_ACC, LM_LREG_4) ;
	lispMachineCode_PopVReg  (pLM, LM_VREG_1) ;
	lispMachineCode_PopLReg  (pLM, LM_LREG_5) ;
	lispMachineCode_PopLReg  (pLM, LM_LREG_4) ;
	lispMachineCode_PopLReg  (pLM, LM_LREG_3) ;
	lispMachineCode_PopLReg  (pLM, LM_LREG_2) ;
	lispMachineCode_PopLReg  (pLM, LM_LREG_1) ;
	return	LMR_RETURN ;
}

