/* # skkinput (Simple Kana-Kanji Input)
 *
 * This file is part of skkinput.
 * Copyright (C) 2002
 * Takashi SAKAMOTO (PXG01715@nifty.ne.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "local.h"
#include <stdio.h>
#include <assert.h>
#include "lmachinep.h"

static	TLMRESULT	lispMachineState_compareNumberCommon	(TLispMachine*, Boolean (*)(TLispManager*, TLispEntity*, TLispEntity*, Boolean*)) ;

/*
 *	+ is a built-in function.
 *
 *	Return sum of any number of arguments, which are numbers or markers.
 */
TLMRESULT
lispMachineState_Plus (
	register TLispMachine*		pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pTarget ;
	TLispEntity*	pCar ;
	TLispEntity*	pCdr ;
	TLispEntity*	pRetval ;
	float			fSum, fValue ;
	long			lSum, lValue ;
	int				iPos ;
	Boolean			fFloat ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pTarget) ;
	assert (pTarget  != NULL) ;

	lSum	= 0 ;
	fSum	= 0.0 ;
	fFloat	= False ;
	while (TFAILED (lispEntity_Nullp (pLispMgr, pTarget))) {
		(void) lispEntity_GetCar (pLispMgr, pTarget, &pCar) ;
		assert (pCar != NULL) ;
		if (TSUCCEEDED (lispEntity_Integerp (pLispMgr, pCar))) {
			(void) lispEntity_GetIntegerValue (pLispMgr, pCar, &lValue) ;
			if (TSUCCEEDED (fFloat)) {
				fSum	+= (float) lValue ;
			} else {
				lSum	+= lValue ;
			}
		} else if (TSUCCEEDED (lispEntity_Floatp (pLispMgr, pCar))) {
			(void) lispEntity_GetFloatValue (pLispMgr, pCar, &fValue) ;
			if (TFAILED (fFloat)) {
				fSum	= (float) lSum ;
				fFloat	= True ;
			}
			fSum	+= fValue ;
		} else if (TSUCCEEDED (lispEntity_Markerp (pLispMgr, pCar))) {
			TLispEntity*	pBuffer ;

			if (TFAILED (lispMarker_GetBufferPosition (pLispMgr, pCar, &pBuffer, &iPos)) ||
				pBuffer == NULL) {
#if defined (DEBUG)
				fprintf (stderr, "Marker does not point anywhere.\n") ;
#endif
				lispMachineCode_SetError (pLM) ;
				return	LMR_RETURN ;
			}
			if (TSUCCEEDED (fFloat)) {
				fSum	+= (float) iPos ;
			} else {
				lSum	+= iPos ;
			}
		} else {
#if defined (DEBUG)
			fprintf (stderr, "Wrong type argument: integer-or-markerp, ") ;
			lispEntity_Print (pLispMgr, pCar) ;
			fprintf (stderr, "\n") ;
#endif
			lispMachineCode_SetError (pLM) ;
			return	LMR_RETURN ;
		}
		(void) lispEntity_GetCdr (pLispMgr, pTarget, &pCdr) ;
		pTarget	= pCdr ;
	}
	if (fFloat) {
		if (TFAILED (lispMgr_CreateFloat (pLispMgr, fSum, &pRetval)) ||
			pRetval == NULL)
			return	LMR_ERROR ;
	} else {
		if (TFAILED (lispMgr_CreateInteger (pLispMgr, lSum, &pRetval)) ||
			pRetval == NULL)
			return	LMR_ERROR ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

/*
 *	- is a built-in function.
 *
 *	Negate number of subtract numbers or markers.
 *	With one arg, negates it. With more than one arg,
 *	subtracts all but the first from the first.
 */
TLMRESULT
lispMachineState_Minus (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pTarget ;
	TLispEntity*	pCar ;
	TLispEntity*	pCdr ;
	TLispEntity*	pRetval ;
	float		fSub, fValue ;
	long		lSub, lValue ;
	int			iPos ;
	Boolean		fFloat ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
#if defined (DEBUG_LV99)
	fprintf (stderr, "state = mins\n") ;
	lispMachine_ShowRegisterValue (pLM) ;
#endif
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pTarget) ;
	assert (pTarget  != NULL) ;

	lSub	= 0 ;
	fSub	= 0.0 ;
	fFloat	= False ;

	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pTarget)))
		goto	exit_minus ;
		
	(void) lispEntity_GetCar (pLispMgr, pTarget, &pCar) ;
	assert (pCar != NULL) ;
	if (TSUCCEEDED (lispEntity_Integerp (pLispMgr, pCar))) {
		(void) lispEntity_GetIntegerValue (pLispMgr, pCar, &lValue) ;
		if (TSUCCEEDED (fFloat)) {
			fSub	= (float) lValue ;
		} else {
			lSub	= lValue ;
		}
	} else if (TSUCCEEDED (lispEntity_Floatp (pLispMgr, pCar))) {
		(void) lispEntity_GetFloatValue (pLispMgr, pCar, &fValue) ;
		if (TFAILED (fFloat)) {
			fSub	= (float) lSub ;
			fFloat	= True ;
		}
		fSub	= fValue ;
	} else if (TSUCCEEDED (lispEntity_Markerp (pLispMgr, pCar))) {
		TLispEntity*	pBuffer ;

		if (TFAILED (lispMarker_GetBufferPosition (pLispMgr, pCar, &pBuffer, &iPos)) ||
			pBuffer == NULL) {
#if defined (DEBUG)
			fprintf (stderr, "Marker does not point anywhere.\n") ;
#endif
			lispMachineCode_SetError (pLM) ;
			return	LMR_RETURN ;
		}
		if (TSUCCEEDED (fFloat)) {
			fSub	= (float) iPos ;
		} else {
			lSub	= iPos ;
		}
	} else {
#if defined (DEBUG)
		fprintf (stderr, "Wrong type argument: integer-or-markerp, ") ;
		lispEntity_Print (pLispMgr, pCar) ;
		fprintf (stderr, "\n") ;
#endif	
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	(void) lispEntity_GetCdr (pLispMgr, pTarget, &pCdr) ;
	pTarget	= pCdr ;
	
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pTarget))) {
		if (fFloat) {
			fSub	= - fSub ;
		} else {
			lSub	= - lSub ;
		}
		goto	exit_minus ;
	}

	while (TFAILED (lispEntity_Nullp (pLispMgr, pTarget))) {
		(void) lispEntity_GetCar (pLispMgr, pTarget, &pCar) ;
		assert (pCar != NULL) ;
		if (TSUCCEEDED (lispEntity_Integerp (pLispMgr, pCar))) {
			(void) lispEntity_GetIntegerValue (pLispMgr, pCar, &lValue) ;
			if (TSUCCEEDED (fFloat)) {
				fSub	-= (float) lValue ;
			} else {
				lSub	-= lValue ;
			}
		} else if (TSUCCEEDED (lispEntity_Floatp (pLispMgr, pCar))) {
			(void) lispEntity_GetFloatValue (pLispMgr, pCar, &fValue) ;
			if (TFAILED (fFloat)) {
				fSub	= (float) lSub ;
				fFloat	= True ;
			}
			fSub	-= fValue ;
		} else if (TSUCCEEDED (lispEntity_Markerp (pLispMgr, pCar))) {
			TLispEntity*	pBuffer ;
			if (TFAILED (lispMarker_GetBufferPosition (pLispMgr, pCar, &pBuffer, &iPos)) ||
				pBuffer == NULL) {
#if defined (DEBUG)
				fprintf (stderr, "Marker does not point anywhere.\n") ;
#endif
				lispMachineCode_SetError (pLM) ;
				return	LMR_RETURN ;
			}
			if (TSUCCEEDED (fFloat)) {
				fSub	-= (float) iPos ;
			} else {
				lSub	-= iPos ;
			}
		} else {
#if defined (DEBUG)
			fprintf (stderr, "Wrong type argument: integer-or-markerp, ") ;
			lispEntity_Print (pLispMgr, pCar) ;
			fprintf (stderr, "\n") ;
#endif
			lispMachineCode_SetError (pLM) ;
			return	LMR_RETURN ;
		}
		(void) lispEntity_GetCdr (pLispMgr, pTarget, &pCdr) ;
		pTarget	= pCdr ;
	}

 exit_minus:
	if (fFloat) {
		if (TFAILED (lispMgr_CreateFloat (pLispMgr, fSub, &pRetval)) ||
			pRetval == NULL)
			return	LMR_ERROR ;
	} else {
		if (TFAILED (lispMgr_CreateInteger (pLispMgr, lSub, &pRetval)) ||
			pRetval == NULL)
			return	LMR_ERROR ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

/*
 * * is a built-in function.
 *
 *	Returns product of any number of arguments, which are numbers or markers.
 */
TLMRESULT
lispMachineState_Multiply (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pTarget ;
	TLispEntity*	pCar ;
	TLispEntity*	pCdr ;
	TLispEntity*	pRetval ;
	float		fMult, fValue ;
	long		lMult, lValue ;
	int			iPos ;
	Boolean		fFloat ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pTarget) ;
	assert (pTarget  != NULL) ;

	lMult	= 1 ;
	fMult	= 1.0 ;
	fFloat	= False ;
	while (TFAILED (lispEntity_Nullp (pLispMgr, pTarget))) {
		(void) lispEntity_GetCar (pLispMgr, pTarget, &pCar) ;
		assert (pCar != NULL) ;
		if (TSUCCEEDED (lispEntity_Integerp (pLispMgr, pCar))) {
			(void) lispEntity_GetIntegerValue (pLispMgr, pCar, &lValue) ;
			if (TSUCCEEDED (fFloat)) {
				fMult	*= (float) lValue ;
			} else {
				lMult	*= lValue ;
			}
		} else if (TSUCCEEDED (lispEntity_Floatp (pLispMgr, pCar))) {
			(void) lispEntity_GetFloatValue (pLispMgr, pCar, &fValue) ;
			if (TFAILED (fFloat)) {
				fMult	= (float) lMult ;
				fFloat	= True ;
			}
			fMult	*= fValue ;
		} else if (TSUCCEEDED (lispEntity_Markerp (pLispMgr, pCar))) {
			TLispEntity*	pBuffer ;
			if (TFAILED (lispMarker_GetBufferPosition (pLispMgr, pCar, &pBuffer, &iPos)) ||
				pBuffer == NULL) {
#if defined (DEBUG)
				fprintf (stderr, "Marker does not point anywhere.\n") ;
#endif
				lispMachineCode_SetError (pLM) ;
				return	LMR_RETURN ;
			}
			if (TSUCCEEDED (fFloat)) {
				fMult	*= (float) iPos ;
			} else {
				lMult	*= iPos ;
			}
		} else {
#if defined (DEBUG)
			fprintf (stderr, "Wrong type argument: integer-or-markerp, ") ;
			lispEntity_Print (pLispMgr, pCar) ;
			fprintf (stderr, "\n") ;
#endif
			lispMachineCode_SetError (pLM) ;
			return	LMR_RETURN ;
		}
		(void) lispEntity_GetCdr (pLispMgr, pTarget, &pCdr) ;
		pTarget	= pCdr ;
	}
	if (fFloat) {
		if (TFAILED (lispMgr_CreateFloat (pLispMgr, fMult, &pRetval)) ||
			pRetval == NULL)
			return	LMR_ERROR ;
	} else {
		if (TFAILED (lispMgr_CreateInteger (pLispMgr, lMult, &pRetval)) ||
			pRetval == NULL)
			return	LMR_ERROR ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

/*
 *	/ is a built-in function.
 */
TLMRESULT
lispMachineState_Divide (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pTarget ;
	TLispEntity*	pCar ;
	TLispEntity*	pCdr ;
	TLispEntity*	pRetval ;
	float		fDiv, fValue ;
	long		lDiv, lValue ;
	int			iPos ;
	Boolean		fFloat ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pTarget) ;
	assert (pTarget  != NULL) ;

	lDiv	= 0 ;
	fDiv	= 0.0 ;
	fFloat	= False ;

	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pTarget)))
		goto	exit_minus ;
		
	(void) lispEntity_GetCar (pLispMgr, pTarget, &pCar) ;
	assert (pCar != NULL) ;
	if (TSUCCEEDED (lispEntity_Integerp (pLispMgr, pCar))) {
		(void) lispEntity_GetIntegerValue (pLispMgr, pCar, &lValue) ;
		if (TSUCCEEDED (fFloat)) {
			fDiv	= (float) lValue ;
		} else {
			lDiv	= lValue ;
		}
	} else if (TSUCCEEDED (lispEntity_Floatp (pLispMgr, pCar))) {
		(void) lispEntity_GetFloatValue (pLispMgr, pCar, &fValue) ;
		if (TFAILED (fFloat)) {
			fDiv	= (float) lDiv ;
			fFloat	= True ;
		}
		fDiv	= fValue ;
	} else if (TSUCCEEDED (lispEntity_Markerp (pLispMgr, pCar))) {
		TLispEntity*	pBuffer ;
		if (TFAILED (lispMarker_GetBufferPosition (pLispMgr, pCar, &pBuffer, &iPos)) ||
			pBuffer == NULL) {
#if defined (DEBUG)
			fprintf (stderr, "Marker does not point anywhere.\n") ;
#endif
			lispMachineCode_SetError (pLM) ;
			return	LMR_RETURN ;
		}
		if (TSUCCEEDED (fFloat)) {
			fDiv	= (float) iPos ;
		} else {
			lDiv	= iPos ;
		}
	} else {
#if defined (DEBUG)
		fprintf (stderr, "Wrong type argument: integer-or-markerp, ") ;
		lispEntity_Print (pLispMgr, pCar) ;
		fprintf (stderr, "\n") ;
#endif	
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	(void) lispEntity_GetCdr (pLispMgr, pTarget, &pCdr) ;
	pTarget	= pCdr ;
	
	while (TFAILED (lispEntity_Nullp (pLispMgr, pTarget))) {
		(void) lispEntity_GetCar (pLispMgr, pTarget, &pCar) ;
		assert (pCar != NULL) ;
		if (TSUCCEEDED (lispEntity_Integerp (pLispMgr, pCar))) {
			(void) lispEntity_GetIntegerValue (pLispMgr, pCar, &lValue) ;
			if (lValue == 0) {
				lispMachineCode_SetError (pLM) ;
				return	LMR_RETURN ;
			}
			if (TSUCCEEDED (fFloat)) {
				fDiv	/= (float) lValue ;
			} else {
				lDiv	/= lValue ;
			}
		} else if (TSUCCEEDED (lispEntity_Floatp (pLispMgr, pCar))) {
			(void) lispEntity_GetFloatValue (pLispMgr, pCar, &fValue) ;
			if (fValue == 0.0) {
				lispMachineCode_SetError (pLM) ;
				return	LMR_RETURN ;
			}
			if (TFAILED (fFloat)) {
				fDiv	= (float) lDiv ;
				fFloat	= True ;
			}
			fDiv	/= fValue ;
		} else if (TSUCCEEDED (lispEntity_Markerp (pLispMgr, pCar))) {
			TLispEntity*	pBuffer ;
			if (TFAILED (lispMarker_GetBufferPosition (pLispMgr, pCar, &pBuffer, &iPos)) ||
				pBuffer == NULL) {
#if defined (DEBUG)
				fprintf (stderr, "Marker does not point anywhere.\n") ;
#endif
				lispMachineCode_SetError (pLM) ;
				return	LMR_RETURN ;
			}
			if (iPos == 0) {
				lispMachineCode_SetError (pLM) ;
				return	LMR_RETURN ;
			}
			if (TSUCCEEDED (fFloat)) {
				fDiv	/= (float) iPos ;
			} else {
				lDiv	/= iPos ;
			}
		} else {
#if defined (DEBUG)
			fprintf (stderr, "Wrong type argument: integer-or-markerp, ") ;
			lispEntity_Print (pLispMgr, pCar) ;
			fprintf (stderr, "\n") ;
#endif
			lispMachineCode_SetError (pLM) ;
			return	LMR_RETURN ;
		}
		(void) lispEntity_GetCdr (pLispMgr, pTarget, &pCdr) ;
		pTarget	= pCdr ;
	}

 exit_minus:
	if (fFloat) {
		if (TFAILED (lispMgr_CreateFloat (pLispMgr, fDiv, &pRetval)) ||
			pRetval == NULL)
			return	LMR_ERROR ;
	} else {
		if (TFAILED (lispMgr_CreateInteger (pLispMgr, lDiv, &pRetval)) ||
			pRetval == NULL)
			return	LMR_ERROR ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

/*	% is a built-in function.
 *
 *	Returns remainder of X divided by Y.
 *	Both must be integers or markers.
 *
 *	(% X Y)
 */
TLMRESULT
lispMachineState_Remainder (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntX ;
	TLispEntity*	pEntY ;
	TLispEntity*	pEntRetval ;
	TLispNumber		numX, numY ;
	register long	lX, lY ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist  != NULL) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntX)) ||
		TFAILED (lispEntity_IntegerOrMarkerp (pLispMgr, pEntX)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pEntArglist, &pEntY)) ||
		TFAILED (lispEntity_IntegerOrMarkerp (pLispMgr, pEntY))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	(void) lispEntity_GetNumberValueOrMarkerPosition (pLispMgr, pEntX, &numX) ;
	(void) lispEntity_GetNumberValueOrMarkerPosition (pLispMgr, pEntY, &numY) ;
	lX	= numX.m_Value.m_lLong ;
	lY	= numY.m_Value.m_lLong ;
	if (lY == 0) {
#if defined (DEBUG)
		fprintf (stderr, "Arthmetic Error\n") ;
#endif
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispMgr_CreateInteger (pLispMgr, lX % lY, &pEntRetval)) ||
		pEntRetval == NULL)
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_MathGreaterThan (TLispMachine* pLM)
{
	return	lispMachineState_compareNumberCommon (pLM, &lispMgr_MathGreaterThan) ;
}

TLMRESULT
lispMachineState_MathGreaterEqual (TLispMachine* pLM)
{
	return	lispMachineState_compareNumberCommon (pLM, &lispMgr_MathGreaterEqual) ;
}

TLMRESULT
lispMachineState_MathLessThan (TLispMachine* pLM)
{
	return	lispMachineState_compareNumberCommon (pLM, &lispMgr_MathLessThan) ;
}

TLMRESULT
lispMachineState_MathLessEqual (TLispMachine* pLM)
{
	return	lispMachineState_compareNumberCommon (pLM, &lispMgr_MathLessEqual) ;
}

TLMRESULT
lispMachineState_MathEqual (
	register TLispMachine*	pLM)
{
	return	lispMachineState_compareNumberCommon (pLM, &lispMgr_MathEqual) ;
}

TLMRESULT
lispMachineState_compareNumberCommon (
	register TLispMachine*	pLM,
	register Boolean		(*pCompFunc)(TLispManager*, TLispEntity*, TLispEntity*, Boolean*))
{
	TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pNum1 ;
	TLispEntity*	pNum2 ;
	TLispEntity*	pRetval ;
	Boolean		fResult ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	lispEntity_GetCar  (pLispMgr, pArglist, &pNum1) ;
	lispEntity_GetCadr (pLispMgr, pArglist, &pNum2) ;
	if (TFAILED ((*pCompFunc) (pLispMgr, pNum1, pNum2, &fResult))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (fResult) {
		lispMgr_CreateT   (pLispMgr, &pRetval) ;
	} else {
		lispMgr_CreateNil (pLispMgr, &pRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_1Plus (TLispMachine* pLM)
{
	TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pNil ;
	TLispEntity*	pOne ;
	TLispEntity*	pHead ;
	TLispEntity*	pTail ;
	TLispEntity*	pNumber ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	assert (pArglist != NULL) ;
	
	if (TFAILED (lispEntity_GetCar (pLispMgr, pArglist, &pNumber))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMgr_CreateNil (pLispMgr, &pNil) ;
	if (TFAILED (lispMgr_CreateConscell (pLispMgr, pNumber, pNil, &pHead)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pHead) ;
	if (TFAILED (lispMgr_CreateConscell (pLispMgr, pNil, pNil, &pTail)))
		return	LMR_ERROR ;
	lispEntity_SetCdr (pLispMgr, pHead, pTail) ;
	if (TFAILED (lispMgr_CreateInteger (pLispMgr, 1, &pOne)))
		return	LMR_ERROR ;
	lispEntity_SetCar (pLispMgr, pTail, pOne) ;

	pLM->m_pState	= &lispMachineState_Plus ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_1Minus (TLispMachine* pLM)
{
	TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pNil ;
	TLispEntity*	pOne ;
	TLispEntity*	pHead ;
	TLispEntity*	pTail ;
	TLispEntity*	pNumber ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	assert (pArglist != NULL) ;

	if (TFAILED (lispEntity_GetCar (pLispMgr, pArglist, &pNumber))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMgr_CreateNil (pLispMgr, &pNil) ;
	if (TFAILED (lispMgr_CreateConscell (pLispMgr, pNumber, pNil, &pHead)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pHead) ;
	if (TFAILED (lispMgr_CreateConscell (pLispMgr, pNil, pNil, &pTail)))
		return	LMR_ERROR ;
	lispEntity_SetCdr (pLispMgr, pHead, pTail) ;
	if (TFAILED (lispMgr_CreateInteger (pLispMgr, 1, &pOne)))
		return	LMR_ERROR ;
	lispEntity_SetCar (pLispMgr, pTail, pOne) ;

	pLM->m_pState	= &lispMachineState_Minus ;
	return	LMR_CONTINUE ;
}

/*
 *	NUMBER ʤ顢t ֤
 *
 *	(zerop NUMBER)
 */
TLMRESULT
lispMachineState_Zerop (TLispMachine* pLM)
{
	TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pRetval ;
	TLispEntity*	pNumber ;
	TLispNumber		number ;
	Boolean			fRetval ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	assert (pArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pArglist, &pNumber)) ||
		TFAILED (lispEntity_Numberp (pLispMgr, pNumber))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	(void) lispEntity_GetNumberValueOrMarkerPosition (pLispMgr, pNumber, &number) ;
	if (number.m_fFloatp) {
		fRetval	= (number.m_Value.m_fFloat == 0.0f) ;
	} else {
		fRetval	= (number.m_Value.m_lLong  == 0) ;
	}
	if (fRetval) {
		lispMgr_CreateT   (pLispMgr, &pRetval) ;
	} else {
		lispMgr_CreateNil (pLispMgr, &pRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

/*
 *	(min NUMBER-OR-MARKER &rest NUMBERS-OR-MARKERS)
 *
 *	(ͤޡǤʤФʤʤ)ǤäȤ⾮֤ͤ
 *	֤ͤϤĤͤǤ롣ޡϿͤѴ롣
 */
TLMRESULT
lispMachineState_Min (
	register TLispMachine*	pLM)
{
	TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntRetval ;
	TLispEntity*	pEntCar ;
	TLispNumber		num ;
	register float	fMin, fValue ;
	register long	lMin ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntCar)) ||
		TFAILED (lispEntity_GetNumberOrMarkerValue (pLispMgr, pEntCar, &num))) 
		goto	error ;
	lispEntity_GetCdr (pLispMgr, pEntArglist, &pEntArglist) ;
	if (!num.m_fFloatp) {
		lMin	= num.m_Value.m_lLong ;
		while (TFAILED (lispEntity_Nullp (pLispMgr, pEntArglist))) {
			if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntCar)) ||
				TFAILED (lispEntity_GetCdr (pLispMgr, pEntArglist, &pEntArglist)) ||
				TFAILED (lispEntity_GetNumberOrMarkerValue (pLispMgr, pEntCar, &num))) 
				goto	error ;
			if (num.m_fFloatp) {
				fMin	= (float) lMin ;
				if (fMin > num.m_Value.m_fFloat)
					fMin	= num.m_Value.m_fFloat ;
				goto	float_mode ;
			}
			if (lMin > num.m_Value.m_lLong)
				lMin	= num.m_Value.m_lLong ;
		}
		if (TFAILED (lispMgr_CreateInteger (pLispMgr, lMin, &pEntRetval)))
			return	LMR_ERROR ;
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
		return	LMR_RETURN ;
	}
	fMin	= num.m_Value.m_fFloat ;
  float_mode:
	while (TFAILED (lispEntity_Nullp (pLispMgr, pEntArglist))) {
		if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntCar)) ||
			TFAILED (lispEntity_GetCdr (pLispMgr, pEntArglist, &pEntArglist)) ||
			TFAILED (lispEntity_GetNumberOrMarkerValue (pLispMgr, pEntCar, &num))) 
			goto	error ;
		if (!num.m_fFloatp) {
			fValue	= (float) num.m_Value.m_lLong ;
		} else {
			fValue	= num.m_Value.m_fFloat ;
		}
		if (fMin > fValue)
			fMin	= fValue ;
	}
	if (TFAILED (lispMgr_CreateFloat (pLispMgr, fMin, &pEntRetval)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;

  error:
	lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Max (
	register TLispMachine*	pLM)
{
	TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntRetval ;
	TLispEntity*	pEntCar ;
	TLispNumber		num ;
	register float	fMax, fValue ;
	register long	lMax ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntCar)) ||
		TFAILED (lispEntity_GetNumberOrMarkerValue (pLispMgr, pEntCar, &num))) 
		goto	error ;
	lispEntity_GetCdr (pLispMgr, pEntArglist, &pEntArglist) ;
	if (!num.m_fFloatp) {
		lMax	= num.m_Value.m_lLong ;
		while (TFAILED (lispEntity_Nullp (pLispMgr, pEntArglist))) {
			if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntCar)) ||
				TFAILED (lispEntity_GetCdr (pLispMgr, pEntArglist, &pEntArglist)) ||
				TFAILED (lispEntity_GetNumberOrMarkerValue (pLispMgr, pEntCar, &num))) 
				goto	error ;
			if (num.m_fFloatp) {
				fMax	= (float) lMax ;
				if (fMax < num.m_Value.m_fFloat)
					fMax	= num.m_Value.m_fFloat ;
				goto	float_mode ;
			}
			if (lMax < num.m_Value.m_lLong)
				lMax	= num.m_Value.m_lLong ;
		}
		if (TFAILED (lispMgr_CreateInteger (pLispMgr, lMax, &pEntRetval)))
			return	LMR_ERROR ;
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
		return	LMR_RETURN ;
	}
	fMax	= num.m_Value.m_fFloat ;
  float_mode:
	while (TFAILED (lispEntity_Nullp (pLispMgr, pEntArglist))) {
		if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntCar)) ||
			TFAILED (lispEntity_GetCdr (pLispMgr, pEntArglist, &pEntArglist)) ||
			TFAILED (lispEntity_GetNumberOrMarkerValue (pLispMgr, pEntCar, &num))) 
			goto	error ;
		if (!num.m_fFloatp) {
			fValue	= (float) num.m_Value.m_lLong ;
		} else {
			fValue	= num.m_Value.m_fFloat ;
		}
		if (fMax < fValue)
			fMax	= fValue ;
	}
	if (TFAILED (lispMgr_CreateFloat (pLispMgr, fMax, &pEntRetval)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;

  error:
	lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

/*	built-in function.
 *		(mod X Y)
 *
 *	Returns X modulo Y.
 *	The result falls between zero (inclusive) and Y (exclusive).
 *	Both X and Y must be numbers or markers.
 */
TLMRESULT
lispMachineState_Mod (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntX ;
	TLispEntity*	pEntY ;
	TLispEntity*	pEntRetval ;
	TLispNumber		numX, numY ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntX) ;
	lispEntity_GetCadr (pLispMgr, pEntArglist, &pEntY) ;
	if (TFAILED (lispEntity_GetNumberOrMarkerValue (pLispMgr, pEntX, &numX)) ||
		TFAILED (lispEntity_GetNumberOrMarkerValue (pLispMgr, pEntY, &numY))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	/*	Ǥʤ modulo äƤɤƤΤ...*/
	if (numX.m_fFloatp || numY.m_fFloatp) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	/*	ϥ顼ˤʤ롣*/
	if (numY.m_Value.m_lLong == 0) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispMgr_CreateInteger (pLispMgr, numX.m_Value.m_lLong % numY.m_Value.m_lLong, &pEntRetval)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*	builtin-function:
 *		(/= NUM1 NUM2)
 *
 *	NUM1  NUM2 ʤ t ֤NUM1, NUM2  number ޤ marker
 *	ǤʤФʤʤ
 */
TLMRESULT
lispMachineState_SlashEqual (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	register Boolean		fRetval ;
	TLispEntity*		pEntArglist ;
	TLispEntity*		pEntNUM1 ;
	TLispEntity*		pEntNUM2 ;
	TLispEntity*		pEntRetval ;
	TLispNumber			num1, num2 ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntNUM1) ;
	lispEntity_GetCadr (pLispMgr, pEntArglist, &pEntNUM2) ;
	if (TFAILED (lispEntity_GetNumberOrMarkerValue (pLispMgr, pEntNUM1, &num1)) ||
		TFAILED (lispEntity_GetNumberOrMarkerValue (pLispMgr, pEntNUM2, &num2))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TSUCCEEDED (num1.m_fFloatp)) {
		if (TSUCCEEDED (num2.m_fFloatp)) {
			fRetval	= (num1.m_Value.m_fFloat != num2.m_Value.m_fFloat) ;
		} else {
			fRetval	= (num1.m_Value.m_fFloat != num2.m_Value.m_lLong) ;
		}
	}  else {
		if (TSUCCEEDED (num2.m_fFloatp)) {
			fRetval	= (num1.m_Value.m_lLong != num2.m_Value.m_fFloat) ;
		} else {
			fRetval	= (num1.m_Value.m_lLong != num2.m_Value.m_lLong) ;
		}
	}
	if (fRetval) {
		lispMgr_CreateT   (pLispMgr, &pEntRetval) ;
	} else {
		lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

