/* # skkinput (Simple Kana-Kanji Input)
 *
 * This file is part of skkinput.
 * Copyright (C) 2002
 * Takashi SAKAMOTO (PXG01715@nifty.ne.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "local.h"
#include <stdio.h>
#include <assert.h>
#include "lmachinep.h"

#define	PROTOLSTATEFUNC(func)	static TLMRESULT func (TLispMachine*)

PROTOLSTATEFUNC(lispMachineState_andStep) ;
PROTOLSTATEFUNC(lispMachineState_orStep) ;
PROTOLSTATEFUNC(lispMachineState_ifCond) ;
PROTOLSTATEFUNC(lispMachineState_ifElse) ;
PROTOLSTATEFUNC(lispMachineState_ifFinalize) ;
PROTOLSTATEFUNC(lispMachineState_condTestClause) ;
PROTOLSTATEFUNC(lispMachineState_condPostTestClause) ;
PROTOLSTATEFUNC(lispMachineState_condEvalRestClause) ;
PROTOLSTATEFUNC(lispMachineState_condFinalize) ;
PROTOLSTATEFUNC(lispMachineState_whileTest) ;
PROTOLSTATEFUNC(lispMachineState_whileBody) ;
PROTOLSTATEFUNC(lispMachineState_whileFinalize) ;
PROTOLSTATEFUNC(lispMachineState_catchTag) ;
PROTOLSTATEFUNC(lispMachineState_catchBody) ;
PROTOLSTATEFUNC(lispMachineState_catchFinalize) ;
PROTOLSTATEFUNC(lispMachineState_conditionCaseAfter) ;
PROTOLSTATEFUNC(lispMachineState_conditionCaseHandler) ;
PROTOLSTATEFUNC(lispMachineState_conditionCaseFinalize) ;
PROTOLSTATEFUNC(lispMachineState_prognStep) ;
PROTOLSTATEFUNC(lispMachineState_prog1Step1) ;
PROTOLSTATEFUNC(lispMachineState_prog1Step2) ;
PROTOLSTATEFUNC(lispMachineState_prog1Finalize) ;
PROTOLSTATEFUNC(lispMachineState_prog2Step1) ;
PROTOLSTATEFUNC(lispMachineState_setqGetSym) ;
PROTOLSTATEFUNC(lispMachineState_setqSetVal) ;
PROTOLSTATEFUNC(lispMachineState_setqFinalize) ;
PROTOLSTATEFUNC(lispMachineState_defvarPostEvalInitvalue) ;
PROTOLSTATEFUNC(lispMachineState_unwindProtectAfterEvalBodyform) ;
PROTOLSTATEFUNC(lispMachineState_unwindProtectAfterEvalUnwindForms) ;
PROTOLSTATEFUNC(lispMachineState_setqCommon) ;
PROTOLSTATEFUNC(lispMachineState_saveExcursionFinalize) ;
PROTOLSTATEFUNC(lispMachineState_saveRestrictionFinalize) ;
#undef	PROTOLSTATEFUNC

TLMRESULT
lispMachineState_Done (register TLispMachine* pLM)
{
	assert (pLM != NULL) ;
	return	LMR_DONE ;
}

/*
 *	Function:	(identity ARG)
 *
 *	ѹ֤
 */
TLMRESULT
lispMachineState_Identity (register TLispMachine* pLM)
{
	TLispEntity*	pEntArglist ;
	TLispEntity*	pCAR ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispEntity_GetCar (pLM->m_pLispMgr, pEntArglist, &pCAR) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pCAR) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Quote (register TLispMachine* pLM)
{
	TLispEntity*	pEntArglist ;
	TLispEntity*	pCAR ;

#if defined (DEBUG_LV99)
	fprintf (stderr, "state = 'quote'\n") ;
#endif
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispEntity_GetCar (pLM->m_pLispMgr, pEntArglist, &pCAR) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pCAR) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_And (
	register TLispMachine*	pLM)
{
	TLispEntity*	pEntT ;

	lispMachineCode_PushLReg (pLM, LM_LREG_1) ;
	lispMachineCode_MoveLReg (pLM, LM_LREG_1,   LM_LREG_ACC) ;

	lispMgr_CreateT (pLM->m_pLispMgr, &pEntT) ;
	lispMachineCode_SetLReg  (pLM, LM_LREG_ACC, pEntT) ;
	lispMachineCode_SetState (pLM, &lispMachineState_andStep) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_andStep (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pBody ;
	TLispEntity*	pResult ;
	TLispEntity*	pCar ;
	
	assert (pLM != NULL) ;

	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) {
		lispMachineCode_PopLReg (pLM, LM_LREG_1) ;
		return	LMR_RETURN ;
	}

	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pResult) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_1,   &pBody) ;

	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pResult)) ||
		TSUCCEEDED (lispEntity_Nullp (pLispMgr, pBody))) {
		lispMachineCode_PopLReg (pLM, LM_LREG_1) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispEntity_GetCar (pLispMgr, pBody, &pCar)) ||
		pCar == NULL) {
		lispMachineCode_PopLReg (pLM, LM_LREG_1) ;
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMachineCode_Cdr   (pLM, LM_LREG_1, LM_LREG_1) ;
	lispMachineCode_Evaln (pLM, pCar, &lispMachineState_andStep) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_Or (register TLispMachine* pLM)
{
	TLispEntity*	pEntNil ;

	lispMachineCode_PushLReg (pLM, LM_LREG_1) ;
	lispMachineCode_MoveLReg (pLM, LM_LREG_1,   LM_LREG_ACC) ;

	lispMgr_CreateNil (pLM->m_pLispMgr, &pEntNil) ;
	lispMachineCode_SetLReg  (pLM, LM_LREG_ACC, pEntNil) ;
	lispMachineCode_SetState (pLM, &lispMachineState_orStep) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_orStep (register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntBody ;
	TLispEntity*	pEntResult ;
	TLispEntity*	pEntCar ;
	
	assert (pLM != NULL) ;

	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) {
		lispMachineCode_PopLReg (pLM, LM_LREG_1) ;
		return	LMR_RETURN ;
	}

	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntResult) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_1,   &pEntBody) ;

	if (TFAILED    (lispEntity_Nullp (pLispMgr, pEntResult)) ||
		TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntBody))) {
		lispMachineCode_PopLReg (pLM, LM_LREG_1) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntBody, &pEntCar))) {
		lispMachineCode_PopLReg (pLM, LM_LREG_1) ;
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMachineCode_Cdr   (pLM, LM_LREG_1, LM_LREG_1) ;
	lispMachineCode_Evaln (pLM, pEntCar, &lispMachineState_orStep) ;
	return	LMR_CONTINUE ;
}

/*
 *	(if COND THEN ELSE ...)
 */
TLMRESULT
lispMachineState_If (register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntBody ;
	TLispEntity*	pEntCond ;
	TLispEntity*	pEntRest ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	
	lispMachineCode_GetLReg  (pLM, LM_LREG_ACC, &pEntBody) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntBody)) ||
		TFAILED (lispEntity_GetCar (pLispMgr, pEntBody, &pEntCond)) ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pEntBody, &pEntRest)) ||
		TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntRest))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMachineCode_PushLReg (pLM, LM_LREG_1) ;
	lispMachineCode_SetLReg  (pLM, LM_LREG_1, pEntRest) ;
	lispMachineCode_Evaln    (pLM, pEntCond, &lispMachineState_ifCond) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_ifCond (register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntCond ;
	TLispEntity*	pEntBody ;
	
	assert (pLM != NULL) ;

	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) {
		lispMachineCode_SetState (pLM, &lispMachineState_ifFinalize) ;
		return	LMR_CONTINUE ;
	}
	
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntCond) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_1,   &pEntBody) ;
	
	if (TFAILED (lispEntity_Nullp (pLispMgr, pEntCond))) {
		TLispEntity*	pEntThen ;
		lispEntity_GetCar     (pLispMgr, pEntBody, &pEntThen) ;
		lispMachineCode_Evaln (pLM, pEntThen, &lispMachineState_ifFinalize) ;
	} else {
		TLispEntity*	pEntElse ;
		lispEntity_GetCdr        (pLispMgr, pEntBody, &pEntElse) ;
		lispMachineCode_SetLReg  (pLM, LM_LREG_1, pEntElse) ;
		lispMachineCode_SetState (pLM, &lispMachineState_ifElse) ;
	}
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_ifElse (register TLispMachine* pLM)
{
	TLispEntity*	pEntElse ;
	TLispEntity*	pEntCar ;
	
	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) {
		lispMachineCode_SetState (pLM, &lispMachineState_ifFinalize) ;
		return	LMR_CONTINUE ;
	}
	
	lispMachineCode_GetLReg (pLM, LM_LREG_1, &pEntElse) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLM->m_pLispMgr, pEntElse))) {
		lispMachineCode_SetState (pLM, &lispMachineState_ifFinalize) ;
		return	LMR_CONTINUE ;
	}
	(void) lispEntity_GetCar (pLM->m_pLispMgr, pEntElse, &pEntCar) ;
	assert (pEntCar != NULL) ;
	lispMachineCode_Evaln   (pLM, pEntCar, &lispMachineState_ifElse) ;
	lispMachineCode_Cdr     (pLM, LM_LREG_1, LM_LREG_1) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_ifFinalize (register TLispMachine* pLM)
{
	lispMachineCode_PopLReg (pLM, LM_LREG_1) ;
	return	LMR_RETURN ;
}

/*
 *	(cond CLAUSES...)
 */
TLMRESULT
lispMachineState_Cond (
	register TLispMachine*	pLM)
{
	lispMachineCode_PushLReg (pLM, LM_LREG_1) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_2) ;
	lispMachineCode_MoveLReg (pLM, LM_LREG_1, LM_LREG_ACC) ;
	lispMachineCode_SetState (pLM, &lispMachineState_condTestClause) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_condTestClause (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntClauses ;
	TLispEntity*	pEntClause ;
	TLispEntity*	pEntCond ;
		
	pLispMgr	= pLM->m_pLispMgr ;
	lispMachineCode_GetLReg (pLM, LM_LREG_1, &pEntClauses) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntClauses))) {
		lispMachineCode_SetState (pLM, &lispMachineState_condFinalize) ;
		return	LMR_CONTINUE ;
	}
	(void) lispEntity_GetCar (pLispMgr, pEntClauses, &pEntClause) ;
	if (TFAILED (lispEntity_Listp (pLispMgr, pEntClause))) {
		lispMachineCode_SetError (pLM) ;
		lispMachineCode_SetState (pLM, &lispMachineState_condFinalize) ;
		return	LMR_CONTINUE ;
	}
	(void) lispEntity_GetCar (pLispMgr, pEntClause, &pEntCond) ;
	assert (pEntCond != NULL) ;
	lispMachineCode_Evaln (pLM, pEntCond, &lispMachineState_condPostTestClause) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_condPostTestClause (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntResult ;
	
	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) {
		lispMachineCode_SetState (pLM, &lispMachineState_condFinalize) ;
		return	LMR_CONTINUE ;
	}
	pLispMgr	= pLM->m_pLispMgr ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntResult) ;
	if (TFAILED (lispEntity_Nullp (pLispMgr, pEntResult))) {
		lispMachineCode_Car (pLM, LM_LREG_1, LM_LREG_1) ;
		lispMachineCode_Cdr (pLM, LM_LREG_1, LM_LREG_1) ;
		lispMachineCode_SetState (pLM, &lispMachineState_condEvalRestClause) ;
	} else {
		lispMachineCode_Cdr (pLM, LM_LREG_1, LM_LREG_1) ;
		lispMachineCode_SetState (pLM, &lispMachineState_condTestClause) ;
	}
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_condEvalRestClause (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntCar ;
	TLispEntity*	pEntBody ;
	
	assert (pLM != NULL) ;

	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) {
		lispMachineCode_SetState (pLM, &lispMachineState_condFinalize) ;
		return	LMR_CONTINUE ;
	}
	
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_1, &pEntBody) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntBody))) {
		lispMachineCode_SetState (pLM, &lispMachineState_condFinalize) ;
		return	LMR_CONTINUE ;
	}
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntBody, &pEntCar)) ||
		pEntCar == NULL) {
		lispMachineCode_SetError (pLM) ;
		lispMachineCode_SetState (pLM, &lispMachineState_condFinalize) ;
		return	LMR_CONTINUE ;
	}
	lispMachineCode_Cdr   (pLM, LM_LREG_1, LM_LREG_1) ;
	lispMachineCode_Evaln (pLM, pEntCar, &lispMachineState_condEvalRestClause) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_condFinalize (
	register TLispMachine*	pLM)
{
	lispMachineCode_PopLReg (pLM, LM_LREG_2) ;
	lispMachineCode_PopLReg (pLM, LM_LREG_1) ;
	return	LMR_RETURN ;
}

/*
 *	(while TEST BODY...)
 */
TLMRESULT
lispMachineState_While (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntTestBody ;
	TLispEntity*	pEntTest ;
	
	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg  (pLM, LM_LREG_ACC, &pEntTestBody) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntTestBody)) ||
		TFAILED (lispEntity_GetCar (pLispMgr, pEntTestBody, &pEntTest))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMachineCode_PushLReg (pLM, LM_LREG_1) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_2) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_3) ;
	lispMachineCode_Cdr      (pLM, LM_LREG_1, LM_LREG_ACC) ;
	lispMachineCode_MoveLReg (pLM, LM_LREG_2, LM_LREG_1) ;
	lispMachineCode_SetLReg  (pLM, LM_LREG_3, pEntTest) ;
	lispMachineCode_Evaln    (pLM, pEntTest, &lispMachineState_whileTest) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_whileTest (
	register TLispMachine*	pLM)
{
	TLispEntity*	pEntTest ;
	
	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) {
		lispMachineCode_SetState (pLM, &lispMachineState_whileFinalize) ;
		return	LMR_CONTINUE ;
	}
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntTest) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLM->m_pLispMgr, pEntTest))) {
		lispMachineCode_SetState (pLM, &lispMachineState_whileFinalize) ;
	} else {
		lispMachineCode_SetState (pLM, &lispMachineState_whileBody) ;
	}
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_whileBody (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntBodylist ;
	TLispEntity*	pEntBody ;
	
	assert (pLM != NULL) ;
	
	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) {
		lispMachineCode_SetState (pLM, &lispMachineState_whileFinalize) ;
		return	LMR_CONTINUE ;
	}

	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_1, &pEntBodylist) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntBodylist))) {
		TLispEntity*	pEntTest ;
		
		lispMachineCode_MoveLReg (pLM, LM_LREG_1, LM_LREG_2) ;
		lispMachineCode_GetLReg  (pLM, LM_LREG_3, &pEntTest) ;
		lispMachineCode_Evaln    (pLM, pEntTest, &lispMachineState_whileTest) ;
		return	LMR_CONTINUE ;
	}
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntBodylist, &pEntBody))) {
		lispMachineCode_SetError (pLM) ;
		lispMachineCode_SetState (pLM, &lispMachineState_whileFinalize) ;
		return	LMR_CONTINUE ;
	}
	lispMachineCode_Evaln (pLM, pEntBody, &lispMachineState_whileBody) ;
	lispMachineCode_Cdr   (pLM, LM_LREG_1, LM_LREG_1) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_whileFinalize (
	register TLispMachine*	pLM)
{
	lispMachineCode_PopLReg (pLM, LM_LREG_3) ;
	lispMachineCode_PopLReg (pLM, LM_LREG_2) ;
	lispMachineCode_PopLReg (pLM, LM_LREG_1) ;
	return	LMR_RETURN ;
}

/*
 *	(catch TAG BODY...)
 */
TLMRESULT
lispMachineState_Catch (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntTagBody ;
	TLispEntity*	pEntTag ;

	pLispMgr	= pLM->m_pLispMgr ;
	lispMachineCode_GetLReg  (pLM, LM_LREG_ACC, &pEntTagBody) ;
	
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntTagBody)) ||
		TFAILED (lispEntity_GetCar (pLispMgr, pEntTagBody, &pEntTag))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	
	lispMachineCode_PushLReg (pLM, LM_LREG_1) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_2) ;
	lispMachineCode_Cdr      (pLM, LM_LREG_1, LM_LREG_ACC) ;
	lispMachineCode_Evaln    (pLM, pEntTag, &lispMachineState_catchTag) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_catchTag (
	register TLispMachine*	pLM)
{
	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) {
		lispMachineCode_SetState (pLM, &lispMachineState_catchFinalize) ;
		return	LMR_CONTINUE ;
	}
	lispMachineCode_MoveLReg (pLM, LM_LREG_2, LM_LREG_ACC) ;
	lispMachineCode_SetState (pLM, &lispMachineState_catchBody) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_catchBody (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntBodylist ;
	TLispEntity*	pEntBody ;
	
	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) {
		lispMachineCode_SetState (pLM, &lispMachineState_catchFinalize) ;
		return	LMR_CONTINUE ;
	}
	pLispMgr	= pLM->m_pLispMgr ;
	lispMachineCode_GetLReg (pLM, LM_LREG_1, &pEntBodylist) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntBodylist))) {
		lispMachineCode_SetState (pLM, &lispMachineState_catchFinalize) ;
		return	LMR_CONTINUE ;
	}
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntBodylist, &pEntBody))) {
		lispMachineCode_SetError (pLM) ;
		lispMachineCode_SetState (pLM, &lispMachineState_catchFinalize) ;
		return	LMR_CONTINUE ;
	}
	lispMachineCode_Cdr     (pLM, LM_LREG_1, LM_LREG_1) ;
	lispMachineCode_Evaln   (pLM, pEntBody, &lispMachineState_catchBody) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_catchFinalize (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	
	if (LISPMACHINE_EXCEPTIONP (pLM)) {
		TLispEntity*	pEntTag ;
		TLispEntity*	pEntThrownTag ;
		TLispEntity*	pEntValue ;

		lispMachineCode_GetException (pLM, &pEntThrownTag, &pEntValue) ;
		lispMachineCode_GetLReg (pLM, LM_LREG_2, &pEntTag) ;

		if (TSUCCEEDED (lispEntity_Eq (pLispMgr, pEntTag, pEntThrownTag))) {
#if defined (DEBUG_LV99)
			fprintf (stderr, "Catch ") ;
			lispEntity_Print (pLispMgr, pEntTag) ;
			fprintf (stderr, ", ") ;
			lispEntity_Print (pLispMgr, pEntValue) ;
			fprintf (stderr, "\n") ;
#endif
			lispMachineCode_ResetException (pLM) ;
			lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntValue) ;
		}
	}
	lispMachineCode_PopLReg (pLM, LM_LREG_2) ;
	lispMachineCode_PopLReg (pLM, LM_LREG_1) ;
	return	LMR_RETURN ;
}

/*
 *	(condition-case VAR BODYFORM HANDLERS ...)
 */
TLMRESULT
lispMachineState_ConditionCase (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntVar ;
	TLispEntity*	pEntList ;
	TLispEntity*	pEntBodyform ;
	TLispEntity*	pEntHandlers ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntList) ;

	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntList, &pEntVar)) ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pEntList, &pEntList)) ||
		TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntList)) ||
		TFAILED (lispEntity_GetCar (pLispMgr, pEntList, &pEntBodyform)) ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pEntList, &pEntHandlers))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMachineCode_PushLReg (pLM, LM_LREG_1) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_2) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_3) ;
	lispMachineCode_SetLReg  (pLM, LM_LREG_1, pEntVar) ;
	lispMachineCode_SetLReg  (pLM, LM_LREG_2, pEntHandlers) ;
	lispMachineCode_Evaln    (pLM, pEntBodyform, &lispMachineState_conditionCaseAfter) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_conditionCaseAfter (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntVar ;
	TLispEntity*	pEntBound ;
	TLispEntity*	pEntOrgBound ;
	TLispEntity*	pEntHandlers ;
	TLispEntity*	pEntHandler ;
	TLispEntity*	pEntCar ;
	TLispEntity*	pEntSignal ;
	TLispEntity*	pEntSignalValue ;
	register TLMRESULT		lres	= LMR_RETURN ;

	/*	߼ signal  error  quit */
	if (!LISPMACHINE_SIGNALP (pLM)) 
		goto	quit ;

	lispMachineCode_GetSignal (pLM, &pEntSignal, &pEntSignalValue) ;

	pLispMgr	= pLM->m_pLispMgr ;
	lispMachineCode_GetLReg (pLM, LM_LREG_1, &pEntVar) ;
	if (TFAILED (lispMgr_CreateConscell (pLispMgr, pEntSignal, pEntSignalValue, &pEntBound))) {
		lres	= LMR_ERROR ;
		goto	quit ;
	}
	if (TFAILED (lispMachine_GetCurrentSymbolValue (pLM, pEntVar, &pEntOrgBound))) 
		lispMgr_CreateVoid (pLispMgr, &pEntOrgBound) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_3, pEntOrgBound) ;
	lispMachine_SetCurrentSymbolValue (pLM, pEntVar, pEntBound) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_2, &pEntHandlers) ;
#if defined (DEBUG_LV99)
	fprintf (stderr, "handlers = ") ;
	lispEntity_Print (pLispMgr, pEntHandlers) ;
	fprintf (stderr, "\n") ;
#endif
	while (TFAILED (lispEntity_Nullp (pLispMgr, pEntHandlers))) {
		if (TFAILED (lispEntity_GetCar (pLispMgr, pEntHandlers, &pEntHandler)) ||
			TFAILED (lispEntity_GetCar (pLispMgr, pEntHandler,  &pEntCar)))
			goto	quit ;
		if (pEntCar == pEntSignal) {
			TLispEntity*	pEntNil ;
			
			lispEntity_GetCdr (pLispMgr, pEntHandler, &pEntHandler) ;
			lispMachineCode_SetLReg (pLM, LM_LREG_2, pEntHandler) ;
			lispMgr_CreateNil (pLispMgr, &pEntNil) ;
			lispMachineCode_SetLReg  (pLM, LM_LREG_ACC, pEntNil) ;
			lispMachineCode_SetState (pLM, &lispMachineState_conditionCaseHandler) ;
			lispMachineCode_ResetSignal (pLM) ;
			return	LMR_CONTINUE ;
		}
		lispEntity_GetCdr (pLispMgr, pEntHandlers, &pEntHandlers) ;
	}
 quit:
	lispMachineCode_PopLReg (pLM, LM_LREG_3) ;
	lispMachineCode_PopLReg (pLM, LM_LREG_2) ;
	lispMachineCode_PopLReg (pLM, LM_LREG_1) ;
	return	lres ;
}

TLMRESULT
lispMachineState_conditionCaseHandler (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntBodylist ;
	TLispEntity*	pEntBody ;

	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) {
		lispMachineCode_SetState (pLM, &lispMachineState_conditionCaseFinalize) ;
		return	LMR_CONTINUE ;
	}
	pLispMgr	= pLM->m_pLispMgr ;
	lispMachineCode_GetLReg (pLM, LM_LREG_2, &pEntBodylist) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntBodylist))) {
		lispMachineCode_SetState (pLM, &lispMachineState_conditionCaseFinalize) ;
		return	LMR_CONTINUE ;
	}
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntBodylist, &pEntBody))) {
		lispMachineCode_SetError (pLM) ;
		lispMachineCode_SetState (pLM, &lispMachineState_conditionCaseFinalize) ;
		return	LMR_CONTINUE ;
	}
	lispMachineCode_Cdr (pLM, LM_LREG_2, LM_LREG_2) ;
	lispMachineCode_Evaln (pLM, pEntBody, &lispMachineState_conditionCaseHandler) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_conditionCaseFinalize (
	register TLispMachine*	pLM)
{
	TLispEntity*	pEntVar ;
	TLispEntity*	pEntOrgBound ;

	lispMachineCode_GetLReg (pLM, LM_LREG_3, &pEntOrgBound) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_1, &pEntVar) ;
	lispMachine_SetCurrentSymbolValue (pLM, pEntVar, pEntOrgBound) ;
	lispMachineCode_PopLReg (pLM, LM_LREG_3) ;
	lispMachineCode_PopLReg (pLM, LM_LREG_2) ;
	lispMachineCode_PopLReg (pLM, LM_LREG_1) ;
	return	LMR_RETURN ;
}

/*
 *
 */
TLMRESULT
lispMachineState_Progn (
	register TLispMachine*	pLM)
{
	assert (pLM != NULL) ;

	lispMachineCode_PushLReg (pLM, LM_LREG_1) ;
	lispMachineCode_MoveLReg (pLM, LM_LREG_1, LM_LREG_ACC) ;
	lispMachineCode_SetState (pLM, &lispMachineState_prognStep) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_prognStep (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntCAR ;

	assert (pLM != NULL) ;

	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) {
		lispMachineCode_PopLReg (pLM, LM_LREG_1) ;
		return	LMR_RETURN ;
	}
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_1, &pEntArglist) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntArglist))) {
		lispMachineCode_PopLReg (pLM, LM_LREG_1) ;
		return	LMR_RETURN ;
	}
	(void) lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntCAR) ;
	assert (pEntCAR != NULL) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntCAR) ;
	lispMachineCode_Cdr     (pLM, LM_LREG_1, LM_LREG_1) ;
	lispMachineCode_Evaln   (pLM, pEntCAR, &lispMachineState_prognStep) ;
	return	LMR_CONTINUE ;
}

/*
 *	(prog1 OBJ ...)
 *
 *	Prog1 ưϼ̤Ǥ롣
 *
 *		(1) Prog1
 *			prog1 ɾ줿ɬ롣
 *		(2) prog1Step1
 *			ƬΥ֥Ȥɾ줿塢ƤӽФ롣
 *		(3) prog1Step2
 *			ĤΥ֥Ȥɾ򷫤֤
 *		(4) prog1Finalize
 *			prog1 νλ
 *
 *	(1) -> (2) -> (3)η֤ -> (4)λ
 *	Ȥʤ롣
 */
TLMRESULT
lispMachineState_Prog1 (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntBodylist ;
	TLispEntity*	pEntBody ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg  (pLM, LM_LREG_ACC, &pEntBodylist) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntBodylist)) ||
		TFAILED (lispEntity_GetCar (pLispMgr, pEntBodylist, &pEntBody))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMachineCode_PushLReg (pLM, LM_LREG_1) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_2) ;
	lispMachineCode_Cdr      (pLM, LM_LREG_1, LM_LREG_ACC) ;
	lispMachineCode_Evaln    (pLM, pEntBody, &lispMachineState_prog1Step1) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_prog1Step1 (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntBodylist ;
	TLispEntity*	pEntBody ;

	assert (pLM != NULL) ;
	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) {
		lispMachineCode_SetState (pLM, &lispMachineState_prog1Finalize) ;
		return	LMR_CONTINUE ;
	}
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg  (pLM, LM_LREG_1, &pEntBodylist) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntBodylist))) {
		lispMachineCode_SetState (pLM, &lispMachineState_prog1Finalize) ;
		return	LMR_CONTINUE ;
	}
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntBodylist, &pEntBody))) {
		lispMachineCode_SetError (pLM) ;
		lispMachineCode_SetState (pLM, &lispMachineState_prog1Finalize) ;
		return	LMR_CONTINUE ;
	}
	lispMachineCode_MoveLReg (pLM, LM_LREG_2, LM_LREG_ACC) ;
	lispMachineCode_Cdr      (pLM, LM_LREG_1, LM_LREG_1) ;
	lispMachineCode_Evaln    (pLM, pEntBody, &lispMachineState_prog1Step2) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_prog1Step2 (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntBodylist ;
	TLispEntity*	pEntBody ;

	assert (pLM != NULL) ;

	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) {
		lispMachineCode_SetState (pLM, &lispMachineState_prog1Finalize) ;
		return	LMR_CONTINUE ;
	}
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg  (pLM, LM_LREG_1, &pEntBodylist) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntBodylist))) {
		lispMachineCode_MoveLReg (pLM, LM_LREG_ACC, LM_LREG_2) ;
		lispMachineCode_SetState (pLM, &lispMachineState_prog1Finalize) ;
		return	LMR_CONTINUE ;
	}
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntBodylist, &pEntBody))) {
		lispMachineCode_SetError (pLM) ;
		lispMachineCode_SetState (pLM, &lispMachineState_prog1Finalize) ;
		return	LMR_CONTINUE ;
	}
	lispMachineCode_Cdr   (pLM, LM_LREG_1, LM_LREG_1) ;
	lispMachineCode_Evaln (pLM, pEntBody, &lispMachineState_prog1Step2) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_prog1Finalize (
	register TLispMachine* pLM)
{
	assert (pLM != NULL) ;

	lispMachineCode_PopLReg (pLM, LM_LREG_2) ;
	lispMachineCode_PopLReg (pLM, LM_LREG_1) ;
	return	LMR_RETURN ;
}

/*	(prog2 X Y BODY...)
 */
TLMRESULT
lispMachineState_Prog2 (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntBodylist ;
	TLispEntity*	pEntBody ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg  (pLM, LM_LREG_ACC, &pEntBodylist) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntBodylist)) ||
		TFAILED (lispEntity_GetCar (pLispMgr, pEntBodylist, &pEntBody))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMachineCode_PushLReg (pLM, LM_LREG_1) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_2) ;
	lispMachineCode_Cdr      (pLM, LM_LREG_1, LM_LREG_ACC) ;
	lispMachineCode_Evaln    (pLM, pEntBody, &lispMachineState_prog2Step1) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_prog2Step1 (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntBodylist ;
	TLispEntity*	pEntBody ;

	assert (pLM != NULL) ;
	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) {
		lispMachineCode_SetState (pLM, &lispMachineState_prog1Finalize) ;
		return	LMR_CONTINUE ;
	}
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg  (pLM, LM_LREG_1, &pEntBodylist) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntBodylist))) {
		lispMachineCode_SetState (pLM, &lispMachineState_prog1Finalize) ;
		return	LMR_CONTINUE ;
	}
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntBodylist, &pEntBody))) {
		lispMachineCode_SetError (pLM) ;
		lispMachineCode_SetState (pLM, &lispMachineState_prog1Finalize) ;
		return	LMR_CONTINUE ;
	}
	lispMachineCode_MoveLReg (pLM, LM_LREG_2, LM_LREG_ACC) ;
	lispMachineCode_Cdr      (pLM, LM_LREG_1, LM_LREG_1) ;
	lispMachineCode_Evaln    (pLM, pEntBody, &lispMachineState_prog1Step1) ;
	return	LMR_CONTINUE ;
}

/*
 *	(setq SYM VAL SYM VAL ...)
 */
TLMRESULT
lispMachineState_Setq (
	register TLispMachine*	pLM)
{
	assert (pLM != NULL) ;

	lispMachineCode_PushVReg (pLM, LM_VREG_1) ;
	lispMachineCode_SetVRegP (pLM, LM_VREG_1, &lispMachine_SetCurrentSymbolValue) ;
	return	lispMachineState_setqCommon (pLM) ;
}

TLMRESULT
lispMachineState_SetqDefault (
	register TLispMachine*	pLM)
{
	assert (pLM != NULL) ;

	lispMachineCode_PushVReg (pLM, LM_VREG_1) ;
	lispMachineCode_SetVRegP (pLM, LM_VREG_1, &lispMachine_SetGlobalSymbolValue) ;
	return	lispMachineState_setqCommon (pLM) ;
}

TLMRESULT
lispMachineState_setqCommon (
	register TLispMachine* pLM)
{
	lispMachineCode_PushLReg (pLM, LM_LREG_1) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_2) ;
	lispMachineCode_MoveLReg (pLM, LM_LREG_1, LM_LREG_ACC) ;
	lispMachineCode_SetState (pLM, &lispMachineState_setqGetSym) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_setqGetSym (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pSym ;
	TLispEntity*	pVal ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pNextArglist ;

	assert (pLM      != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_1, &pEntArglist) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntArglist))) {
		lispMachineCode_SetState (pLM, &lispMachineState_setqFinalize) ;
		return	LMR_CONTINUE ;
	}
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pSym)) ||
		TFAILED (lispEntity_Symbolp (pLispMgr, pSym)) ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pEntArglist, &pNextArglist)) ||
		TFAILED (lispEntity_GetCar (pLispMgr, pNextArglist, &pVal))) {
		lispMachineCode_SetError (pLM) ;
		lispMachineCode_SetState (pLM, &lispMachineState_setqFinalize) ;
		return	LMR_CONTINUE ;
	}
	lispEntity_GetCdr (pLispMgr, pNextArglist, &pEntArglist) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_1, pEntArglist) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_2, pSym) ;
	lispMachineCode_Evaln   (pLM, pVal, &lispMachineState_setqSetVal) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_setqSetVal (
	register TLispMachine* pLM)
{
	TLispEntity*	pEntSym ;
	TLispEntity*	pEntVal ;
	Boolean			(*pSetFunc)(TLispMachine*, TLispEntity*, TLispEntity*) ;

	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) {
		lispMachineCode_SetState (pLM, &lispMachineState_setqFinalize) ;
		return	LMR_CONTINUE ;
	}
	lispMachineCode_GetLReg    (pLM, LM_LREG_2,   &pEntSym) ;
	lispMachineCode_GetLReg    (pLM, LM_LREG_ACC, &pEntVal) ;
	lispMachineCode_GetVRegP   (pLM, LM_VREG_1,   (void *)&pSetFunc) ;
	assert (pSetFunc != NULL) ;
	if (TFAILED ((pSetFunc) (pLM, pEntSym, pEntVal))) {
		lispMachineCode_SetError (pLM) ;
		lispMachineCode_SetState (pLM, &lispMachineState_setqFinalize) ;
	} else {
#if defined (DEBUG_LV99)
		fprintf (stderr, "setq: ") ;
		lispEntity_Print (pLM->m_pLispMgr, pEntSym) ;
		fprintf (stderr, " <- ") ;
		lispEntity_Print (pLM->m_pLispMgr, pEntVal) ;
		fprintf (stderr, "\n") ;
#endif
		lispMachineCode_SetState (pLM, &lispMachineState_setqGetSym) ;
	}
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_setqFinalize (
	register TLispMachine* pLM)
{
	lispMachineCode_PopLReg (pLM, LM_LREG_2) ;
	lispMachineCode_PopLReg (pLM, LM_LREG_1) ;
	lispMachineCode_PopVReg (pLM, LM_VREG_1) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Defun (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	register TLispEntity*	pLambda ;
	TLispEntity*	pLambdaForm ;
	TLispEntity*	pName ;

	pLispMgr	= pLM->m_pLispMgr ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pLambdaForm) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pLambdaForm, &pName)) ||
		TFAILED (lispEntity_Symbolp (pLispMgr, pName))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	pLambda	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_LAMBDA) ;
	assert (pLambda != NULL) ;
	lispEntity_SetCar (pLispMgr, pLambdaForm, pLambda) ;
	if (TFAILED (lispMachine_SetSymbolFunctionValue (pLM, pName, pLambdaForm))) 
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pName) ;
	return	LMR_RETURN ;
}

/*
 *	(defvar SYMBOL INITVALUE DOCSTRING)
 *	
 *	SYMBOL ѿȤ롣ѿȤ뤳Ȥ
 *	׵ᤵƤʤƤȥɥơ
 *	ͤʤ󤫤Ǥƴ򤷤
 */
TLMRESULT
lispMachineState_Defvar (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntSymbol ;
	TLispEntity*	pEntValue ;
	TLispEntity*	pEntInitval ;
	TLispEntity*	pCddr ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntSymbol)) ||
		TFAILED (lispEntity_Symbolp (pLispMgr, pEntSymbol))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	/*	global-variable ¸ߤƤСѤͿʤ*/
	if (TSUCCEEDED (lispMachine_GetGlobalSymbolValue (pLM, pEntSymbol, &pEntValue)) &&
		TFAILED (lispEntity_Voidp (pLispMgr, pEntValue))) {
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntSymbol) ;
		return	LMR_RETURN ;
	}
	(void) lispEntity_GetCdr (pLispMgr, pEntArglist, &pEntArglist) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntArglist))) {
		/*	SYMBOL  buffer-local ξˤϡȤͤ
		 *	void ǤƶͿʤ*/
		lispMachine_SetGlobalSymbolValue (pLM, pEntSymbol, pEntArglist) ;
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntSymbol) ;
		return	LMR_RETURN ;
	}
	/*	˰οå롣*/
	if (TFAILED (lispEntity_GetCddr (pLispMgr, pEntArglist, &pCddr)) ||
		TFAILED (lispEntity_Nullp (pLispMgr, pCddr))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMachineCode_PushLReg (pLM, LM_LREG_1) ;
	lispMachineCode_SetLReg  (pLM, LM_LREG_1,   pEntSymbol) ;
	(void) lispEntity_GetCar (pLispMgr, pEntArglist, &pEntInitval) ;
	lispMachineCode_Evaln    (pLM, pEntInitval, &lispMachineState_defvarPostEvalInitvalue) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_defvarPostEvalInitvalue (register TLispMachine* pLM)
{
	if (!LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) {
		TLispEntity*	pEntSymbol ;
		TLispEntity*	pInitvalue ;
	
		lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pInitvalue) ;
		lispMachineCode_GetLReg (pLM, LM_LREG_1,   &pEntSymbol) ;
		lispMachine_SetGlobalSymbolValue (pLM, pEntSymbol, pInitvalue) ;
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntSymbol) ;
	}
	lispMachineCode_PopLReg (pLM, LM_LREG_1) ;
	return	LMR_RETURN ;
}

/*
 *	unwind-protect
 *
 *	UNWINDFORMS ݸʤ()BODYFORM ¹Ԥ롣
 *	(unwind-protect BODYFORM UNWINDFORMS ...) Τ褦˻Ȥ롣
 *	⤷ BODYFORM ̤˴λСͤ UNWINDFORMS μ¹Ը
 *	֤롣⤷ BODYFORM  nonlocally ȴСUNWINDFORMS 
 *	Ƥ˳Ѽ¹Ԥ롣
 */
TLMRESULT
lispMachineState_UnwindProtect (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntBodyform ;
	TLispEntity*	pEntUnwindForms ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntArglist))) {
		/*	BODY ¸ߤƤʤФʤʤ*/
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispEntity_GetCar (pLispMgr, pEntArglist,  &pEntBodyform) ;
	lispEntity_GetCdr (pLispMgr, pEntArglist,  &pEntUnwindForms) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_1) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_2) ;
	lispMachineCode_SetLReg  (pLM, LM_LREG_1, pEntUnwindForms) ;
	lispMachineCode_Evaln    (pLM, pEntBodyform, &lispMachineState_unwindProtectAfterEvalBodyform) ;
	return	LMR_CONTINUE ;
}

/*
 *	BODYFORM ɾ줿ʬǤ롣SIGNAL ʤɤäƤƤ
 *	괺 UNWINDFORMS ɾ롣
 */
TLMRESULT
lispMachineState_unwindProtectAfterEvalBodyform (
	register TLispMachine* pLM)
{
	TLispEntity*	pEntSignal ;
	TLispEntity*	pEntSignalValue ;
	TLispEntity*	pEntException ;
	TLispEntity*	pEntExceptionValue ;

#if defined (DEBUG)
	fprintf (stderr, "unwind-protect-after-eval-bodyform\n") ;
#endif
	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) {
		/*	ξˤϡߤ FLAG ξ򵭲ޤ UNWINDFORMS
		 *	ɾơȤ᤹ɬפ롣*/
		lispMachineCode_PushLReg    (pLM, LM_LREG_3) ;
		lispMachineCode_PushLReg    (pLM, LM_LREG_4) ;
		lispMachineCode_PushLReg    (pLM, LM_LREG_5) ;
		lispMachineCode_PushLReg    (pLM, LM_LREG_6) ;
		lispMachineCode_GetSignal   (pLM, &pEntSignal, &pEntSignalValue) ;
		lispMachineCode_SetLReg     (pLM, LM_LREG_3, pEntSignal) ;
		lispMachineCode_SetLReg     (pLM, LM_LREG_4, pEntSignalValue) ;
		lispMachineCode_GetException (pLM, &pEntException, &pEntExceptionValue) ;
		lispMachineCode_SetLReg     (pLM, LM_LREG_5, pEntException) ;
		lispMachineCode_SetLReg     (pLM, LM_LREG_6, pEntExceptionValue) ;
		lispMachineCode_ResetSignal    (pLM) ;
		lispMachineCode_ResetException (pLM) ;
		lispMachineCode_PushState (pLM, &lispMachineState_unwindProtectAfterEvalUnwindForms) ;
	}
	lispMachineCode_SetState (pLM, &lispMachineState_prog1Step1) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_unwindProtectAfterEvalUnwindForms (
	register TLispMachine* pLM)
{
	TLispEntity*	pEntSignal ;
	TLispEntity*	pEntSignalValue ;
	TLispEntity*	pEntException ;
	TLispEntity*	pEntExceptionValue ;

#if defined (DEBUG)
	fprintf (stderr, "unwind-protect-after-eval-unwindforms\n") ;
#endif
	lispMachineCode_GetLReg      (pLM, LM_LREG_3, &pEntSignal) ;
	lispMachineCode_GetLReg      (pLM, LM_LREG_4, &pEntSignalValue) ;
	lispMachineCode_SetSignal    (pLM, pEntSignal, pEntSignalValue) ;
	lispMachineCode_GetLReg      (pLM, LM_LREG_5, &pEntException) ;
	lispMachineCode_GetLReg      (pLM, LM_LREG_6, &pEntExceptionValue) ;
	lispMachineCode_SetException (pLM, pEntException, pEntExceptionValue) ;
	lispMachineCode_PopLReg    (pLM, LM_LREG_6) ;
	lispMachineCode_PopLReg    (pLM, LM_LREG_5) ;
	lispMachineCode_PopLReg    (pLM, LM_LREG_4) ;
	lispMachineCode_PopLReg    (pLM, LM_LREG_3) ;
	return	LMR_RETURN ;
}

/*
 *	(save-excursion &rest BODY)
 *
 *	point, mark, current-buffer ¸ơBODY ¹Ԥ
 *	θ夽롣BODY μ¹Ԥ progn ƱǤ롣
 *	point  mark, current buffer ͤ㤨㳰ȯ
 *	ư۾ｪλǤ롣
 *	ޡͭ/̵ξ֤ޤ롣
 *
 *	 construct  deactivate-mark ¸ʤ
 */
TLMRESULT
lispMachineState_SaveExcursion (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pEntBuffer ;
	TLispEntity*			pEntMkBuffer ;
	TLispEntity*			pEntMkPointBak ;
	TLispEntity*			pEntMkPoint ;
	TLispEntity*			pEntMkMark ;
	TLispEntity*			pEntMkMarkBak ;
	int						nPos ;
	
	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	assert (pEntBuffer != NULL) ;

	/*	point-marker  backup marker ڤ mark-marker  backup marker
	 *	롣*/
	if (TFAILED (lispMgr_CreateMarker (pLispMgr, &pEntMkPointBak)) ||
		TFAILED (lispMgr_CreateMarker (pLispMgr, &pEntMkMarkBak)))
		return	LMR_ERROR ;
	if (TFAILED (lispBuffer_PointMarker (pLispMgr, pEntBuffer, &pEntMkPoint)))
		return	LMR_ERROR ;
	(void) lispMarker_GetBufferPosition (pLispMgr, pEntMkPoint, NULL, &nPos) ;
	lispBuffer_AddMarker (pLispMgr, pEntBuffer, pEntMkPointBak) ;
	lispMarker_SetBufferPosition (pLispMgr, pEntMkPointBak, pEntBuffer, nPos) ;
#if defined (DEBUG)
	fprintf (stderr, "(save-excursion) point_back = ") ;
	lispMarker_Print (pLispMgr, pEntMkPointBak) ;
	fprintf (stderr, "\n(save-excursion) point = ") ;
	lispMarker_Print (pLispMgr, pEntMkPoint) ;
	fprintf (stderr, "\n") ;
#endif
	if (TSUCCEEDED (lispBuffer_MarkMarker  (pLispMgr, pEntBuffer, &pEntMkMark)) &&
		TSUCCEEDED (lispMarker_GetBufferPosition (pLispMgr, pEntMkMark, &pEntMkBuffer, &nPos)) &&
		pEntMkBuffer != NULL) {
		lispBuffer_AddMarker (pLispMgr, pEntBuffer, pEntMkMarkBak) ;
		lispMarker_SetBufferPosition (pLispMgr, pEntMkMarkBak, pEntBuffer, nPos) ;
	}

	lispMachineCode_PushLReg (pLM, LM_LREG_1) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_2) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_3) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_1, pEntBuffer) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_2, pEntMkPointBak) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_3, pEntMkMarkBak) ;
	lispMachineCode_SetState (pLM, &lispMachineState_Progn) ;
	lispMachineCode_PushState (pLM, &lispMachineState_saveExcursionFinalize) ;
	return	LMR_CONTINUE ;
}

/*
 *	(save-excursion &rest BODY)  BODY ʬμ¹Ԥλơ
 *	current-buffer, point, mark ᤵô롣
 *	exception, signal ϰݻ롣
 */
TLMRESULT
lispMachineState_saveExcursionFinalize (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntMkPointBak ;
	TLispEntity*	pEntMkMarkBak ;
	TLispEntity*	pEntMkPoint ;
	TLispEntity*	pEntMkMark ;
	TLispEntity*	pEntMkBuffer ;
	TLispEntity*	pEntBuffer ;
	int				nMkPos ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_1, &pEntBuffer) ;
	assert (pEntBuffer != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_2, &pEntMkPointBak) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_3, &pEntMkMarkBak) ;

	/*	point-marker */
	lispMarker_GetBufferPosition (pLispMgr, pEntMkPointBak, &pEntMkBuffer, &nMkPos) ;
	assert (pEntMkBuffer != NULL) ;
	lispBuffer_PointMarker (pLispMgr, pEntBuffer, &pEntMkPoint) ;
#if defined (DEBUG)
	fprintf (stderr, "(save-excursion-fin) point = ") ;
	lispMarker_Print (pLispMgr, pEntMkPoint) ;
	fprintf (stderr, "\n(save-excursion-fin) point_bak = ") ;
	lispMarker_Print (pLispMgr, pEntMkPointBak) ;
	fprintf (stderr, "\n") ;
#endif
	lispMarker_SetBufferPosition (pLispMgr, pEntMkPoint, pEntMkBuffer, nMkPos) ;
	lispBuffer_RemoveMarker (pLispMgr, pEntMkPointBak) ;

	/*	mark-marker */
	lispBuffer_MarkMarker (pLispMgr, pEntBuffer, &pEntMkMark) ;
	lispMarker_GetBufferPosition (pLispMgr, pEntMkMark, &pEntMkBuffer, NULL) ;
	if (pEntMkMark != NULL)
		lispBuffer_RemoveMarker (pLispMgr, pEntMkMark) ;

	lispMarker_GetBufferPosition (pLispMgr, pEntMkMarkBak, &pEntMkBuffer, &nMkPos) ;
	if (pEntMkBuffer != NULL) {
		lispBuffer_AddMarker (pLispMgr, pEntMkBuffer, pEntMkMark) ;
		lispMarker_SetBufferPosition (pLispMgr, pEntMkMark, pEntMkBuffer, nMkPos) ;
		lispBuffer_RemoveMarker (pLispMgr, pEntMkMarkBak) ;
	}

	lispMachineCode_SetCurrentBuffer (pLM, pEntBuffer) ;

	lispMachineCode_PopLReg (pLM, LM_LREG_3) ;
	lispMachineCode_PopLReg (pLM, LM_LREG_2) ;
	lispMachineCode_PopLReg (pLM, LM_LREG_1) ;
	return	LMR_RETURN ;
}

/*
 *	(save-restriction &rest BODY)
 *
 *	save-restriction  special form Ǥ롣ȥХåե restriction 
 *	¸ BODY ¹ԡ᤹Хåե restriction ϸʤ
 *	Ƚλʬʤ롣(narrow-to-region ǽ̤ơwiden Ǥɤ)
 */
TLMRESULT
lispMachineState_SaveRestriction (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntBuffer ;
	int				nPointMin, nPointMax ;
	TLispEntity*	pEntPointMin ;
	TLispEntity*	pEntPointMax ;

	lispMachineCode_PushLReg (pLM, LM_LREG_2) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_3) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_4) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_1) ;

	lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	lispMachineCode_SetLReg  (pLM, LM_LREG_2, pEntBuffer) ;
	/*	ХåեԽǽΰ롣*/
	lispBuffer_PointMin (pLispMgr, pEntBuffer, &nPointMin) ;
	lispBuffer_PointMax (pLispMgr, pEntBuffer, &nPointMax) ;
	/*	ޡˤ¸롣*/
	if (TFAILED (lispMachineCode_CreateMarker (pLM, pEntBuffer, nPointMin, &pEntPointMin)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_3, pEntPointMin) ;
	if (TFAILED (lispMachineCode_CreateMarker (pLM, pEntBuffer, nPointMax, &pEntPointMax)))
		return	LMR_ERROR ;
	lispMarker_SetInsertionType (pLispMgr, pEntPointMax, True) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_4, pEntPointMax) ;
#if defined (DEBUG)
	lispBuffer_PointMin (pLispMgr, pEntBuffer, &nPointMin) ;
	lispBuffer_PointMax (pLispMgr, pEntBuffer, &nPointMax) ;
	fprintf (stderr, "(save-restriction)[min ... max] = [%d ... %d]\n",
			 nPointMin, nPointMax) ;
#endif
	lispMachineCode_MoveLReg  (pLM, LM_LREG_1, LM_LREG_ACC) ;
	lispMachineCode_PushState (pLM, &lispMachineState_saveRestrictionFinalize) ;
	lispMachineCode_SetState  (pLM, &lispMachineState_prognStep) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_saveRestrictionFinalize (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntBuffer ;
	TLispEntity*	pEntPointMin ;
	TLispEntity*	pEntPointMax ;
	int				nPointMin, nPointMax ;

	/*	Хåե restriction 򸵤᤹*/
	lispMachineCode_GetLReg (pLM, LM_LREG_2, &pEntBuffer) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_3, &pEntPointMin) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_4, &pEntPointMax) ;
	lispMarker_GetBufferPosition (pLispMgr, pEntPointMin, NULL, &nPointMin) ;
	lispMarker_GetBufferPosition (pLispMgr, pEntPointMax, NULL, &nPointMax) ;
	lispBuffer_Narrow (pLispMgr, pEntBuffer, nPointMin, nPointMax) ;
#if defined (DEBUG)
	lispBuffer_PointMin (pLispMgr, pEntBuffer, &nPointMin) ;
	lispBuffer_PointMax (pLispMgr, pEntBuffer, &nPointMax) ;
	fprintf (stderr, "(/save-restriction) [min ... max] = [%d ... %d]\n",
			 nPointMin, nPointMax) ;
#endif
	lispMachineCode_PopLReg (pLM, LM_LREG_4) ;
	lispMachineCode_PopLReg (pLM, LM_LREG_3) ;
	lispMachineCode_PopLReg (pLM, LM_LREG_2) ;
	return	LMR_RETURN ;
}



