module SkyPeer
  class HTAccessParseError < Exception; end
  ALLOW = 0; DENY = 1, MUTUAL_FAILURE = 2;

  class HTAccess

    def initialize( conffile )
      lines = File::open( conffile, 'r' ) { |io| io.read( ); }
      lines = lines.split( /#{$/}/ );
      @root = Block.new( lines );
    end
    
    def check( req )
      @root.check( req );
    end

  end

  class Block
    require 'ipaddr';

    def initialize( lines )
      @children = Array.new( );
      @allow_hosts = Array.new( );
      @deny_hosts = Array.new( );
      @order = ALLOW;
      @arg = //;
      parse_lines( lines );
    end

    def parse_lines( lines )
      while( line = lines.shift )
	line.strip!( );
	if( line =~ /^<(\w\S+)\s+([^>]+)>/ )
	  name = $1;
	  arg = $2;
	  unless( valid_child?( name ) ) then
	    raise HTAccessParseError, "Error: #{line}";
	  end
	  lines1 = Array.new( );
	  while( ( line = lines.shift ) !~ /^\s*<\/#{name}>/ && !lines.empty? ) do
	    lines1 << line;
	  end
	  case name
	  when 'Directory'
	    block = DirectoryBlock.new( arg, lines1 );
	  when 'DirectoryMatch'
	    block = DirecotryMatch.new( arg, lines1 )
	  when 'Files'
	    block = FilesBlock.new( arg, lines1 );
	  when 'FilesMatch'
	    block = FilesMatchBlock.new( arg, lines1 );
	  when 'Limit'
	    block = LimitBlock.new( arg, lines1 );
	  else
	    raise HTAccessParseError, "ERROR: #{name}";
	  end
	  @children.push( block );
	elsif( line =~ /^order\s+allow\s*,\s*deny$/i ) then
	  @order = ALLOW;
	elsif( line =~ /^order\s+deny\s*,\s*allow$/i ) then	  
	  @order = DENY;
	elsif( line =~ /^order\s+mutual-failure$/i ) then	  
	  @order = MUTUAL_FAILURE;
	elsif( line =~ /^deny\s+from\s+(.+)$/ ) then
	  $1.split( /\s+/ ).each { |host|
	    @allow_hosts.push( get_host_expression( host ) );
	  }
	elsif( line =~ /^allow\s+from\s+(.*)$/ ) then	  
	  $1.split( /\s+/ ).each { |host|
	    @deny_hosts.push( get_host_expression( host ) );
	  }
	elsif( line =~ /^(?:#.*)?$/ ) then
	elsif( line =~ /^<\/[^>]+>/ ) then
	else
	  raise HTAccessParseError, "#{line}"
	end
      end
    end
    
    def valid_child?( name )
      return( true );
    end
    
    def get_host_expression( host )
      if( %r!\d+\.\d+\.\d+\.\d+/(\d+\.\d+\.\d+\.\d+|\d+)! =~ host ) then
	host = IPAddr.new( host );
      elsif( host =~ /all/i ) then host = //;
      elsif( host =~ /^\.\w/ ) then host = /#{host}$/;
      elsif( host =~ /\w\.$/ ) then host = /^#{host}/;
      end
      return( host );
    end

    def match?( req )
      return( true );
    end

    def check( req )
      if( match?( req ) ) then
	@children.each { |child|
	  child.check( req );
	}
	if( @order == DENY ) then
	  @deny_hosts.each { |deny_host|
	    denied_host?( deny_host, req.host, req.addr );
	  }
	elsif( @order == ALLOW ) then
	  @allow_hosts.each { |allow_host|
	    return if( allowed_host?( allow_host, req.host ) );
	  }
	elsif( @order == MUTUAL_FAILURE ) then
	  @allow_hosts.each { |allow_host|
	    return if( allowed_host?( allow_host, req.host ) );
	  }
	  @deny_hosts.each { |deny_host|
	    denied_host?( deny_host, req.host );
	  }
	end
      end
    end
    
    def allowed_host?( ref_host, host )
      if( ref_host.class == IPAddr && ref_host.include?( host ) ) then
	return( true );
      elsif( ref_host.class == Regexp && ref_host.match( host ) ) then 
	return( true );
      elsif( ref_host.class == String && ref_host == host ) then
	return( true );
      end
      return( false );
    end

    def denied_host?( ref_host, host )
      if( ref_host.class == IPAddr && ref_host.include?( host ) ) then
	raise HTTPStatus::Forbidden; 
      elsif( ref_host.class == Regexp && ref_host.match( host ) ) then 
	raise HTTPStatus::Forbidden; 
      elsif( ref_host.class == String && ref_host == host ) then
	raise HTTPStatus::Forbidden; 
      end
    end
  end

  ## Directory, DirectoryMatch は使えない
  class DirectoryBlock < Block

    def initialize( arg, lines )
      super( lines );
      @arg = parse_arg( arg );
    end

    def parse_arg( arg )
      if( arg =~ /^~\s+"?([^\"]+)"?/ )
	return( /#{File::expand_path( $1 )}/ );
      elsif( arg =~ /^"?([^\"]+)"?/ )
	return( File::expand_path( $1 ) );
      else
	raise HTAccessParseError, "#{arg}"
      end
    end
    private :parse_arg;

    def match?( req )
      dirname = File::dirname( req.path );
      p dirname;
      case @arg.class
      when String
	return( File::fnmatch( @arg, dirname ) );
      when Regexp
	return( @arg =~ dirname )
      end
    end

    def valid_child?( name )
      if( name =~ /^Directory(?:Match)?$/ ) then return( false ); end
      return( true );
    end
  end

  class DirectoryMatchBlock < Block
    
    def initialize( arg, lines );
      super( lines );
      @arg = parse_arg( arg );
    end

    def parse_arg( arg )
      if( arg =~ /^"?([^\"]+)"?/ )
	return( /^#{File::expand_path( $1 )}/ );
      else
	raise HTAccessParseError, "#{arg}"
      end
    end
    private :parse_arg;

    def match?( req )
      dirname = File::dirname( req.path );
      return( @arg =~ File::dirname( dirname ) );
    end

    def valid_child?( name )
      if( name =~ /^Directory(?:Match)?$/ ) then return( false ); end
      return( true );
    end

  end

  class FilesBlock < Block
    def initialize( arg, lines );
      super( lines );
      @arg = parse_arg( arg );
    end

    def parse_arg( arg )
      if( arg =~ /^~\s+"?([^\"]+)"?/ )
	return( /#{$1}/ );
      elsif( arg =~ /^"?([^\"]+)"?/ )
	return( $1 );
      else
	raise HTAccessParseError, "#{arg}"
      end
    end
    private :parse_arg;

    def match?( req )
      baseame = File::basename( req.path );
      case @arg.class
      when String
	return( File::fnmatch( @arg, basename ) );
      when Regexp
	return( @arg =~ basename )
      end
    end

    def valid_child?( name )
      if( name =~ /^Files(?:Match)?$/ ) then return( false ); end
      return( true );
    end

  end

  class FilesMatchBlock < Block
    def initialize( arg, lines );
      super( lines );
      @arg = parse_arg( arg );
    end

    def parse_arg( arg )
      if( arg =~ /"?([^\"]+)"?/ )
	return( /#{$1}/ );
      else
	raise HTAccessParseError, arg
      end
    end
    private :parse_arg;

    def match?( req )
      return( @arg =~ File::basename( req.path ) );
    end

    def valid_child?( name )
      if( name =~ /^Files(?:Match)?$/ ) then return( false ); end
      return( true );
    end

  end

  class LimitBlock
    METHODS = [ 'GET', 'POST', 'HEAD', 'PUT', 'DELETE', 'CONNECT', 'OPTIONS' ];
    def initialize( arg, lines );
      super( lines );
      @arg = parse_arg( arg );
    end

    def parse_arg( arg )
      methods = Array.new;
      arg.split( /\s+/ ) { |method|
	method.upcase!( );
	if( METHODS.include?( method ) ) then
	  methods.push( method );
	else
	  raise HTAccessParseError, arg;
	end
	return( methods.uniq );
      }
    end
    private :parse_arg;

    def valid_child?( name )
      if( name =~ /^Files(?:Match)?$/ ) then return( false ); end
      return( true );
    end

    def match?( req )
      return( @arg.include?( req.request_method.upcase( ) ) );
    end
  end

end
