/******************************************************************************/
/* SlunkCrypt, by LoRd_MuldeR <MuldeR2@GMX.de>                                */
/* This work has been released under the CC0 1.0 Universal license!           */
/******************************************************************************/

#ifdef _WIN32
#	define _CRT_SECURE_NO_WARNINGS 1
#else
#	define _GNU_SOURCE 1
#endif

/* Internal */
#include "crypt.h"
#include <slunkcrypt.h>
#include "utils.h"
#include "test_data.h"
#include "blake2.h"

/* CRT */
#include <time.h>
#include <inttypes.h>
#include <ctype.h>
#include <assert.h>

// ==========================================================================
// Self-test routines
// ==========================================================================

static int run_testcase(const char* const message, const uint64_t nonce, const uint64_t checksum_message, const uint64_t checksum_expected)
{
	static const char* const TEST_PASSPHRASE = "OrpheanBeh0lderScry!Doubt";

	int status, result = EXIT_FAILURE;
	const size_t length = strlen(message) + 1U;
	slunkcrypt_t ctx = SLUNKCRYPT_NULL;

	char* const text_temp = strdup(message);
	if (!text_temp)
	{
		FPUTS(T("\n\nWhoops: Failed to allocate text buffer!\n\n"), stderr);
		goto clean_up;
	}

	const uint64_t checksum_original = blake2s_compute((uint8_t*)text_temp, length);
	if (checksum_original != checksum_message)
	{
		FPRINTF(stderr, T("\n\nWhoops: Checksum mismatch detected! [expected: 0x%016") T(PRIX64) T(", actual: 0x%016") T(PRIX64) T("]\n\n"), checksum_message, checksum_original);
		goto clean_up;
	}

	ctx = slunkcrypt_alloc(nonce, (const uint8_t*)TEST_PASSPHRASE, strlen(TEST_PASSPHRASE), SLUNKCRYPT_ENCRYPT);
	if (!ctx)
	{
		FPUTS(g_slunkcrypt_abort_flag ? T("\n\nProcess interrupted!\n\n") : T("\n\nWhoops: Failed to initialize encoder!\n\n"), stderr);
		goto clean_up;
	}

	status = slunkcrypt_inplace(ctx, (uint8_t*)text_temp, length);
	if (status != SLUNKCRYPT_SUCCESS)
	{
		FPUTS((status == SLUNKCRYPT_ABORTED) ? T("\n\nProcess interrupted!\n\n") : T("\n\nWhoops: Failed to encrypt the message!\n\n"), stderr);
		goto clean_up;
	}

	if (strncmp(message, text_temp, length) == 0)
	{
		FPUTS(T("\n\nWhoops: Encrypted message equals the original message!\n\n"), stderr);
		goto clean_up;
	}

	const uint64_t checksum_encrypted = blake2s_compute((uint8_t*)text_temp, length);
	if (checksum_encrypted != checksum_expected)
	{
		FPRINTF(stderr, T("\n\nWhoops: Checksum mismatch detected! [expected: 0x%016") T(PRIX64) T(", actual: 0x%016") T(PRIX64) T("]\n\n"), checksum_expected, checksum_encrypted);
		goto clean_up;
	}

	status = slunkcrypt_reset(ctx, nonce, (const uint8_t*)TEST_PASSPHRASE, strlen(TEST_PASSPHRASE), SLUNKCRYPT_DECRYPT);
	if (status != SLUNKCRYPT_SUCCESS)
	{
		FPUTS((status == SLUNKCRYPT_ABORTED) ? T("\n\nProcess interrupted!\n\n") : T("\n\nWhoops: Failed to initialize decoder!\n\n"), stderr);
		goto clean_up;
	}

	status = slunkcrypt_inplace(ctx, (uint8_t*)text_temp, length);
	if (status != SLUNKCRYPT_SUCCESS)
	{
		FPUTS((status == SLUNKCRYPT_ABORTED) ? T("\n\nProcess interrupted!\n\n") : T("\n\nWhoops: Failed to decrypt the message!\n\n"), stderr);
		goto clean_up;
	}

	if (memcmp(message, text_temp, length * sizeof(char)) != 0)
	{
		FPUTS(T("\n\nWhoops: Decrypted message does *not* match the original message!\n\n"), stderr);
		goto clean_up;
	}

	const uint64_t checksum_decrypted = blake2s_compute((uint8_t*)text_temp, length);
	if (checksum_decrypted != checksum_original)
	{
		FPRINTF(stderr, T("\n\nWhoops: Checksum mismatch detected! [expected: 0x%016") T(PRIX64) T(", actual: 0x%016") T(PRIX64) T("]\n\n"), checksum_original, checksum_decrypted);
		goto clean_up;
	}

	result = EXIT_SUCCESS;

clean_up:

	if (ctx)
	{
		slunkcrypt_free(ctx);
	}

	if (text_temp)
	{
		slunkcrypt_bzero(text_temp, strlen(text_temp));
		free(text_temp);
	}

	return result;
}

int run_selftest_routine(void)
{
	static const uint64_t TEST_NONCE[] = { 0x243F6A8885A308D3, 0x13198A2E03707344 };
	const struct
	{
		const char* text;
		uint64_t check_orig, check_encr[2U];
	}
	TEST_STAGE[] =
	{
		{ TEST_DATA_0, TEST_CHCK_ORIG_0, { TEST_CHCK_ENCR_0[0U], TEST_CHCK_ENCR_0[1U] } },
		{ TEST_DATA_1, TEST_CHCK_ORIG_1, { TEST_CHCK_ENCR_1[0U], TEST_CHCK_ENCR_1[1U] } },
		{ TEST_DATA_2, TEST_CHCK_ORIG_2, { TEST_CHCK_ENCR_2[0U], TEST_CHCK_ENCR_2[1U] } },
		{ TEST_DATA_3, TEST_CHCK_ORIG_3, { TEST_CHCK_ENCR_3[0U], TEST_CHCK_ENCR_3[1U] } },
	};

	const size_t total = ARRAY_SIZE(TEST_NONCE) * ARRAY_SIZE(TEST_STAGE);
	FPRINTF(stderr, T("Self-test is in progress, please be patient... stage %u/%u "), 0U, (unsigned)total);
	fflush(stderr);

	for (size_t i = 0U, count = 0U; i < ARRAY_SIZE(TEST_STAGE); ++i)
	{
		for (size_t j = 0U; j < ARRAY_SIZE(TEST_NONCE); ++j)
		{
			FPRINTF(stderr, T("\b\b\b\b%u/%u "), (unsigned)++count, (unsigned)total);
			fflush(stderr);
			if (run_testcase(TEST_STAGE[i].text, TEST_NONCE[j], TEST_STAGE[i].check_orig, TEST_STAGE[i].check_encr[j]) != EXIT_SUCCESS)
			{
				return EXIT_FAILURE;
			}
		}
	}

	FPRINTF(stderr, T("\b\b\b\b%u/%u\n\nCompleted successfully.\n\n"), (unsigned)total, (unsigned)total);
	fflush(stderr);

	return EXIT_SUCCESS;
}
