/******************************************************************************/
/* SlunkCrypt, by LoRd_MuldeR <MuldeR2@GMX.de>                                */
/* This work has been released under the CC0 1.0 Universal license!           */
/******************************************************************************/

#ifdef _WIN32
#	define _CRT_SECURE_NO_WARNINGS 1
#else
#	define _GNU_SOURCE 1
#endif

/* Internal */
#include "pwgen.h"
#include <slunkcrypt.h>
#include "utils.h"

/* CRT */
#include <ctype.h>

// ==========================================================================
// Constants
// ==========================================================================

static const size_t MIX_ROUNDS = 131U;

static const char PASSWD_SYMBOLS[] =
{
	'!', '#', '$', '%', '&', '(', ')', '*', '+', ',', '-', '.', '/', '0', '1',
	'2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@',
	'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O',
	'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', ']', '^', '_',
	'a', 'b', 'c', 'd',	'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o',
	'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'
};

// ==========================================================================
// Read passphrase from file
// ==========================================================================

char *read_passphrase(const CHR *const file_name)
{
	if ((!file_name) || (!file_name[0U]))
	{
		FPUTS(T("Error: The passphrase input file name must not be empty!\n\n"), stderr);
		return NULL;
	}

	const size_t max_len = SLUNKCRYPT_PWDLEN_MAX + 2U;
	char *const buffer = (char*) malloc(max_len * sizeof(char));
	if (!buffer)
	{
		FPUTS(T("Error: Failed to allocate the passphrase buffer!\n\n"), stderr);
		return NULL;
	}

	const int use_stdin = (!file_name) || (!STRICMP(file_name, T("-")));
	FILE *const file_in = use_stdin ? stdin : FOPEN(file_name, T("rb"));
	if (!file_in)
	{
		FPRINTF(stderr, T("Error: Failed to open input file \"%") T(PRISTR) T("\" for reading!\n\n"), file_name);
		free(buffer);
		return NULL;
	}

	do
	{
		if (!fgets(buffer, (int)max_len, file_in))
		{
			buffer[0U] = '\0';
			goto finish;
		}
		size_t length = strlen(buffer);
		while ((length > 0U) && ((buffer[length - 1U] == '\r') || (buffer[length - 1U] == '\n')))
		{
			buffer[--length] = '\0';
		}
	}
	while (!buffer[0U]);

finish:

	if ((!use_stdin) && file_in)
	{
		fclose(file_in);
	}

	return buffer;
}

// ==========================================================================
// Passphrase generator
// ==========================================================================

#define NEXT_RANDOM(OUTVAR,LIMIT) do \
{ \
	uint64_t _rnd; \
	if (slunkcrypt_generate_nonce(&_rnd) != SLUNKCRYPT_SUCCESS) \
	{ \
		FPUTS(T("Error: Failed to generate next random number!\n\n"), stderr); \
		goto clean_up; \
	} \
	OUTVAR = (size_t) (_rnd % (LIMIT)); \
} \
while (0)

int weak_passphrase(const char *str)
{
	unsigned int flags = 0U;
	while ((*str) && (flags != 0xF))
	{
		const int c = *str++;
		if (isalpha(c))
		{
			flags |= isupper(c) ? 0x1 : 0x2;
		}
		else 
		{
			flags |= isdigit(c) ? 0x4 : 0x8;
		}
	}
	return (flags != 0xF);
}

int generate_passphrase(const size_t length)
{
	int result = EXIT_FAILURE;
	char *buffer = NULL, *passwd_symbols = NULL;
	size_t i, j, n, m;
	const size_t passwd_len = BOUND(SLUNKCRYPT_PWDLEN_MIN, length, SLUNKCRYPT_PWDLEN_MAX);

	if (!(buffer = (char*) malloc((passwd_len + 1U) * sizeof(char))))
	{
		FPUTS(T("Error: Failed to allocate memory buffer!\n\n"), stderr);
		goto clean_up;
	}

	if (!(passwd_symbols = (char*) malloc(ARRAY_SIZE(PASSWD_SYMBOLS) * sizeof(char))))
	{
		FPUTS(T("Error: Failed to allocate memory buffer!\n\n"), stderr);
		goto clean_up;
	}

	for (i = 0U; i < ARRAY_SIZE(PASSWD_SYMBOLS); ++i)
	{
		NEXT_RANDOM(j, i + 1U);
		if (j != i)
		{
			passwd_symbols[i] = passwd_symbols[j];
		}
		passwd_symbols[j] = PASSWD_SYMBOLS[i];
	}

	do
	{
		for (n = 0U; n < passwd_len; ++n)
		{
			for (m = 0U; m < MIX_ROUNDS; ++m)
			{
				for (i = ARRAY_SIZE(PASSWD_SYMBOLS) - 1U; i > 0; --i)
				{
					NEXT_RANDOM(j, i + 1U);
					if (j != i)
					{
						const char tmpval = passwd_symbols[i];
						passwd_symbols[i] = passwd_symbols[j];
						passwd_symbols[j] = tmpval;
					}
				}
			}
			buffer[n] = passwd_symbols[0U];
		}
		buffer[passwd_len] = '\0';
	}
	while ((!isalnum((int)buffer[0U])) || (!isalnum((int)buffer[passwd_len - 1U])) || weak_passphrase(buffer));

	FPRINTF(stdout, T("%") T(PRIstr) T("\n\n"), buffer);
	fflush(stdout);
	result = EXIT_SUCCESS;

clean_up:

	if (buffer)
	{
		slunkcrypt_bzero(buffer, passwd_len * sizeof(char));
		free(buffer);
	}

	if (passwd_symbols)
	{
		slunkcrypt_bzero(passwd_symbols, ARRAY_SIZE(PASSWD_SYMBOLS) * sizeof(char));
		free(passwd_symbols);
	}

	return result;
}
