﻿/******************************************************************************/
/* SlunkCrypt, by LoRd_MuldeR <MuldeR2@GMX.de>                                */
/* This work has been released under the CC0 1.0 Universal license!           */
/******************************************************************************/

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using System.Windows.Threading;

using com.muldersoft.slunkcrypt.gui.utils;

namespace com.muldersoft.slunkcrypt.gui.process
{
    internal class SlunkCryptRunner : ProcessRunner
    {
        public enum Mode { Encrypt, Decrypt }

        private const string COMMAND_ENCRYPT = "-e";
        private const string COMMAND_DECRYPT = "-d";

        private static readonly Regex RX_PROGRESS = new Regex(@"(\d+)\.(\d)%", RegexOptions.Compiled);

        private readonly FileStream m_executableFile;

        // =============================================================================
        // Constructor
        // =============================================================================

        public SlunkCryptRunner(Dispatcher dispatcher, ProcessPriorityClass? priorityClass = null) : base(dispatcher, priorityClass)
        {
            m_executableFile = ExecutableHelper.GetExecutableFile();
        }

        // =============================================================================
        // Public methods
        // =============================================================================

        public async Task<int> ExecuteAsync(Mode mode, string inputFile, string outputFile, string password)
        {
            Dictionary<string, string> environmentVariables = new Dictionary<string, string>();
            environmentVariables.Add("SLUNK_PASSPHRASE", password);
            string[] arguments = new string[] { GetCommandString(mode), inputFile, outputFile };
            return await ExecAsnyc(m_executableFile.Name, arguments, environmentVariables);
        }

        public override void Dispose()
        {
            base.Dispose();
            try
            {
                m_executableFile.Dispose();
            }
            catch { }
        }

        // =============================================================================
        // Internal methods
        // =============================================================================

        protected override double ParseProgressString(ref string currentLine, bool stream)
        {
            double temp, result = double.NaN;
            int index = int.MaxValue;
            Match match = RX_PROGRESS.Match(currentLine);
            while (match.Success)
            {
                if (TryParseProgressValue(match, out temp))
                {
                    result = temp;
                }
                index = Math.Min(index, match.Index);
                match = RX_PROGRESS.Match(currentLine, match.Index + match.Length);
            }
            if (index != int.MaxValue)
            {
                currentLine = (index > 0) ? currentLine.Substring(0, index - 1).TrimEnd() : string.Empty;
            }
            return result;
        }

        private static bool TryParseProgressValue(Match match, out double progress)
        {
            uint intPart, fractPart;
            if (uint.TryParse(match.Groups[1].Value, out intPart) && uint.TryParse(match.Groups[2].Value, out fractPart))
            {
                progress = ((intPart * 10) + fractPart) / 1000.0;
                return true;
            }
            progress = double.NaN;
            return false;
        }

        private static string GetCommandString(Mode mode)
        {
            switch(mode)
            {
                case Mode.Encrypt:
                    return COMMAND_ENCRYPT;
                case Mode.Decrypt:
                    return COMMAND_DECRYPT;
                default:
                    throw new ArgumentException("Invalid mode!");
            }
        }
    }
}
