/******************************************************************************/
/* SlunkCrypt, by LoRd_MuldeR <MuldeR2@GMX.de>                                */
/* This work has been released under the CC0 1.0 Universal license!           */
/******************************************************************************/

#ifdef _WIN32
#	define WIN32_LEAN_AND_MEAN 1
#	define _CRT_SECURE_NO_WARNINGS 1
#else
#	define _GNU_SOURCE 1
#endif

/* Internal */
#include "slunkcrypt.h"
#include "compiler.h"

/* CRT */
#include <string.h>
#include <fcntl.h>
#include <limits.h>

/* Utils */
static INLINE size_t MIN_SIZE(const size_t a, const size_t b) { return (a > b) ? b : a; }

// ==========================================================================
// Platform compatibility
// ==========================================================================

#ifdef _WIN32
#	include <Windows.h>
#	include <io.h>
#else
#	include <unistd.h>
#	include <pthread.h>
#endif

/* detect destructor support */
#undef ATTRIB_DESTRUCTOR
#if defined(__GNUC__) || defined(__clang__)
#	define ATTRIB_DESTRUCTOR __attribute__((destructor))
#endif

/* detect getentropy() support */
#undef GETENTROPY
#if (defined(__linux__) && (__linux__ >= 1)) || (defined(__FreeBSD__) && (__FreeBSD__ >= 12)) || (defined(__OpenBSD__) && (__OpenBSD__ >= 1))
#	define GETENTROPY getentropy
#endif

/* detect explicit_bzero() support */
#undef EXPLICIT_BZERO
#if defined(_WIN32) && (_WIN32 >= 1) && defined(SecureZeroMemory)
#	define EXPLICIT_BZERO SecureZeroMemory
#elif (defined(__linux__) && (__linux__ >= 1)) || (defined(__FreeBSD__) && (__FreeBSD__ >= 11)) || (defined(__OpenBSD__) && (__OpenBSD__ >= 1))
#	define EXPLICIT_BZERO explicit_bzero
#endif

// ==========================================================================
// Call once support
// ==========================================================================

#ifdef _WIN32
#	define CALL_ONCE win32_call_once
#	define CALL_ONCE_TYPE volatile LONG
#	define CALL_ONCE_INIT 0L
#else
#	define CALL_ONCE pthread_once
#	define CALL_ONCE_TYPE pthread_once_t
#	define CALL_ONCE_INIT PTHREAD_ONCE_INIT
#endif

#ifdef _WIN32
static void win32_call_once(CALL_ONCE_TYPE *const control, void (*init_routine)(void))
{
	LONG status;
	while ((status = InterlockedCompareExchange(control, -1L, 0L)) != 0L)
	{
		if(status > 0L)
		{
			return; /*already initialized*/
		}
		SwitchToThread();
	}
	init_routine();
	InterlockedExchange(control, 1L);
}
#endif

// ==========================================================================
// Random bytes
// ==========================================================================

#define MAX_COUNT 1048576U

/* Global state */
static CALL_ONCE_TYPE s_random_is_initialized = CALL_ONCE_INIT;
#if defined(_WIN32)
typedef BOOLEAN(WINAPI *ptr_genrandom_t)(void *buffer, ULONG length);
static HMODULE s_advapi32 = NULL;
static ptr_genrandom_t s_genrandom = NULL;
#else
static const char *const DEV_RANDOM[] = { "/dev/urandom", "/dev/arandom", "/dev/random", NULL };
static int s_random_fd = -1;
#endif

/* De-initialize CSRNG */
static void exit_random_source(void)
{
#ifdef _WIN32
	if (s_genrandom)
	{
		s_genrandom = NULL;
	}
	if (s_advapi32)
	{
		FreeLibrary(s_advapi32);
		s_advapi32 = NULL;
	}
#else
	if (s_random_fd >= 0)
	{
		close(s_random_fd);
		s_random_fd = -1;
	}
#endif
}

/* Initialize CSRNG */
static void init_random_source(void)
{
#ifdef _WIN32
	if ((s_advapi32 = LoadLibraryW(L"advapi32.dll")))
	{
		s_genrandom = (ptr_genrandom_t) GetProcAddress(s_advapi32, "SystemFunction036");
	}
#else
#if defined(GETENTROPY)
	uint8_t temp;
	if (GETENTROPY(&temp, sizeof(uint8_t)) >= 0)
	{
		goto init_completed;
	}
#endif
	for (size_t i = 0U; DEV_RANDOM[i]; ++i)
	{
		if ((s_random_fd = open(DEV_RANDOM[i], O_RDONLY)) >= 0)
		{
			goto init_completed;
		}
	}
init_completed: ;
#endif
#if !defined(ATTRIB_DESTRUCTOR)
	atexit(exit_random_source);
#endif
}

/* Generate random bytes */
size_t slunkcrypt_random_bytes(uint8_t* const buffer, const size_t length)
{
	size_t offset;
	CALL_ONCE(&s_random_is_initialized, init_random_source);
#ifdef _WIN32
	if (s_genrandom)
	{
		ULONG count;
		for (offset = 0U; offset < length; offset += count)
		{
			count = (ULONG) MIN_SIZE(length - offset, MAX_COUNT);
			if (!s_genrandom(buffer + offset, count))
			{
				break; /*failed*/
			}
		}
		return offset;
	}
	return 0U;
#else
	if (s_random_fd >= 0)
	{
		ssize_t count;
		for (offset = 0; offset < length; offset += (size_t)count)
		{
			if (!((count = read(s_random_fd, buffer + offset, MIN_SIZE(length - offset, MAX_COUNT))) > 0))
			{
				break; /*failed*/
			}
		}
		return offset;
	}
#if defined(GETENTROPY)
	else
	{
		size_t count;
		for (offset = 0U; offset < length; offset += count)
		{
			count = MIN_SIZE(length - offset, 256U); /*the maximum permitted value is 256*/
			if (GETENTROPY(buffer + offset, count) < 0)
			{
				break; /*failed*/
			}
		}
		return offset;
	}
#else
	return 0U;
#endif
#endif
}

// ==========================================================================
// Zero memory
// ==========================================================================

void slunkcrypt_bzero(void* const buffer, const size_t length)
{
	if ((buffer) && (length > 0U))
	{
#if defined(EXPLICIT_BZERO)
		EXPLICIT_BZERO(buffer, length);
#else
		volatile uint8_t* ptr = (volatile uint8_t*) buffer;
		for (size_t i = 0U; i < length; ++i)
		{
			ptr[i] = 0U;
		}
#endif
	}
}

// ==========================================================================
// Destructor
// ==========================================================================

#if defined(ATTRIB_DESTRUCTOR)
ATTRIB_DESTRUCTOR void slunkcrypt_destructor()
{
	exit_random_source();
}
#endif
