/* $Id: BookmarkView.java 1058 2016-01-17 17:09:20Z shayashi $ */
package smart_gs.drawing_tool.view;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.awt.geom.Point2D.Double;

import javax.swing.ImageIcon;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import smart_gs.GSConstants;

import smart_gs.logical.Preference;
import smart_gs.logical.URICreator;
import smart_gs.swingui.WorkspaceWindow;


public class BookmarkView extends View {
	public static final double HUGE = 1.5;
	public static final double LARGE = 1.25;
	public static final double NORMAL = 1;
	public static final double SMALL = 0.75;
	public static final double TINY = 0.5;

	public static final String PATH_TO_BLUE = "bookmark_blue.png";
	public static final String PATH_TO_GREEN = "bookmark_green.png";
	public static final String PATH_TO_RED = "bookmark_red.png";
	public static final String PATH_TO_GRAY = "bookmark_gray.png";
	public static final String PATH_TO_PURPLE = "bookmark_purple.png";
	public static final String DEFAULT_PATH = "bookmark_red.png";
	
	public static double DEFAULT_WIDTH = 45;
	public static double DEFAULT_HEIGHT = 42;
	
	private Rectangle2D rectangle;
	private double size;
	private Point origin;
	private double imageScale;
	private String colorPath = DEFAULT_PATH;

	
	public BookmarkView(Point origin, String title, double size) {
		this.origin = origin;
		this.title = title;
		this.setSize(size);
		this.colorPath = Preference.getInstance().getDefaultBookmarkPath();
	}

	@Override
	public boolean contains(Point2D point) {
		double w = (this.rectangle.getWidth()/2) / imageScale;
		double h = (this.rectangle.getHeight()) / imageScale;
		double x = this.origin.getX() - w/2;
		double y = this.origin.getY() - h;
		Rectangle2D rect = new Rectangle.Double(x,y,w,h);
		return rect.contains(point);
	}

	@Override
	public Element createXMLElement(Document document) {
		Element element = document.createElement("view");
		element.setAttribute("title",this.title);
		element.setAttribute("type", URICreator.BOOKMARK);
		element.setAttribute("size",((int)(this.size*100))+"");
		element.setAttribute("colorPath", this.colorPath);
		element.setAttribute("originX",(int)this.origin.getX()+"");
		element.setAttribute("originY",(int)this.origin.getY()+"");
		return element;
	}
	public static View restore(Element elem) {
		double size = new Integer(elem.getAttribute("size")) / 100.0;
		String title = elem.getAttribute("title");
		String colorPath = DEFAULT_PATH;
		if (elem.getAttribute("colorPath").length() > 0) {
			colorPath = elem.getAttribute("colorPath");
		}
		int originX = new Integer(elem.getAttribute("originX"));
		int originY = new Integer(elem.getAttribute("originY"));
		BookmarkView view = new BookmarkView(new Point(originX, originY), title, size);
		view.size = size;
		view.colorPath = colorPath;
		return view;
	}

	@Override
	public void drawLinkedShape(Graphics2D g, Image image) {
		this.drawShape(g);
		this.drawLinkIcon(g, image);
	}
	
	@Override
	public void drawTemporaryShape(Graphics2D g) {
		this.draw(g);
		this.drawTemporaryIcon(g);		
	}
	
	@Override
	public void drawInBasketShape(Graphics2D g) {
		this.draw(g);
		this.drawInBasketIcon(g);	
	}

	@Override
	protected void drawInBasketIcon(Graphics2D g) {
		Image image = new ImageIcon(GSConstants.ICON_PATH + "basketWithItem.png").getImage();
		Rectangle rect = this.getRectangle();
		int w = image.getWidth(null) / 4 + 3;
		int h = image.getHeight(null) / 4 + 3;
		int x = (int)(rect.getX() + rect.getWidth()/2.5 - 2);
		int y = (int)(rect.getY()) - h/2 + 2;
		g.drawImage(image, x, y, w, h, null);
	}

	@Override
	public void drawRWShape(Graphics2D g) {
		this.draw(g);
		this.drawRWIcon(g);	
	}
	@Override
	protected void drawRWIcon(Graphics2D g) {
		Image image = new ImageIcon(GSConstants.ICON_PATH + "reasoning_web.png").getImage();
		Rectangle rect = this.getRectangle();
		int w = image.getWidth(null) / 4 + 3;
		int h = image.getHeight(null) / 4 + 3;
		int x = (int)(rect.getX() + rect.getWidth()/2.5 - 2);
		int y = (int)(rect.getY()) - h/2 + 2;
		g.drawImage(image, x, y, w, h, null);
	}
	
	@Override
	public void drawTemporaryAndInBasketShape(Graphics2D g) {
		this.draw(g);
		Image tempIcon = this.getAuxIcon(tempIconName);
		Image basketIcon = this.getAuxIcon(inBasketIconName);
		int w = tempIcon.getWidth(null) / 4 + 3;
		int h = tempIcon.getHeight(null) / 4 + 3;
		Point2D auxIconPoint1 = this.getFirstAuxIconPoint(w, h);
		int x = (int)auxIconPoint1.getX();
		int y = (int)auxIconPoint1.getY();
		int w2 = basketIcon.getWidth(null) / 3;
		int h2 = basketIcon.getHeight(null) / 3;
		Point2D auxIconPoint2 = this.getSecondAuxIconPoint(x, y, w, h);
		int x2 = (int)auxIconPoint2.getX();
		int y2 = (int)auxIconPoint2.getY();
		g.drawImage(tempIcon, x, y, w, h, null);
		g.drawImage(basketIcon, x2, y2, w2, h2, null);
	}
	@Override
	public Point2D getFirstAuxIconPoint(int w, int h) {
		Rectangle rect = this.getRectangle();
		int x = (int)(rect.getX() + rect.getWidth()/2.5 - 2);
		int y = (int)(rect.getY()) - h/2 + 2;
		return new Point2D.Double(x, y);
	}

	
	@Override
	protected void drawShape(Graphics2D g) {
		if (this.isSelected) {
			this.drawSelectedIcon(g);
		}
		String imageFileName = this.colorPath;
		if (imageFileName == null) {
			imageFileName = DEFAULT_PATH;
		}
		
		ImageIcon imageIcon = new ImageIcon(
				GSConstants.ICON_PATH + imageFileName);
		Image image = imageIcon.getImage();
		int x = (int)this.rectangle.getX();
		int y = (int)this.rectangle.getY();
		int width = (int)this.rectangle.getWidth();
		int height = (int)this.rectangle.getHeight();
		g.drawImage(image, x, y, width, height, null);
		if (this.isProtected) {
			int xx = (int)this.rectangle.getX();
			int yy = (int)this.rectangle.getY();
			int ww = (int)(this.rectangle.getWidth() / 2);
			int hh = (int)(this.rectangle.getHeight());
			Rectangle rect = new Rectangle(xx, yy, ww, hh);
			g.setStroke(new BasicStroke(1));
			g.setColor(Color.BLACK);
			g.draw(rect);
		}
	}

	@Override
	public BookmarkView enlargedView(double ratio, double gapX, double gapY) {
		this.imageScale = ratio;
		int x = (int)(this.origin.getX() * ratio + gapX);
		int y = (int)(this.origin.getY() * ratio + gapY);
		Point newOrigin = new Point(x,y);
		BookmarkView view = new BookmarkView(newOrigin, title, this.size);
		view.setIsSelected(this.isSelected);
		view.isProtected = this.isProtected;
		view.setColorPath(this.colorPath);
		return view;
	}

	public void setColorPath(String colorPath) {
		this.colorPath = colorPath;
	}
	
	public String getColorPath() {
		return this.colorPath;
	}

	@Override
	public Point getCenterPoint() {
		return new Point((int)rectangle.getCenterX(), (int)rectangle.getCenterY());
	}

	@Override
	public Rectangle2D getRectangle2D() {
		return this.rectangle;
	}

	@Override
	public int getType() {
		return View.BOOKMARK;
	}

	@Override
	public String getTypeString() {
		return URICreator.BOOKMARK;
	}

	@Override
	public void moveBy(Point2D d) {
		int x = (int)(this.origin.getX()+d.getX());
		int y = (int)(this.origin.getY()+d.getY());
		this.origin = new Point(x,y);
		this.setSize(this.size);
	}

	@Override
	public String getTypeStringForDisplay() {
		return "Bookmark";
	}
	public void setSize(double size) {
		this.size = size;
		double originX = this.origin.getX();
		double originY = this.origin.getY();
		double w = DEFAULT_WIDTH * size;
		double h = DEFAULT_HEIGHT * size;
		double x = originX - (w/4);
		double y = originY - h;
		this.rectangle = new Rectangle2D.Double(x, y, w, h);
	}

	public double getSize() {
		return this.size;
	}

	public static boolean properSize(double bookmarkSize) {
		return bookmarkSize == HUGE 
			|| bookmarkSize == LARGE
			|| bookmarkSize == NORMAL
			|| bookmarkSize == SMALL
			|| bookmarkSize == TINY;
	}
	@Override
	public void drawLinkIcon(Graphics2D g, Image image) {
		Rectangle rect = this.getRectangle();
		int w = image.getWidth(null) / 3;
		int h = image.getHeight(null) / 3;
		int x = (int)(rect.getX() + rect.getWidth()/2.5);
		int y = (int)(rect.getY()) - h/2;
		g.drawImage(image, x, y, w, h, null);
	}

	@Override
	public void drawTemporaryIcon(Graphics2D g) {
		Image image = new ImageIcon(GSConstants.ICON_PATH + "Temporary.png").getImage();
		
		Rectangle rect = this.getRectangle();
		int w = image.getWidth(null) / 4 + 3;
		int h = image.getHeight(null) / 4 + 3;
		int x = (int)(rect.getX() + rect.getWidth()/2.5 - 2);
		int y = (int)(rect.getY()) - h/2 + 2;
		g.drawImage(image, x, y, w, h, null);
	}


	@Override
	public void drawSelectedIcon(Graphics2D g) {
		ImageIcon selected = new ImageIcon(GSConstants.ICON_PATH + "selected_mark.png");
		Image image = selected.getImage();
		Rectangle rect = this.getRectangle();
		int w = (int)(image.getWidth(null) / 2.5);
		int h = (int)(image.getHeight(null) / 2.5);
		int x = (int)(rect.getX()) - w;
		int y = (int)(rect.getY()) - h;
		g.drawImage(image, x, y, w, h, null);		
	}

	@Override
	public boolean containsInCloseUpIcon(Point2D point) {
		// TODO Auto-generated method stub
		return false;
	}

	@Override
	protected Rectangle closeUpImageRectangle() {
		// TODO Auto-generated method stub
		return null;
	}

	
}