#include "create_symboliclink.h"
#include "struct_CPDD.h"
#include "struct_CPInfo.h"
#include "struct_OPArg.h"
#include "struct_SDir.h"
#include "struct_VList.h"
#include "print_error.h"
#include "read_write.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <alloca.h>
#include <glib.h>
#include <utime.h>

static enum I_OverWrite_Type
{
	INFO,
	THIS_DIR_YES,
	THIS_DIR_NO,
} iowt;

#define THIS_DIR_FREE \
{\
	if(this_dir != NULL)\
	{\
		free(this_dir);\
	}\
}

#define DO_NOT_OVERWRITE_FLAG \
{\
	cpinfo->write = false;\
	cpinfo->skip = true;\
}

#define FILE_WRITE_OR_CREATE_LINK_OVERWRITE \
{\
	if(cpinfo->from_type == REGULAR)\
	{\
		_Bool check = read_write(oparg, cpinfo, sdir, cpdd);\
		(check == true) ? (cpinfo->write = true) : (cpinfo->write = false);\
	}\
	else \
	{\
		errno = 0;\
		if(unlink(cpinfo->to) == -1)\
		{\
			cpinfo->write = false;\
			print_error("unlink", __FILE__, __LINE__, cpdd);\
			fprintf(stderr, "ファイルを削除出来ませんでした (%s) \n", cpinfo->to);\
			fprintf(stderr, "シンボリックリンクを上書きできません\n");\
		}\
		else \
		{\
			_Bool check = create_symboliclink(cpinfo, cpdd);\
			(check == true) ? (cpinfo->write = true) : (cpinfo->write = false);\
		}\
	}\
}

// 関数プロトタイプ
void process_file_list(const OPArg *oparg, const VList *file_list, SDir *sdir, CPDD *cpdd);
static inline void ch_time_and_mod_dir(const CPInfo *cpinfo, CPDD *cpdd) __attribute__((always_inline));
static inline void clean(const char stdin_buf[], const int INPUT_LEN) __attribute__((always_inline));
static inline void fgets_is_null(void) __attribute__((always_inline));

/*******************************************************************************
*******************************************************************************/
void process_file_list(const OPArg *oparg, const VList *file_list, SDir *sdir, CPDD *cpdd)
{
	iowt = INFO;
	char *this_dir = NULL;

	struct DirList
	{
		CPInfo *data;
		struct DirList *next;
		// tailは先頭の要素のみ使用、他はNULL
		struct DirList *tail;
	};
	struct DirList *dir_list = NULL;

	// 変更される可能性があるオプションはローカル変数に入れておく
	OPFlag I = oparg->I;
	OPFlag W = oparg->W;
	int wmode = oparg->write_mode;

	/*
	 * キャストしないと警告が出る
	 * warning: initialization discards qualifiers from pointer target type
	*/
	VList *tmp = (VList *)file_list;

	for(;;)
	{
		CPInfo *cpinfo = (CPInfo *)tmp->data;

		gboolean fcheck = g_file_test(cpinfo->to, G_FILE_TEST_EXISTS);

		if(cpinfo->from_type != DIRECTORY)
		{
			// REGULAR or SYMBOLICLINK

			// TRUE・FALSE　→　glib
			// true・false　→　stdbool
			if(fcheck == FALSE)
			{
				if(cpinfo->from_type == REGULAR)
				{
					_Bool check = read_write(oparg, cpinfo, sdir, cpdd);
					(check == true) ? (cpinfo->write = true) : (cpinfo->write = false);
				}
				else
				{
					_Bool check = create_symboliclink(cpinfo, cpdd);
					(check == true) ? (cpinfo->write = true) : (cpinfo->write = false);
				}
			}
			else
			{
				cpinfo->file_test = true;

				if(W == NOT)
				{
					// 上書きモード

					FILE_WRITE_OR_CREATE_LINK_OVERWRITE
				}
				else if(W == WRITE_MODE)
				{
					if(wmode == 0)
					{
						// -w 0（非上書き）モード

						DO_NOT_OVERWRITE_FLAG
					}
					else
					{
						// 上書きモード

						FILE_WRITE_OR_CREATE_LINK_OVERWRITE
					}
				}
				else if(I == INTERACTIVE)
				{
					// -i（上書き確認）モード

					// ここの処理はgoto使わずに関数にまとめた方が分かりやすいかもしれない
INPUT:
					// gotoのラベルの直後で変数の宣言をすると、
					// a label can only be part of a statement and a declaration is not a statement
					// というエラーが出る。

					if(iowt == INFO)
					{
						char stdin_buf[NAME_MAX];

						printf("%sは既に存在します。\n", cpinfo->to);
						puts("上書きしますか? (y: Yes  a: 全てYes  d: このフォルダのみYes  n: No  o: このフォルダのみNo  z: 全てNo)");
						printf(" : ");

						if(fgets(stdin_buf, PATH_MAX, stdin) == NULL)
						{
							fgets_is_null();
						}
						clean(stdin_buf, NAME_MAX);

						if(
							stdin_buf[0] != 'y' &&
							stdin_buf[0] != 'a' &&
							stdin_buf[0] != 'd' &&
							stdin_buf[0] != 'n' &&
							stdin_buf[0] != 'o' &&
							stdin_buf[0] != 'z'
						)
						{
							goto INPUT;
						}

						switch(stdin_buf[0])
						{
							case 'y':	// 上書き
								FILE_WRITE_OR_CREATE_LINK_OVERWRITE
								break;

							case 'a':	// 全て上書き
								I = NOT;	// 上書き確認モード無効
								FILE_WRITE_OR_CREATE_LINK_OVERWRITE
								break;

							case 'd':	// このフォルダでのみ上書き
								iowt = THIS_DIR_YES;
								THIS_DIR_FREE
								this_dir = g_path_get_dirname(cpinfo->from);
								FILE_WRITE_OR_CREATE_LINK_OVERWRITE
								break;

							case 'n':	// 上書きしない
								DO_NOT_OVERWRITE_FLAG
								break;

							case 'o':	// このフォルダでは上書きしない
								DO_NOT_OVERWRITE_FLAG
								iowt = THIS_DIR_NO;
								THIS_DIR_FREE
								this_dir = g_path_get_dirname(cpinfo->from);
								break;

							case 'z':	// 全て上書きしない
								DO_NOT_OVERWRITE_FLAG
								W = WRITE_MODE;	// 非上書きモード
								wmode = 0;
								I = NOT;	// 上書き確認モード無効
								break;
						}
					}
					else if(iowt == THIS_DIR_YES)
					{
						if(strstr(cpinfo->from, this_dir) != NULL)
						{
							FILE_WRITE_OR_CREATE_LINK_OVERWRITE
						}
						else
						{
							// 再び上書き確認モードに
							iowt = INFO;
							goto INPUT;
						}
					}
					else
					{
						// THIS_DIR_NO

						if(strstr(cpinfo->from, this_dir) != NULL)
						{
							DO_NOT_OVERWRITE_FLAG
						}
						else
						{
							// 再び上書き確認モードに
							iowt = INFO;
							goto INPUT;
						}
					}
				}
				else
				{
					// ここは実行されない

					print_error("process_file", __FILE__, __LINE__, cpdd);
					fprintf(stderr, "オプションエラー\n");
				}
			}
		}
		else
		{
			// cpinfo->from_type == DIRECTORY

			errno = 0;
			if(g_file_test(cpinfo->to, G_FILE_TEST_EXISTS) == TRUE)
			{
				// フォルダの作成は行わない
				DO_NOT_OVERWRITE_FLAG
			}
			else if(mkdir(cpinfo->to, 0755) == -1)
			{
				cpinfo->write = false;

				if((errno != EEXIST) && (errno != 0))
				{
					print_error("mkdir", __FILE__, __LINE__, cpdd);
					fprintf(stderr, "フォルダの作成に失敗しました (%s) \n", cpinfo->to);
				}
			}
			else
			{
				cpinfo->write = true;
				cpdd->mk_dir_count++;

				// alloca使いたかったんで関数にしなかった
				if(dir_list != NULL)
				{
					struct DirList *now = alloca(sizeof(struct DirList));
					struct DirList *before_last = dir_list->tail;
					now->data = cpinfo;
					now->next = NULL;
					now->tail = NULL;

					before_last->next = now;

					dir_list->tail = now;
				}
				// リストに最初の要素を追加する
				else
				{
					dir_list = alloca(sizeof(struct DirList));
					dir_list->data = cpinfo;
					dir_list->next = NULL;
					dir_list->tail = dir_list;
				}
			}
		}

		if(tmp->next == NULL)
		{
			break;
		}
		tmp = tmp->next;
	}

	if(dir_list != NULL)
	{
		for(;;)
		{
			CPInfo *cpi = dir_list->data;

			ch_time_and_mod_dir(cpi, cpdd);

			if(dir_list->next == NULL)
			{
				break;
			}
			dir_list = dir_list->next;
		}
	}

	THIS_DIR_FREE
}

/*******************************************************************************
*******************************************************************************/
static inline void ch_time_and_mod_dir(const CPInfo *cpinfo, CPDD *cpdd)
{
	struct stat stat_buf;
	struct utimbuf times;
	errno = 0;

	if(stat(cpinfo->from, &stat_buf) == -1)
	{
		print_error("stat", __FILE__, __LINE__, cpdd);
		fprintf(stderr, "フォルダ情報取得エラーです (%s) \n", cpinfo->from);
		return;
	}

	if(chmod(cpinfo->to, stat_buf.st_mode) == -1)
	{
		print_error("chmod", __FILE__, __LINE__, cpdd);
		fprintf(stderr, "フォルダ情報変更エラーです (%s) \n", cpinfo->to);
	}

	if(chown(cpinfo->to, stat_buf.st_uid, stat_buf.st_gid) == -1)
	{
		print_error("chown", __FILE__, __LINE__, cpdd);
		fprintf(stderr, "フォルダ情報変更エラーです (%s) \n", cpinfo->to);
	}

	times.actime  = stat_buf.st_atime;
	times.modtime = stat_buf.st_mtime;

	if(utime(cpinfo->to, &times) == -1)
	{
		print_error("utime", __FILE__, __LINE__, cpdd);
		fprintf(stderr, "フォルダ情報変更エラーです (%s) \n", cpinfo->to);
	}
}

/*******************************************************************************
 * 入力バッファのクリア
*******************************************************************************/
static inline void clean(const char stdin_buf[], const int INPUT_LEN)
{
	// バッファチェック
	_Bool flag = false;
	for(int i = 0; i < INPUT_LEN; i++)
	{
		if(stdin_buf[i] == '\n')
		{
			flag = true;
			break;
		}
	}
	// 入力バッファの掃除
	if(flag == false)
	{
		while(getchar() != '\n');
	}
}

/*******************************************************************************
 * fgetsの戻り値がNULLだった場合
*******************************************************************************/
static inline void fgets_is_null(void)
{
	fprintf(stderr, "エラーが発生しました、fgetsの戻り値がNULLです\n");
	exit(EXIT_FAILURE);
}
