<?php

function fwrite_safe($name, $text) {
  $tmp = tempnam(TMP_DIR, "TMP");
  $tmpfp = fopen($tmp, "w") or die("temporary file open error: $tmp");
  fwrite($tmpfp, $text, strlen($text));
  fflush($tmpfp);
  fclose($tmpfp);
  unlink($name) && rename($tmp, $name);
  chmod($name, 0666);
}

class WikiPageStorage {

  //----------------------------------
  // PUBLIC FUNCTIONS
  //----------------------------------

  function get($name) {
    if (file_exists(WikiPageStorage::filePath($name))) {
      $text = file(WikiPageStorage::filePath($name));
      $text = str_replace("\r\n", "\n", $text);
      $text = str_replace("\r", "\n", $text);
      $text = join("", $text);
      return $text;
    }
    else {
      return "";
    }
  }

  function put($name, $text, $memo, $isSage) {
    if ($isSage)
      WikiPageStorage::putSage($name, $text, $memo);
    else
      WikiPageStorage::putAge($name, $text, $memo);
  }

  function pageExists($name) {
    return file_exists(WikiPageStorage::filePath($name));
  }

  function pageMtime($name) {
    clearstatcache();
    return filemtime(WikiPageStorage::filePath($name));
  }

  function recentChanges($num) {
    $property = Property::instance();
    $files = WikiPageStorage::allFiles();
    usort($files, array("WikiPageStorage","cmpRecent"));
    $result = array();
    // if arg $num is zero, retrieve all pages
    if ($num == 0) $num = count($files);
    foreach (array_slice($files, 0, $num) as $file) {
      $time = date("Y-m-d H:i:s", filemtime(DATA_DIR . basename($file)));
      $wikiname = WikiPageStorage::pageName($file);
      $user = $property->get($wikiname, 'last_modified_user');
      $result[] = array($time, $wikiname, $user);
    }
    return $result;
  }

  function pageName($fileName) {
    return urldecode(basename($fileName, ".txt"));
  }

  function existsPages() {
    $pfiles = WikiPageStorage::allFiles();
    $pages = array_map("basename", $pfiles, array_pad(array(), count($pfiles), ".txt"));

    $cfiles = WikiCommentStorage::allFiles();
    $comments = array_map("basename", $cfiles, array_pad(array(), count($cfiles), ".cmt"));

    $files = array_unique(array_merge($pages, $comments));
    return array_map("urldecode", $files); 
  }

  function editHistory($name) {
    $path = WikiPageStorage::historyFileName($name);
    $result = array();
    if (!file_exists($path)) return $result;
    $lines = file($path);
    rsort($lines);
    foreach ($lines as $line) {
      rtrim($line);
      if (preg_match("/\t/", $line)) {
				list($timestamp, $user, $memo) = split("\t", $line);
      }
      else {
        // mature history data occured.
        break;
      }
      $date = preg_replace('/(\d{4})(\d{2})(\d{2})(\d{2})(\d{2})(\d{2})/',
         '\1-\2-\3 \4:\5:\6', $timestamp);
      $result[] = array($date, $user, rtrim($memo), $timestamp);
    }
    return $result;
  }

  function checkSlash($name) {
    basename($name) != $name and die("invalid name:".htmlspecialchars($name));
  }

  function allFiles() {
    $handle = opendir(DATA_DIR) or die("data dir open error");
    $files = array();
    while (false !== ($file = readdir($handle))) {
      if (preg_match("/\.txt$/", $file)) {
        $files[] = $file;
      }
    }
    return $files;
  }

  function metaDataDir($name) {
    $encoded = urlencode($name);
    $basename = basename($encoded, ".txt");
    $dir = METADATA_DIR . $basename;
    if (!file_exists($dir)) {
      mkdir($dir, 0777);
    }
    return $dir;
  }

  function filePath($name) {
    $encoded = urlencode($name);
    $basename = basename($encoded, ".txt");
    return DATA_DIR . $basename . ".txt";
  }

  function getBackup($name, $timestamp) {
    $path = WikiPageStorage::backupFilePath($name, $timestamp);
    $fp = fopen($path, "r") or die("backup file open error: $path");
    $text = fread($fp, filesize($path));
    fclose($fp);
    return $text;
  }

  function backupFilePath($name, $timestamp) {
    $filename = preg_replace("/(\d\d\d\d\d\d\d\d)(\d\d\d\d\d\d)/", '\1-\2.txt', $timestamp);
    $path = WikiPageStorage::metaDataDir($name) . "/" . $filename;
    return $path;
  }

  function search($str) {
    $pat = "/" . preg_quote($str, "/") . "/";
    $pfiles = WikiPageStorage::allFiles();
    $result = array();
    foreach ($pfiles as $f) {
      $path = DATA_DIR . $f;
      $content = file($path);
      $hit = count(preg_grep($pat, $content));
      if ($hit > 0)
        $result[] = WikiPageStorage::pageName($f);
    }
    return $result;
  }

  //----------------------------------
  // PRIVATE FUNCTIONS
  //----------------------------------

  function putAge($name, $text, $memo) {
    $property = Property::instance();
    if (WikiPageStorage::pageExists($name)) {
      WikiPageStorage::setPageDiff($name, $text, WikiPageStorage::get($name));
      WikiPageStorage::saveHistory($name);
      WikiPageStorage::backupOld($name);
    }
    else {
      WikiPageStorage::setPageCreator($name);
      WikiPageStorage::setPageDiff($name, $text, "");
    }
    WikiPageStorage::putFile($name, $text);
    WikiPageStorage::setPageEditMemo($name, $memo);
    WikiPageStorage::setPageModifiedUser($name, $memo);
    Rss::writeRecentChanges();
  }

  function putSage($name, $text, $memo) {
    $property = Property::instance();
    if (WikiPageStorage::pageExists($name)) {
      WikiPageStorage::setPageDiff($name, $text, WikiPageStorage::getLastBackup($name));
      $lastTimestamp = WikiPageStorage::pageMtime($name);
      WikiPageStorage::putFile($name, $text);
      touch(WikiPageStorage::filePath($name), $lastTimestamp);
    }
    else {
      WikiPageStorage::setPageCreator($name);
      WikiPageStorage::setPageDiff($name, $text, "");
      WikiPageStorage::putFile($name, $text);
    }
    WikiPageStorage::setPageEditMemo($name, $memo);
    WikiPageStorage::setPageModifiedUser($name, $memo);
    Rss::writeRecentChanges();
  }

  function getLastBackup($name) {
    $lastHistory = WikiPageStorage::lastHistory($name);
    $timestamp = $lastHistory[3];
    if ($timestamp) {
      $text = WikiPageStorage::getBackup($name, $timestamp);
      return $text;
    }
    else {
      return "";
    }
  }

  function setPageDiff($name, $newText, $oldText) {
    $diff = WikiPageStorage::computeDifference($newText, $oldText);
    $property = Property::instance();
    $property->set($name, 'diff', $diff);
  }

  function setPageEditMemo($name, $memo) {
    $property = Property::instance();
    $property->set($name, 'edit_memo', $memo);
  }

  function setPageModifiedUser($name) {
    $user = WebSession::loginUserName();
    $property = Property::instance();
    $property->set($name, 'last_modified_user', $user);
  }

  function setPageCreator($name) {
    $user = WebSession::loginUserName();
    $property = Property::instance();
    $property->set($name, 'creator', $user);
  }

  function saveHistory($name) {
    $property = Property::instance();
    $memo = $property->get($name, 'edit_memo');
    $date = date("YmdHis", WikiPageStorage::pageMtime($name));
    $user = $property->get($name, 'last_modified_user');

    $path = WikiPageStorage::historyFileName($name);
    $fp = fopen($path, "a") or die("file open error: edit history file ($path)");
    fwrite($fp, sprintf("%s\t%s\t%s\n", $date, $user, $memo));
    fflush($fp);
    fclose($fp);
  }

  function computeDifference($new, $old) {
    $newArray = explode("\n", Formatter::normalizeLineBreak($new));
    $oldArray = explode("\n", Formatter::normalizeLineBreak($old));
    $diff = array_diff($newArray, $oldArray);
    return $diff;
  }
  
  function putFile($name, $text) {
    $text = Formatter::normalizeLineBreak($text);
    $file = WikiPageStorage::filePath($name);
    $fp = fopen($file,'w') or die("file open error");
    fputs($fp,$text);
    fflush($fp);
    fclose($fp);
    clearstatcache();
  }

  function backupOld($name) {
    $src = WikiPageStorage::filePath($name);
    $dest = WikiPageStorage::newBackupFilePath($name);
    rename($src, $dest);
  }

  function newBackupFilePath($name) {
    $dir = WikiPageStorage::metaDataDir($name);
    return $dir."/".date("Ymd-His", WikiPageStorage::pageMtime($name)).".txt";
  }

  function cmpRecent($a, $b) {
    $at = filemtime(DATA_DIR . basename($a));
    $bt = filemtime(DATA_DIR . basename($b));
    if ($at == $bt) return 0;
    return $at > $bt ? -1 : 1;
  }

  function lastHistory($name) {
    $histories = WikiPageStorage::editHistory($name);
    $i = count($histories);
    if ($i > 1) {
      return $histories[0];
    }
    else {
      return array(null, null, null, null);
    }
  }

  function historyFileName($name) {
    WikiPageStorage::checkSlash($name);
    return WikiPageStorage::metaDataDir($name) . "/history.txt";
  }
}

class WikiCommentStorage {

  function writeComments($page, $comments) {
    $path = WikiCommentStorage::filePath($page);
    $arr = array();
    foreach ($comments as $c) {
      $arr[] = serialize($c);
    }
    $text = join("\n", $arr)."\n";
    fwrite_safe($path, $text);
  }

  function getComments($page) {
    $result = array();
    $path = WikiCommentStorage::filePath($page);
    if (!file_exists($path)) return $result;
    $tmp = file($path);
    foreach ($tmp as $i) {
      $obj = unserialize($i);
      if (is_object($obj)) $result[] = $obj;
    }
    return $result;
  }

  function appendComment($comment, $page) {
    $path = WikiCommentStorage::filePath($page);
    $fp = fopen($path, "a") or die("comment file open error");
    $comment->escape();
    fwrite($fp, serialize($comment)."\n");
    fflush($fp);
    fclose($fp);
  }

  function filePath($name) {
    $encoded = urlencode($name);
    $basename = basename($encoded, ".cmt");
    return DATA_DIR . $basename . ".cmt";
  }

  function recentComments($num) {
    $property = Property::instance();
    $files = WikiCommentStorage::allFiles();
    usort($files, array("WikiPageStorage","cmpRecent"));
    $result = array();
    // if arg $num is zero, retrieve all pages
    if ($num == 0) $num = count($files);
    foreach (array_slice($files, 0, $num) as $file) {
      $time = date("Y-m-d H:i:s", filemtime(DATA_DIR . basename($file)));
      $wikiname = WikiCommentStorage::pageName($file);
      $user = $property->get($wikiname, 'last_commented_user');
      $result[] = array($time, $wikiname, $user);
    }
    return $result;
  }

  function pageName($fileName) {
    $basename = basename($fileName, ".cmt");
    $decoded = urldecode($basename);
    return $decoded;
  }

  function exists($name) {
    return file_exists(WikiCommentStorage::filePath($name));
  }

  function allFiles() {
    $handle = opendir(DATA_DIR) or die("data dir open error");
    $files = array();
    while (false !== ($file = readdir($handle))) {
      if (preg_match("/\.cmt$/", $file)) {
        $files[] = $file;
      }
    }
    return $files;
  }

  function search($str) {
    $pat = "/" . preg_quote($str, "/") . "/";
    $files = WikiCommentStorage::allFiles();
    $result = array();
    foreach ($files as $f) {
      $path = DATA_DIR . $f;
      $content = file($path);
      $hit = count(preg_grep($pat, $content));
      if ($hit > 0) {
        $result[] = WikiCommentStorage::pageName($f);
      }
    }
    return $result;
  }
}

class HashDB {
  var $hash;

  function HashDB($path) {
    $this->path = $path;
    $this->hash = array();
    if (!file_exists($path)) {
      $this->init();
    }
  }
  function init() {
    $fp = fopen($this->path, "w") or die("database initialization error");
    fflush($fp);
    fclose($fp);
  }
  function exists($key) {
    $this->begin();
    return isset($this->hash[$key]);
  }
  function insert($key, $val) {
    $this->begin();
    if ($this->exists($key)) { return false; }
    $this->hash[$key] = $val;
    $this->commit();
    return true;
  }
  function update($key, $val) {
    $this->begin();
    if (!$this->exists($key)) return false;
    $this->hash[$key] = $val;
    $this->commit();
    return true;
  }
  function fetch($key) {
    $this->begin();
    return $this->hash[$key];
  }
  function keys() {
    $this->begin();
    return array_keys($this->hash);
  }
  function delete($key) {
    $this->begin();
    if (!$this->exists($key)) return false;
    unset($this->hash[$key]);
    $this->commit();
    return true;
  }

  function begin() {
    clearstatcache();
    if (file_exists($this->path)) {
      $fp = fopen($this->path, "r");
      $buf = fread($fp, filesize($this->path));
      fclose($fp);
    }
    else {
      $buf = '';
    }
    if (strlen($buf)) $hash = unserialize($buf);
    else $hash = array();
    $this->hash = $hash;
  }

  function commit() {
    $buf = serialize($this->hash);
    $path = $this->path;
    fwrite_safe($path, $buf);
  }
}

class ReadOnlyHashDB extends HashDB {
  var $isInitialized = false;

  function ReadOnlyHashDB($path) {
    parent::HashDB($path);
  }

  function insert($key, $val) {
    $this->write_error();
  }

  function update($key, $val) {
    $this->write_error();
  }

  function delete($key) {
    $this->write_error();
  }

  function begin() {
    if ($this->isInitialized) return;
    parent::begin();
    $this->isInitialized = true;
  }

  function write_wrror() {
    die("tried to write ReadOnlyHashDB");
  }

}
class Property {
  var $db;

  function Property($db) {
    $this->db = $db;
  }

  function instance() {
    global $pdb;
    $obj = new Property($pdb);
    return $obj;
  }

  function instanceReadOnly() {
    global $pdbReadOnly;
    $obj = new Property($pdbReadOnly);
    return $obj;
  }

  function get($page, $key) {
    if (!$this->db->exists($page)) {
      return null;
    }
       $hash = $this->db->fetch($page);
       if (isset($hash[$key])) {
         return $hash[$key];
       }
       else {
         return "";
       }
  }

  function set($page, $key, $value) {
    if ($this->db->exists($page)) {
      $hash = $this->db->fetch($page);
      $hash[$key] = $value;
      $this->db->update($page, $hash);
    }
    else {
      $hash = array();
      $hash[$key] = $value;
      $this->db->insert($page, $hash);
    }
  }
}

class Queue {
  function pushToList($name, $item) {
    $item = rtrim($item)."\n";
    $lines = file(Queue::fileName($name));
    array_unshift($lines, $item);
    Queue::updateList($name, $lines);
  }

  function appendToList($name, $item) {
    $item = rtrim($item)."\n";
    $lines = file(Queue::fileName($name));
    array_push($lines, $item);
    Queue::updateList($name, $lines);
  }

  function removeFromList($name, $pos) {
    $lines = file(Queue::fileName($name));
    list($removed) = array_splice($lines, $pos, 1);
    Queue::updateList($name, $lines);
    return $removed;
  }

  function updateList($name, $lines) {
    $target = Queue::fileName($name);
    $text = join("", $lines);
    fwrite_safe($target, $text);
  }

  function writeLog($name, $item) {
    $message = Queue::withTimeStamp($item);
    $target = Queue::logFileName($name);
    Queue::writeLogFile($target, $message);
  }

  function writeLogFile($target, $message) {
    $old = file($target);
    $new = Queue::addAndSuppless($old, $message);
    $text = join("", $new);
    fwrite_safe($target, $text);
  }

  function addAndSuppless($lines, $message) {
    array_push($lines, $message);
    if (count($lines) > QUEUE_MAX_LOG_SIZE) {
      array_splice($lines, 0, count($lines) - QUEUE_MAX_LOG_SIZE);
    }
    return $lines;
  }

  function removeFromLog($name, $dels) {
    $target = Queue::logFileName($name);
    $lines = file($target);
    $result = array();
    for ($i = 0; $i < count($lines); $i++) {
      if (!in_array($i, $dels)) {
        $result[] = $lines[$i];
      }
    }
    $text = join("", $result);
    fwrite_safe($target, $text);
  }

  function withTimeStamp($item) {
    $item = rtrim($item)."\n";
    $date = date("Y\tm\td\tH\ti\ts\t");
    return $date.$item;
  }

  function items($name) {
    $target = Queue::fileName($name);
    $lines = file($target);
    return $lines;
  }

  function logs($name) {
    $target = Queue::logFileName($name);
    $lines = file($target);
    return $lines;
  }

  function paramError() {
    die("param error");
  }

  function fileName($name) {
    return Queue::path($name, "queue.txt");
  }

  function logFileName($name) {
    return Queue::path($name, "quelog.txt");
  }

  function path($name, $suffix) {
    $path = WikiPageStorage::metaDataDir($name) ."/".$suffix;
    if (!file_exists($path))
      touch($path);
    return $path;
  }
}

?>
