# -*- coding: utf-8 -*-
class TicketsController < ApplicationController

  def new
    project_id = params[:project_id]

    @ticket = Ticket.new(project_id: project_id, state_id: TicketOpened, priority_id: MinorPriority, workload_id: EasyWork)
    load_attributes()

    respond_to do |format|
      format.html { redirect_to tickets_url }
      format.js
      format.json { render json: @ticket, status: :showed, location: @ticket }
    end
  end

  def create
    @ticket = Ticket.new(params[:ticket])

    respond_to do |format|
      if @ticket.save
        create_ticket_logs()

        format.html { redirect_to tickets_url }
        format.js
        format.json { render json: @ticket,
          status: :created, location: @ticket }

      else
        format.html { redirect_to tickets_url }
        format.js
        format.json { render json: @ticket,
          status: :unprocessable_entity, location: @ticket }
      end
      @project_tickets = create_project_tickets(session[:project_id])
      load_attributes()
    end
  end

  # チケット一覧の表示
  #
  # - プロジェクト毎にチケットを表示する。
  #   - あるプロジェクトのみのチケット表示ができるようにする。
  # - チケットは、優先度の高い順に表示する。
  # - 担当していないチケットを表示しないためのチェックボックスを用意する。
  def index
    # 所属している全てのプロジェクトを取得し、ユーザが指定する順番でソートする
    if params.has_key?(:project_id)
      if params[:project_id].to_i > 0
        session[:project_id] = params[:project_id]
      else
        session[:project_id] = nil
      end
    end
    @project_tickets = create_project_tickets(session[:project_id])
    load_attributes()

    # Ajax 実装のために無効な情報を登録する
    @ticket = 'not_ticket'

    # TODO is_ticket_page というフラグを用いる代わりの仕組みを調べて適用する
    @is_ticket_page = true
  end

  # チケット詳細情報の表示
  def show
    begin
      @ticket = Ticket.find(params[:id])
    rescue ActiveRecord::RecordNotFound
      logger.error "Attempt to access invalid ticket #{params[:id]}"
      redirect_to tickets_url, notice: 'Invalid ticket'
    else
      create_ticket_logs()
      load_attributes()
      @project_tickets = create_project_tickets(session[:project_id])
      @show_comment_only = true

      respond_to do |format|
        format.html { redirect_to tickets_url }
        format.js
        format.json { render json: @ticket, status: :showed, location: @ticket }
      end
    end
  end

  def update
    @ticket = Ticket.find(params[:id])
    if params[:move_to_project_id]
      project_id = params[:move_to_project_id]
      # ユーザが所属しているプロジェクトなのを確認してから
      # チケットの所属プロジェクトを変更する
      if @user.projects.find(project_id)
        @ticket.update_attributes({ project_id: project_id })

        create_ticket_logs()
        @project_tickets = create_project_tickets(session[:project_id])
        load_attributes()
        @show_comment_only = true
      end
    else
      respond_to do |format|
        if @ticket.update_attributes(params[:ticket])

          create_ticket_logs()
          @project_tickets = create_project_tickets(session[:project_id])
          load_attributes()
          @show_comment_only = true

          format.html { redirect_to @ticket }
          format.js
          format.json
        else
          format.html { render action: "show" }
          format.json { render json: @ticket.errors,
            status: :unprocessable_entity }
        end
      end
    end
  end

  private

  def create_project_tickets(project_id)
    projects = @user.projects.where(is_active: true)
    @belong_projects = Project.order_by_user_setting(projects,
                                                     @user.project_order_csv)
    if project_id.nil?
      projects = @belong_projects
    else
      # 指定されたプロジェクトのみを取得する
      selected_project = projects.where(id: project_id)
      projects =
        selected_project.size.zero? ? @belong_projects : selected_project
    end

    project_tickets = []
    no_ticket_project = []
    projects.each { |project|
      # チケット情報の取得
      if @user.is_resolved_show
        tickets = Ticket.where(project_id: project.id).order("state_id, priority_id, updated_at DESC")
      else
        tickets = Ticket.where(project_id: project.id, state_id: TicketOpened).
          order("state_id, priority_id, updated_at DESC")
      end

      if tickets.empty?
        # チケットがないプロジェクトは、リストの最後に追加されるようにする
        no_ticket_project << { project: project, tickets: tickets }
      else
        project_tickets << { project: project, tickets: tickets }
      end
    }
    project_tickets += no_ticket_project
  end

  def create_ticket_logs
    @ticket.comment = ''
    @ticket_logs = TicketLogs.where(ticket_id: @ticket.id)
  end

  private

  def load_attributes
    @all_states = State.all
    @all_priorities = Priority.all
    @all_workloads = Workload.all
  end
end
