# -*- coding: utf-8 -*-
class ApplicationController < ActionController::Base
  protect_from_forgery
  before_filter :authorize
  before_filter :set_i18n_locale_from_params
  before_filter :set_timezone

  TicketOpened = 1
  TicketResolved = 2
  MinorPriority = 3
  TrivialPriority = 4
  EasyWork = 1

  def routing_error
    render :file => "#{Rails.root}/public/404.html", :status => :not_found
  end

  private

  # 認証の処理
  def authorize
    unless User.find_by_id(session[:user_id])
      redirect_to login_url, alert: t('errors.messages.please_login')
    else
      @user = User.find(session[:user_id])
    end
  end

  # 管理者としての認証
  def admin_authorize
    unless User.find_by_id(session[:user_id])
      redirect_to login_url, alert: t('errors.messages.please_login')
    else
      unless User.find(session[:user_id]).is_admin?
        redirect_to login_url, alert: t('errors.messages.you_are_not_an_administratio')
      end
    end
  end

  # ロケールの設定
  def set_i18n_locale_from_params
    if params[:locale]
      if I18n.available_locales.include?(params[:locale].to_sym)
        I18n.locale = params[:locale]
      else
        flash.now[:notice] = "#{params[:locale]} translation not available"
        logger.error flash.now[:notice]
      end
    end
  end

  def default_url_options
    { locale: I18n.locale }
  end

  # タイムゾーンの設定
  def set_timezone
    if params[:timezone]
      session[:tz] = params[:timezone]
    end

    Time.zone = session[:tz] if session[:tz]
  end

  # パスワードの更新
  def change_password(target_user, change_user,
                      current_password, new_password, new_password_confirm)
    check_password = BCrypt::Password.new(change_user.password_digest)
    unless check_password == current_password
      return false, t('alert.messages.current_password_is_invalid')
    end

    if new_password.size.zero?
      return false, t('alert.messages.password_is_empty')
    end

    unless new_password == new_password_confirm
      return false, t('alert.messages.password_confirm_fail')
    end

    target_user.password = new_password
    target_user.save

    return true, t('notice.messages.password_changed')
  end
end
