# -*- coding: utf-8 -*-
class ChartController < ApplicationController

  PastDays = 22
  FutureDays = 8
  DivisionThreshold = 15

  def show
    # 表示の設定変更の指示があれば、更新する
    if params[:is_trivial_use]
      @user.is_trivial_use = params[:is_trivial_use] == 'true' ? true : false
      @user.save
    end
    project_id = params[:id]

    begin
      @project = Project.find(project_id)
    rescue ActiveRecord::RecordNotFound
      logger.error "Attempt to access invalid chart #{project_id}"
      redirect_to tickets_url, notice: 'Invalid chart'
    else
      # Burn down チャートの作成
      burn_up_url = url_for(action: 'burn_up_chart', project_id: project_id)
      @chart = open_flash_chart_object(800, 400, burn_up_url)

      respond_to do |format|
        #format.html { redirect_to tickets_url, notice: 'Invalid chart access' }
        format.html { render template: 'tickets/draw_chart' }
        format.js { render template: 'tickets/draw_chart' }
      end
    end
  end

  def update
  end

  def burn_up_chart
    project_id = params[:project_id]
    project = Project.find(project_id)
    raw_date, total, completed, y_min, y_max = burn_up_data(project_id)

    if raw_date.nil?
      # 表示できるデータがない場合は、メッセージを表示する
      return draw_no_ticket_message()
    end

    total_graph = LineDot.new
    total_graph.set_tooltip(t('chart.show.total') + ': #val#')
    total_graph.set_key(t('chart.show.total'), 14)
    total_graph.width = 3
    total_graph.dot_size = 5
    total_graph.colour = '#dc143c'
    total_graph.values = total

    completed_graph = Bar.new
    completed_graph.set_tooltip(t('chart.show.completed') + ': #val#')
    completed_graph.set_key(t('chart.show.completed'), 14)
    completed_graph.width = 2
    completed_graph.colour = '#3333ff'
    completed_graph.values = completed

    x_labels = XAxisLabels.new
    x_labels.labels = date_labels(raw_date)

    x_axis = XAxis.new
    x_axis.labels = x_labels

    y_axis = YAxis.new
    step = y_range_step(y_min, y_max)
    y_axis.set_range(y_min, y_max, step)

    y_legend = YLegend.new('Workloads')
    y_legend.set_style('{font-size: 14px; color: #000000}')

    chart = OpenFlashChart.new
    chart.set_y_legend(y_legend)
    chart.x_axis = x_axis
    chart.y_axis = y_axis

    chart.add_element(total_graph)
    chart.add_element(completed_graph)

    render :text => chart.to_s
  end

  private

  def burn_up_data(project_id)
    if @user.is_trivial_use
      workloads = CompletedWorkloads.where(project_id: project_id).order("date DESC")
    else
      workloads = CompletedWorkloads.where("project_id = ? and priority_id != ?", project_id, TrivialPriority).order("date DESC")
    end

    return nil if workloads.empty?

    # データの読み出し
    raw_data = {}
    last_total = nil
    last_completed = nil
    workloads.each do |workload|
      date_key = workload.date.to_s

      unless raw_data.has_key?(date_key)
        raw_data[date_key] = {
          total: 0,
          completed: 0,
        }
      end
      raw_data[date_key][:total] += workload.total
      raw_data[date_key][:completed] += workload.completed
    end

    # データを日毎に展開する
    raw_date = []
    total = []
    completed = []
    last_total = nil
    last_completed = nil
    max_total = 0
    day = workloads.last.date

    while day <= Date.today do
      date_key = day.to_s

      raw_date << day

      if raw_data.has_key?(date_key)
        if last_total.nil?
          last_total = 0
          last_completed = 0
        end
        last_total += raw_data[date_key][:total]
        last_completed += raw_data[date_key][:completed]
      end
      total << last_total
      completed << last_completed

      max_total = last_total if not last_total.nil? and (last_total > max_total)

      day += 1
    end
    max_total += 5

    if total.size < PastDays
      add_size = PastDays - total.size
      # 空の領域をデータの先頭に追加する
      total = Array.new(add_size, nil) + total
      completed = Array.new(add_size, nil) + completed
      add_day = raw_date.first - add_size
      add_date = []
      add_size.times do
        add_date << add_day
        add_day += 1
      end
      raw_date = add_date + raw_date
    elsif total.size > PastDays
      # 表示しない範囲のデータを切り捨てる
      total = total[-PastDays, PastDays]
      completed = completed[-PastDays, PastDays]
      raw_date = raw_date[-PastDays, PastDays]
    end

    FutureDays.times do
      raw_date << day
      day += 1
    end

    return raw_date, total, completed, 0, max_total
  end

  def wday_color(date)
    if date.wday == 0
      color = '#ff0000'
    elsif date.wday == 6
      color = '#0000ff'
    else
      color = '#000000'
    end
    return color
  end

  def day_label(date)
    if date.day == 1
      label = date.month.to_s + '/1'
    else
      label = date.day.to_s
    end

    if date == Date.today
      label = "    " + label + "\nToday"
    end

    return label
  end

  def date_labels(raw_date)
    labels = []
    raw_date.each do |date|
      # 曜日によって色を変える
      color = wday_color(date)

      # 表示する年月日の調整
      label = day_label(date)

      labels << XAxisLabel.new(label, color, 12, 0)
    end
    return labels
  end

  def draw_no_ticket_message
    title = Title.new(t("chart.show.this_project_dont_have_a_ticket_yet"))
    title.set_style( "{font-size: 26px; font-weight: bold; color: #000000; text-align: center;}" )

    y_axis = YAxis.new
    y_axis.set_range(0, 15, 1)

    raw_date = []
    date = Date.today - PastDays
    (PastDays + FutureDays).times do |i|
      raw_date << date
      date += 1
    end

    x_labels = XAxisLabels.new
    x_labels.labels = date_labels(raw_date)
    x_axis = XAxis.new
    x_axis.labels = x_labels

    dummy_bar = Bar.new
    dummy_bar.values = [0]

    chart = OpenFlashChart.new
    chart.set_title(title)
    chart.add_element(dummy_bar)
    chart.x_axis = x_axis
    chart.y_axis = y_axis
    render :text => chart.to_s
  end

  def y_range_step(y_min, y_max)
    range = y_max - y_min
    divisors = [1, 2, 5, 10, 20, 25, 50, 100, 200, 250, 500, 1000, 2000 ]

    divisors.each do |divisor|
      if range / divisor < DivisionThreshold
        return divisor
      end
    end
    return divisors.last
  end
end
