# -*- coding: utf-8 -*-
class Project < ActiveRecord::Base
  has_and_belongs_to_many :tickets
  has_and_belongs_to_many :users
  belongs_to :created_by, class_name: 'User', foreign_key: 'created_by_user_id'

  validates :title, presence: true

  after_create :add_creater_to_project

  TicketOpened = 1

  public

  def is_private?
    (self.is_group and self.is_open) ? false : true
  end

  def Project.order_by_user_setting(projects, project_order_csv)
    project_order = project_order_csv.split(',')
    if project_order.size <= 1
      # ソートの指定がないか、プロジェクト数が 1 以下ならばソートしない
      projects
    end

    sorted_projects = []
    project_order.each do |project_id|
      found_project = projects.find_by_id(project_id)
      sorted_projects << found_project unless found_project.nil?
    end

    # ソート対象でないプロジェクトがあるかもしれないため
    # プロジェクトの配列を結合して、重複を取り除いた配列を返す
    return (sorted_projects + projects).uniq
  end

  def Project.belongs_projects(user)
    belong_projects = user.projects.where(is_active: true)
    ordered_projects = Project.order_by_user_setting(belong_projects,
                                                     user.project_order_csv)
    return ordered_projects
  end

  def Project.active_tickets_count(project_id)
    return Ticket.where(state_id: 1, project_id: project_id).count
  end

  def Project.create_order_projects(projects)
    current_order = []
    projects.each do |project|
      current_order << project.id
    end

    order_projects = []
    previous_order_csv = nil
    previous_changed_id = []
    index = 0
    projects.each do |project|
      order_csv = create_order_csv(current_order.dup, index)
      changed_id = [ current_order[index], current_order[index + 1] ]

      order_projects << {
        project_id: project.id,
        up_changed_id: previous_changed_id,
        down_changed_id: changed_id,
        title: project.title,
        up_csv: previous_order_csv,
        down_csv: order_csv,
      }
      previous_order_csv = order_csv
      previous_changed_id = changed_id
      index += 1
    end
    order_projects
  end

  private

  def add_creater_to_project
    # プロジェクトを作成したユーザをメンバーに加える
    ProjectsUsers.create(project_id: self.id, user_id: self.created_by_user_id)
  end

  def Project.create_order_csv(order, index)
    order[index], order[index + 1] = order[index + 1], order[index]

    output = ''
    order.each do |value|
      if value.nil?
        # 最後の要素の down のときには nil を返す
        return nil
      end
      output += value.to_s + ','
    end

    return output
  end

  def Project.classify_users(project)
    {
      create_user: project.created_by,
      members: project.users - [ project.created_by ],
      non_members: User.all - project.users
    }
  end
end
