# -*- coding: utf-8 -*-
class ApplicationController < ActionController::Base
  protect_from_forgery
  before_filter :authorize
  before_filter :set_i18n_locale_from_params
  before_filter :set_timezone

  TicketOpened = 1
  TicketResolved = 2
  MinorPriority = 3
  TrivialPriority = 4
  EasyWork = 1

  def routing_error
    render :file => "#{Rails.root}/public/404.html", :status => :not_found
  end

  private

  # 認証の処理
  def authorize
    unless User.find_by_id(session[:user_id])
      redirect_to login_url, alert: t('errors.messages.please_login')
    else
      @user = User.find(session[:user_id])
    end
  end

  # 管理者としての認証
  def admin_authorize
    unless User.find_by_id(session[:user_id])
      redirect_to login_url, alert: t('errors.messages.please_login')
    else
      unless User.find(session[:user_id]).is_admin?
        redirect_to login_url, alert: t('errors.messages.you_are_not_an_administratio')
      end
    end
  end

  # ロケールの設定
  def set_i18n_locale_from_params
    if params[:locale]
      if I18n.available_locales.include?(params[:locale].to_sym)
        I18n.locale = params[:locale]
      else
        flash.now[:notice] = "#{params[:locale]} translation not available"
        logger.error flash.now[:notice]
      end
    end
  end

  def default_url_options
    { locale: I18n.locale }
  end

  # タイムゾーンの設定
  def set_timezone
    if params[:timezone]
      session[:tz] = params[:timezone]
    end

    Time.zone = session[:tz] if session[:tz]
  end

  # パスワードの更新
  def change_password(target_user, change_user,
                      current_password, new_password, new_password_confirm)
    check_password = BCrypt::Password.new(change_user.password_digest)
    unless check_password == current_password
      return false, t('alert.messages.current_password_is_invalid')
    end

    if new_password.size.zero?
      return false, t('alert.messages.password_is_empty')
    end

    unless new_password == new_password_confirm
      return false, t('alert.messages.password_confirm_fail')
    end

    target_user.password = new_password
    target_user.save

    return true, t('notice.messages.password_changed')
  end

  # チケットログの作成
  def create_ticket_logs
    @ticket.comment = ''
    @ticket_logs = TicketLogs.where(ticket_id: @ticket.id)
  end

  # チケット属性の読み込み
  def load_ticket_attributes
    @all_states = State.all
    @all_priorities = Priority.all
    @all_workloads = Workload.all
  end

  # プロジェクトに所属しているチケットの取得
  def create_project_tickets(project_id)
    projects = @user.projects.where(is_active: true)
    @belong_projects = Project.order_by_user_setting(projects,
                                                     @user.project_order_csv)
    if project_id.nil?
      projects = @belong_projects
    else
      # 指定されたプロジェクトのみを取得する
      selected_project = projects.where(id: project_id)
      projects =
        selected_project.size.zero? ? @belong_projects : selected_project
    end

    project_tickets = []
    no_ticket_project = []
    projects.each { |project|
      # チケット情報の取得
      if @user.is_resolved_show
        tickets = Ticket.where(project_id: project.id).order("state_id, priority_id, updated_at DESC")
      else
        tickets = Ticket.where(project_id: project.id, state_id: TicketOpened).
          order("state_id, priority_id, updated_at DESC")
      end

      if tickets.empty?
        # チケットがないプロジェクトは、リストの最後に追加されるようにする
        no_ticket_project << { project: project, tickets: tickets }
      else
        project_tickets << { project: project, tickets: tickets }
      end
    }

    # 未読チケット情報の取得
    @unread_flags = {}
    unread = UnreadFlag.select("ticket_id").where(user_id: @user.id)
    unread.each { |record|
      @unread_flags[record.ticket_id] = true
    }

    project_tickets += no_ticket_project
  end
end
