package com.ozacc.blog.feed.impl;

import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;

import org.apache.commons.httpclient.Credentials;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.HttpRecoverableException;
import org.apache.commons.httpclient.HttpState;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.UsernamePasswordCredentials;
import org.apache.commons.httpclient.HttpConnection.ConnectionTimeoutException;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.ozacc.blog.feed.Channel;
import com.ozacc.blog.feed.FeedAutoDiscovery;
import com.ozacc.blog.feed.FeedParser;
import com.ozacc.blog.feed.ParseException;
import com.ozacc.blog.feed.ParseTimeoutException;
import com.ozacc.blog.feed.support.XmlInputStreamParser;
import com.ozacc.blog.feed.support.impl.RomeXmlInputSreamParserImpl;
import com.ozacc.blog.util.CommonsHttpClientUtils;

/**
 * フィードの XML 解析を <code>XmlInputStreamParser</code> に委譲する FeedParser 実装クラス。
 * 
 * @see XmlInputStreamParser
 * 
 * @since 1.2
 * @author Tomohiro Otsuka
 * @version $Id: FeedParserImpl.java 197 2005-08-10 01:43:23Z otsuka $
 */
public class FeedParserImpl implements FeedParser {

	/** デフォルトの接続タイムアウト時間、5,000ミリ秒。*/
	public static final int DEFAULT_CONNECTION_TIMEOUT = 5000;

	/** デフォルトの読込タイムアウト時間、5,000ミリ秒。*/
	public static final int DEFAULT_READ_TIMEOUT = 5000;

	private static Log log = LogFactory.getLog(FeedParserImpl.class);

	private int connectionTimeout = DEFAULT_CONNECTION_TIMEOUT;

	private int readTimeout = DEFAULT_READ_TIMEOUT;

	private boolean followRedirect = true;

	private FeedAutoDiscovery feedAutoDiscovery;

	private XmlInputStreamParser xmlInputStreamParser;

	/**
	 * コンストラクタ。
	 */
	public FeedParserImpl() {
		super();
	}

	private void initXmlInputStreamParser() {
		xmlInputStreamParser = new RomeXmlInputSreamParserImpl();
	}

	private void initFeedAutoDiscovery() {
		feedAutoDiscovery = new FeedAutoDiscoveryImpl();
	}

	/**
	 * 接続タイムアウト時間をセットします。単位はミリ秒。
	 * デフォルトは5,000ミリ秒(5秒)です。<br>
	 * 0をセットすると無制限になります。
	 * 
	 * @param connectionTimeout 接続タイムアウト (ms)
	 */
	public void setConnectionTimeout(int connectionTimeout) {
		this.connectionTimeout = connectionTimeout;
	}

	/**
	 * 接続後の読込タイムアウト時間をセットします。単位はミリ秒。
	 * デフォルトは5,000ミリ秒(5秒)です。<br>
	 * 0をセットすると無制限になります。
	 * 
	 * @param timeout 読込タイムアウト (ms)
	 */
	public void setReadTimeout(int timeout) {
		this.readTimeout = timeout;
	}

	/**
	 * XmlInputStreamParserの実装インスタンスをセットします。
	 * 
	 * @param xmlInputStreamParser XmlInputStreamParserの実装インスタンス
	 */
	public void setXmlInputStreamParser(XmlInputStreamParser xmlInputStreamParser) {
		this.xmlInputStreamParser = xmlInputStreamParser;
	}

	/**
	 * @see com.ozacc.blog.feed.FeedParser#parseFeed(java.net.URL)
	 */
	public Channel parseFeed(URL rssUrl) throws ParseException {
		return parseFeed(rssUrl.toString());
	}

	/**
	 * @see com.ozacc.blog.feed.FeedParser#parseFeed(java.lang.String)
	 */
	public Channel parseFeed(String url) throws ParseException {
		return parseFeed(url, null, null);
	}

	private Channel parseXmlInputStream(InputStream is) throws ParseException {
		log.debug("RSSをパースします。");
		return xmlInputStreamParser.parseXmlInputStream(is);
	}

	/**
	 * @see com.ozacc.blog.feed.FeedParser#discoverAndParseFeed(java.net.URL)
	 */
	public Channel discoverAndParseFeed(URL targetPageUrl) throws ParseException {
		if (feedAutoDiscovery == null) {
			initFeedAutoDiscovery();
		}
		URL[] rssUrls;
		try {
			rssUrls = feedAutoDiscovery.discoverFeedUrls(targetPageUrl);
		} catch (ParseException e) {
			throw new ParseException("RSSのAutoDiscoveryに失敗しました。[url='" + targetPageUrl + "']", e);
		}
		if (rssUrls.length > 0) {
			return parseFeed(rssUrls[0]);
		}
		throw new ParseException("指定されたURLのページではRSSのURLを見つけられませんでした。[url='" + targetPageUrl + "']");
	}

	/**
	 * @see com.ozacc.blog.feed.FeedParser#discoverAndParseFeed(java.lang.String)
	 */
	public Channel discoverAndParseFeed(String targetPageUrl) throws ParseException {
		try {
			return discoverAndParseFeed(new URL(targetPageUrl));
		} catch (MalformedURLException e) {
			throw new ParseException("指定されたURLのフォーマットが不正です。[url='" + targetPageUrl + "']", e);
		}
	}

	/**
	 * @since 1.2.3
	 * @see com.ozacc.blog.feed.FeedParser#parseFeed(java.lang.String, java.lang.String, java.lang.String)
	 */
	public Channel parseFeed(String url, String userName, String password) throws ParseException {
		if (xmlInputStreamParser == null) {
			initXmlInputStreamParser();
		}

		HttpClient client = new HttpClient();
		log.debug("接続タイムアウトを設定します。[" + connectionTimeout + "]");
		client.setConnectionTimeout(connectionTimeout);
		log.debug("読込タイムアウトを設定します。[" + readTimeout + "]");
		client.setTimeout(readTimeout);

		HttpMethod method = new GetMethod(url);
		method.setFollowRedirects(followRedirect);
		if (userName != null) {
			log.debug("認証情報を設定します。[userName='" + userName + "', password='" + password + "']");
			HttpState state = client.getState();
			Credentials credentials = new UsernamePasswordCredentials(userName, password);
			state.setCredentials(null, null, credentials);
		}
		try {
			log.debug("HTTP接続を行います。[" + url + "]");
			int statusCode = client.executeMethod(method);
			if (!CommonsHttpClientUtils.isSuccessfulResponse(statusCode, followRedirect)) {
				throw new ParseException("HTTP接続ができませんでした。[HTTP_STATUS='"
						+ HttpStatus.getStatusText(statusCode) + "', rssUrl='" + url + "']");
			}
			InputStream is = method.getResponseBodyAsStream();
			return parseXmlInputStream(is);
		} catch (ConnectionTimeoutException e) {
			throw new ParseTimeoutException("HTTP接続タイムアウト。[rssUrl='" + url + "']", e);
		} catch (HttpRecoverableException e) {
			throw new ParseTimeoutException("HTTP読込タイムアウト。[rssUrl='" + url + "']", e);
		} catch (ParseException e) {
			throw new ParseException("RSSのパースに失敗しました。[rssUrl='" + url + "']", e);
		} catch (IOException e) {
			throw new ParseException("指定されたURLからのRSS取得に失敗しました。[rssUrl='" + url + "']", e);
		} finally {
			method.releaseConnection();
		}
	}

	/**
	 * @since 1.2.3
	 * @see com.ozacc.blog.feed.FeedParser#parseFeed(java.net.URL, java.lang.String, java.lang.String)
	 */
	public Channel parseFeed(URL url, String userName, String password) throws ParseException {
		return parseFeed(url.toString(), userName, password);
	}

	/**
	 * アクセスしたURLがリダイレクトレスポンス(HTTP Status Code 3xx)を返してきた場合に、
	 * リダイレクト先にアクセスするかどうかを設定します。デフォルトでは、リダイレクト先にアクセスします。
	 * 
	 * @since 1.2.4
	 * @param followRedirect リダイレクト先にアクセスする場合 true。デフォルトはtrue。
	 */
	public void setFollowRedirect(boolean followRedirect) {
		this.followRedirect = followRedirect;
	}

	public void setFeedAutoDiscovery(FeedAutoDiscovery feedAutoDiscovery) {
		this.feedAutoDiscovery = feedAutoDiscovery;
	}
}