package com.ozacc.blog.feed.support.impl;

import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.ozacc.blog.feed.Channel;
import com.ozacc.blog.feed.Item;
import com.ozacc.blog.feed.ParseException;
import com.ozacc.blog.feed.impl.ChannelImpl;
import com.ozacc.blog.feed.impl.ItemImpl;
import com.ozacc.blog.feed.support.XmlInputStreamParser;
import com.sun.syndication.feed.synd.SyndCategory;
import com.sun.syndication.feed.synd.SyndContent;
import com.sun.syndication.feed.synd.SyndEntry;
import com.sun.syndication.feed.synd.SyndFeed;
import com.sun.syndication.io.FeedException;
import com.sun.syndication.io.SyndFeedInput;
import com.sun.syndication.io.XmlReader;

/**
 * XmlInputStreamParser インターフェースの実装クラス。
 * Rome ライブラリを使用してフィード XML を解析し、Channel インスタンスを生成します。
 * 
 * @author Tomohiro Otsuka
 * @version $Id: RomeXmlInputSreamParserImpl.java 197 2005-08-10 01:43:23Z otsuka $
 */
public class RomeXmlInputSreamParserImpl implements XmlInputStreamParser {

	private static Log log = LogFactory.getLog(RomeXmlInputSreamParserImpl.class);

	/**
	 * コンストラクタ。
	 */
	public RomeXmlInputSreamParserImpl() {
		super();
	}

	/**
	 * @see com.ozacc.blog.feed.support.XmlInputStreamParser#parseXmlInputStream(java.io.InputStream)
	 */
	public Channel parseXmlInputStream(InputStream is) throws ParseException {
		SyndFeedInput input = new SyndFeedInput();
		SyndFeed feed;
		try {
			feed = input.build(new XmlReader(is));
		} catch (IOException e) {
			throw new ParseException("XMLの読み込みに失敗しました。", e);
		} catch (IllegalArgumentException e) {
			throw new ParseException("対応していないフォーマットのフィードです。", e);
		} catch (FeedException e) {
			throw new ParseException("フィードの解析に失敗しました。", e);
		}
		return buildChannelFromSyndFeed(feed);
	}

	protected Channel buildChannelFromSyndFeed(SyndFeed feed) {
		ChannelImpl channel = new ChannelImpl();
		channel.setCreator(feed.getAuthor());
		channel.setDate(feed.getPublishedDate());
		channel.setDescription(feed.getDescription());
		channel.setLanguage(feed.getLanguage());
		try {
			channel.setLink(new URL(feed.getLink()));
		} catch (MalformedURLException e) {
			log.warn("フィードURLが不正フォーマット。", e);
		}
		channel.setTitle(feed.getTitle());
		//	Category
		List cats = feed.getCategories();
		if (cats.size() > 0) {
			SyndCategory cat = (SyndCategory)cats.get(0);
			channel.setSubject(cat.getName());
		}
		// Description
		List entries = feed.getEntries();
		for (Iterator iter = entries.iterator(); iter.hasNext();) {
			SyndEntry entry = (SyndEntry)iter.next();
			channel.addItem(toItem(entry));
		}

		return channel;
	}

	protected Item toItem(SyndEntry entry) {
		ItemImpl item = new ItemImpl();
		item.setCreator(entry.getAuthor());
		item.setDate(entry.getPublishedDate());
		item.setTitle(entry.getTitle());
		try {
			item.setLink(new URL(entry.getLink()));
		} catch (MalformedURLException e) {
			log.warn("フィードアイテムのURLが不正フォーマット。", e);
		}
		// Category
		List cats = entry.getCategories();
		if (cats.size() > 0) {
			SyndCategory cat = (SyndCategory)cats.get(0);
			item.setSubject(cat.getName());
		}
		// Description
		SyndContent content = entry.getDescription();
		String desc = content.getValue();
		item.setDescription(desc);

		return item;
	}

}
