package com.ozacc.blog.trackback.impl;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.FactoryConfigurationError;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpRecoverableException;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.NameValuePair;
import org.apache.commons.httpclient.HttpConnection.ConnectionTimeoutException;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import com.ozacc.blog.trackback.ConnectionException;
import com.ozacc.blog.trackback.FailedTrackBackException;
import com.ozacc.blog.trackback.TrackBackClient;
import com.ozacc.blog.trackback.TrackBackException;
import com.ozacc.blog.trackback.TrackBackPing;
import com.ozacc.blog.trackback.TrackBackResponseException;
import com.ozacc.blog.util.CommonsHttpClientUtils;

/**
 * TrackBackClientインターフェースの実装クラス。
 * 
 * @since 1.0
 * @author Tomohiro Otsuka
 * @version $Id: TrackBackClientImpl.java 180 2005-07-22 09:26:25Z otsuka $
 */
public class TrackBackClientImpl implements TrackBackClient {

	public static final int DEFAULT_CONNECTION_TIMEOUT = 5000;

	public static final int DEFAULT_READ_TIMEOUT = 5000;

	/** デフォルトの文字コード「UTF-8」。 */
	public static final String DEFAULT_CHARSET = "utf-8";

	private static Log log = LogFactory.getLog(TrackBackClientImpl.class);

	private String userAgent;

	private String charset = DEFAULT_CHARSET;

	private int connectionTimeout = DEFAULT_CONNECTION_TIMEOUT;

	private int readTimeout = DEFAULT_READ_TIMEOUT;

	/**
	 * コンストラクタ。
	 */
	public TrackBackClientImpl() {}

	/**
	 * コンストラクタ。<br>
	 * このクライアントのユーザーエージェント名をセットします。
	 * この値はPing送信先サーバのアクセスログに残る可能性があります。
	 * 
	 * @param userAgent ユーザーエージェント名
	 */
	public TrackBackClientImpl(String userAgent) {
		setUserAgent(userAgent);
	}

	/**
	 * TrackBackClientのユーザエージェント名
	 * 
	 * @return トラックバッククライアントのユーザエージェント名
	 */
	public String getUserAgent() {
		return userAgent;
	}

	/**
	 * トラックバックデータのエンコードに使用する文字コードを返します。
	 * 
	 * @return トラックバックデータのエンコードに使用する文字コード
	 */
	public String getCharset() {
		return charset;
	}

	/**
	 * トラックバックデータのエンコードに使用する文字コードを指定します。
	 * デフォルトはUTF-8です。ほとんどのBlogサービスでは、UTF-8を標準としていますので、デフォルトを変更しないことを推奨します。
	 * 
	 * @param charset トラックバックデータのエンコードに使用する文字コード
	 */
	public void setCharset(String charset) {
		this.charset = charset;
	}

	/**
	 * このTrackBackClientのユーザエージェント名を指定します。
	 * 指定しない場合場合は、トラックバック Ping データに含まれる blog_name の値が使用されます。
	 * 
	 * @param userAgent TrackBackClientのユーザエージェント名
	 */
	public void setUserAgent(String userAgent) {
		this.userAgent = userAgent;
	}

	/**
	 * @see com.ozacc.blog.trackback.TrackBackClient#ping(java.net.URL, com.ozacc.blog.trackback.TrackBackPing)
	 */
	public void ping(URL trackBackPingUrl, TrackBackPing ping) throws TrackBackException {
		ping(trackBackPingUrl.toString(), ping);
	}

	/**
	 * @see com.ozacc.blog.trackback.TrackBackClient#ping(java.lang.String, com.ozacc.blog.trackback.TrackBackPing)
	 */
	public void ping(String trackBackPingUrl, TrackBackPing ping) throws TrackBackException {
		HttpClient client = new HttpClient();
		log.debug("接続タイムアウトを設定します。[" + connectionTimeout + "]");
		client.setConnectionTimeout(connectionTimeout);
		log.debug("読込タイムアウトを設定します。[" + readTimeout + "]");
		client.setTimeout(readTimeout);

		// 使用する文字コードを決定
		String appliedCharset = null;
		if (ping.getCharset() != null && ping.getCharset().length() > 0) {
			appliedCharset = ping.getCharset();
		} else {
			appliedCharset = charset;
		}

		PostMethod post = new PostMethod(trackBackPingUrl);
		NameValuePair[] data = buildPostData(ping);
		post.setRequestBody(data);
		post.setRequestHeader("Content-Type", "application/x-www-form-urlencoded; charset="
				+ appliedCharset);
		// User-Agentのセット
		if (userAgent != null) {
			post.setRequestHeader("User-Agent", userAgent);
		}

		try {
			int statusCode = client.executeMethod(post);
			if (!CommonsHttpClientUtils.isSuccessfulResponse(statusCode, false)) {
				throw new ConnectionException("指定されたTrackBack URLにHTTP接続ができませんでした。[HTTP_STATUS='"
						+ HttpStatus.getStatusText(statusCode) + "', TrackBackURL='"
						+ trackBackPingUrl + "']");
			}
			InputStream is = post.getResponseBodyAsStream();
			String message = parseResponseStream(is);
			if (message != null) {
				throw new FailedTrackBackException(message);
			}
			is.close();
		} catch (ConnectionTimeoutException e) {
			throw new com.ozacc.blog.trackback.ConnectionTimeoutException(
					"HTTP接続タイムアウト。[TrackBackURL='" + trackBackPingUrl + "']", e);
		} catch (HttpRecoverableException e) {
			throw new com.ozacc.blog.trackback.ConnectionTimeoutException(
					"HTTP読込タイムアウト。[TrackBackURL='" + trackBackPingUrl + "']", e);
		} catch (IOException e) {
			throw new ConnectionException("トラックバックの送信に失敗しました。[TrackBackURL='" + trackBackPingUrl
					+ "']", e);
		} finally {
			post.releaseConnection();
		}
	}

	/**
	 * 指定されたトラックバックPingを、Commons-HttpClientが使用するNaveValuePair[]に変換します。
	 * 
	 * @param ping トラックバックPing
	 * @return トラックバックPingデータを示すNaveValuePair[]
	 */
	private NameValuePair[] buildPostData(TrackBackPing ping) {
		List list = new ArrayList();

		// charset
		if (charset != null && charset.length() > 0) {
			list.add(new NameValuePair("charset", charset.toLowerCase()));
		}

		if (ping.getTitle() != null) {
			list.add(new NameValuePair("title", ping.getTitle()));
		}
		if (ping.getUrl() != null) {
			list.add(new NameValuePair("url", ping.getUrl()));
		}
		if (ping.getExcerpt() != null) {
			list.add(new NameValuePair("excerpt", ping.getExcerpt()));
		}
		if (ping.getBlogName() != null) {
			list.add(new NameValuePair("blog_name", ping.getBlogName()));
		}
		return (NameValuePair[])list.toArray(new NameValuePair[list.size()]);
	}

	/**
	 * @param is
	 * @return
	 * @throws TrackBackResponseException
	 * @throws FactoryConfigurationError
	 * @throws SAXException
	 * @throws IOException
	 * @throws ParserConfigurationException 
	 */
	private String parseResponseStream(InputStream is) throws TrackBackResponseException {
		try {
			DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
			factory.setIgnoringElementContentWhitespace(true);
			factory.setIgnoringComments(true);
			Document doc = factory.newDocumentBuilder().parse(is);

			NodeList errorList = doc.getElementsByTagName("error");
			Node errorNode = errorList.item(0);
			String errorCode = errorNode.getFirstChild().getNodeValue();
			if ("0".equals(errorCode)) {
				return null;
			}

			NodeList msgList = doc.getElementsByTagName("message");
			Node msgNode = msgList.item(0);
			String message = msgNode.getFirstChild().getNodeValue();
			return message;
		} catch (Exception ex) {
			throw new TrackBackResponseException("サーバレスポンスの取得に失敗しました。", ex);
		} finally {
			if (is != null) {
				try {
					is.close();
				} catch (IOException e) {
					// ignore
				}
			}
		}
	}

	/**
	 * 接続タイムアウト時間をセットします。単位はミリ秒。
	 * デフォルトは5,000ミリ秒(5秒)です。
	 * 
	 * @param connectionTimeout 接続タイムアウト (ms)
	 */
	public void setConnectionTimeout(int connectionTimeout) {
		this.connectionTimeout = connectionTimeout;
	}

	/**
	 * 接続後の読込タイムアウト時間をセットします。単位はミリ秒。
	 * デフォルトは5,000ミリ秒(5秒)です。
	 * 
	 * @param timeout 読込タイムアウト (ms)
	 */
	public void setReadTimeout(int timeout) {
		this.readTimeout = timeout;
	}
}