package com.ozacc.mail.spring;

import java.io.File;

import org.springframework.beans.factory.config.AbstractFactoryBean;
import org.springframework.core.io.Resource;

import com.ozacc.mail.Mail;
import com.ozacc.mail.MailBuildException;
import com.ozacc.mail.MailBuilder;
import com.ozacc.mail.impl.XMLMailBuilderImpl;

/**
 * Springの設定ファイルで指定されたロケーションのXMLファイルからMailインスタンスを生成するFactoryBean。
 * デフォルトでは、singletonプロパティはfalseに設定されます。
 * <p>
 * location、classPath、filePathの順で、一番先にセットされているプロパティ値がXMLファイルのパスとして使われます。
 * 
 * @see com.ozacc.mail.impl.XMLMailBuilderImpl
 * 
 * @since 1.0
 * @author Tomohiro Otsuka
 * @version $Id: XMLMailFactoryBean.java,v 1.4 2004/09/13 19:48:16 otsuka Exp $
 */
public class XMLMailFactoryBean extends AbstractFactoryBean {

	private String classPath;

	private String filePath;

	private Resource location;

	private MailBuilder mailBuilder;

	/**
	 * コンストラクタ。
	 */
	public XMLMailFactoryBean() {
		setSingleton(false);
	}

	/**
	 * @see org.springframework.beans.factory.config.AbstractFactoryBean#createInstance()
	 */
	protected Object createInstance() throws Exception {
		if (mailBuilder == null) {
			init();
		}

		if (getLocation() != null && getLocation().getFile() != null) {
			return mailBuilder.buildMail(getLocation().getFile());
		}
		if (getClassPath() != null) {
			return mailBuilder.buildMail(getClassPath());
		}
		if (getFilePath() != null) {
			return mailBuilder.buildMail(new File(getFilePath()));
		}
		throw new MailBuildException("Mailインスタンスの生成に失敗しました。XMLデータのロケーションが指定されていません。");
	}

	/**
	 * mailBuilderインスタンスを生成します。
	 */
	private void init() {
		mailBuilder = new XMLMailBuilderImpl();
	}

	/**
	 * @see org.springframework.beans.factory.FactoryBean#getObjectType()
	 */
	public Class getObjectType() {
		return Mail.class;
	}

	/**
	 * <code>MailBuilder</code>インターフェースの実装クラスのインスタンスをセットします。
	 * デフォルトでは、<code>XMLMailBuilderImpl</code>が使用されます。
	 * <p>
	 * ただし、ここでセットしない場合は、<code>XMLMailFactoryBean</code>ひとつに付き、
	 * <code>XMLMailBuilderImpl</code>インスタンス一つが保持されます。
	 * シングルトンの<code>MailBuilder</code>インスタンスをセットすることを推奨します。
	 * 
	 * @param mailBuilder MailBuilderインスタンス
	 */
	public void setMailBuilder(MailBuilder mailBuilder) {
		this.mailBuilder = mailBuilder;
	}

	/**
	 * @return Returns the classPath.
	 */
	public String getClassPath() {
		return classPath;
	}

	/**
	 * @param classPath The classPath to set.
	 */
	public void setClassPath(String classPath) {
		this.classPath = classPath;
	}

	/**
	 * @return Returns the filePath.
	 */
	public String getFilePath() {
		return filePath;
	}

	/**
	 * @param filePath The filePath to set.
	 */
	public void setFilePath(String filePath) {
		this.filePath = filePath;
	}

	/**
	 * @return Returns the location.
	 */
	public Resource getLocation() {
		return location;
	}

	/**
	 * @param location The location to set.
	 */
	public void setLocation(Resource location) {
		this.location = location;
	}
}