package org.sqlite;

import java.sql.SQLException;
import java.util.HashMap;
import java.util.Map;
import org.sqlite.jdbc.JdbcSQLException;
import org.sqlite.swig.SQLite3JNI;
import org.sqlite.types.SQLite3Handle;
import org.sqlite.swig.SWIGTYPE_p_p_sqlite3;
import org.sqlite.swig.SWIGTYPE_p_sqlite3;
import org.sqlite.types.SQLite3Ptr;
import org.sqlite.types.SQLite3StmtPtrPtr;
import static org.sqlite.swig.SQLite3.*;
import static org.sqlite.swig.SQLite3Constants.*;

/**
 * sqlite3 wrapper class.
 * @author calico
 */
public class Database {
    
    private final String url;
    private final SQLite3Handle handle = new SQLite3Handle();
    private Map<Long, Statement> statements;
    
    /** timeout(ms) : sqlite3_busy_timeout */
    private int timeout;
    
    public Database(String scheme, String filename) throws SQLException {
        open(filename);
        url = scheme + filename;
    }
    
    /**
     * Retrieves the URL for this DBMS.
     * @return the URL for this DBMS.
     */
    public String getURL() {
        return url;
    }
    
    /**
     * invoke sqlite3_open() function.
     * @param filename  filename
     * @throws java.sql.SQLException
     */
    private void open(String filename) throws SQLException {
        final SWIGTYPE_p_p_sqlite3 ppDb = handle.getSQLite3PtrPtr();
        int ret = sqlite3_open(filename, ppDb);
        if (ret != SQLITE_OK) {
            SWIGTYPE_p_sqlite3 db = handle.getInstance();
            SQLException ex = new JdbcSQLException(db);
            handle.delete();
            throw ex;
        }
    }
    
    /**
     * Retrieves whether this Database object has been closed.
     * @return true if this Database object is closed. false if it is still open.
     */
    public boolean isClosed() {
        return (handle.isDeleted());
    }
    
    /**
     * invoke sqlite3_close() function.
     * @throws java.sql.SQLException
     */
    public void close() throws SQLException {
        if (!isClosed()) {
            // TODO もし Statement#closeForced() で例外が発生した場合、Database が閉じられない
            closeStatements();

            final SWIGTYPE_p_sqlite3 db = handle.getInstance();
            int ret = sqlite3_close(db);
            if (ret != SQLITE_OK) {
                throw new JdbcSQLException(db);
            }
            handle.delete();
        }
    }
    
    /**
     * invoke sqlite3_get_autocommit() function.
     * @return  true if auto commit mode.
     */
    public boolean getAutoCommit() {
        final SWIGTYPE_p_sqlite3 db = handle.getInstance();
        return (sqlite3_get_autocommit(db) != 0);
    }
    
    /**
     * invoke sqlite3_busy_timeout() function.
     * @param ms    milliseconds
     * @return
     * @throws java.sql.SQLException
     */
    public int setBusyTimeout(int ms) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = handle.getInstance();
        int ret = sqlite3_busy_timeout(db, ms);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(db);
        }
        timeout = (ms < 1 ? 0 : ms);
        return ret;
    }

    /**
     * Retrieves the timeout(ms) value.
     * @return  timeout(ms) value.
     */
    public int getBusyTimeout() {
        return timeout;
    }
    
    /**
     * invoke sqlite3_exec() function.
     * @param sql
     * @return
     * @throws java.sql.SQLException
     */
    public int execute(String sql) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = handle.getInstance();
        final long sqlite3 = SQLite3Ptr.addressOf(db);
        int ret = 0;
        if (timeout == 0) {
            // no limit
            while ((ret = SQLite3JNI.sqlite3_exec(sqlite3, sql, 0, 0, 0)) == SQLITE_BUSY) {
                // waiting...
            }
        } else {
            ret = SQLite3JNI.sqlite3_exec(sqlite3, sql, 0, 0, 0);
            if (ret == SQLITE_BUSY) {
                // timeout
                throw new SQLException("Timeout expired.");
            }
        }
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(db);
        }
        return ret;
    }

    /**
     * execute PRAGMA commands by sqlite3_exec() finction.
     * @param commands
     * @throws java.sql.SQLException
     */
    public void pragma(String[] commands) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = handle.getInstance();
        final long sqlite3 = SQLite3Ptr.addressOf(db);
        for (final String cmd : commands) {
            int ret = SQLite3JNI.sqlite3_exec(sqlite3, "PRAGMA " + cmd, 0, 0, 0);
            if (ret != SQLITE_OK) {
                throw new JdbcSQLException(db);
            }
        }
    }
    
    /**
     * begin transaction.
     * @param type  transaction type.
     * @throws java.sql.SQLException
     */
    public void beginTransaction(TransactionType type) throws SQLException {
        closeStatements();
        execute("BEGIN " + (type != null ? type : ""));
    }
    
    /**
     * commit toransaction.
     * @throws java.sql.SQLException
     */
    public void commitTransaction() throws SQLException {
        closeStatements();
        execute("COMMIT");
    }
    
    /**
     * rollback transaction.
     * @throws java.sql.SQLException
     */
    public void rollbackTransaction() throws SQLException {
        closeStatements();
        execute("ROLLBACK");
    }
    
    /**
     * create MANAGED Statement instance.
     * @param sql
     * @param ppStmt
     * @return
     * @throws java.sql.SQLException
     */
    public Statement prepare(String sql, SQLite3StmtPtrPtr ppStmt) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = handle.getInstance();
        int ret = sqlite3_prepare(db, sql, -1, ppStmt, null);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(db);
        }
        return new Statement(this, ppStmt.getSQLite3StmtPtr());
//        final Statement stmt = new Statement(this, pStmt);
//        final long addr = SQLite3StmtPtr.getAddress(pStmt);
//        if (statements.contains(addr)) {
//            throw new SQLException("Duplicate handle error.");
//        } else {
//            statements.add(addr);
//        }
//        return stmt;
    }
    
    /**
     * create UNMANAGED Statement instance.
     * @param sql
     * @return
     * @throws java.sql.SQLException
     */
    public Statement prepare(String sql) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = handle.getInstance();
        final SQLite3StmtPtrPtr ppStmt = new SQLite3StmtPtrPtr();
        int ret = sqlite3_prepare(db, sql, -1, ppStmt, null);
        if (ret != SQLITE_OK) {
            ppStmt.delete();
            throw new JdbcSQLException(db);
        }
        return new Statement(this, ppStmt);
    }
    
    /**
     * invoke sqlite3_interrupt() function.
     */
    public void interrupt() {
        final SWIGTYPE_p_sqlite3 db = handle.getInstance();
        sqlite3_interrupt(db);
    }
    
    /**
     * invoke sqlite3_changes() function.
     * @return
     */
    public int changes() {
        final SWIGTYPE_p_sqlite3 db = handle.getInstance();
        return sqlite3_changes(db);
    }
    
    /**
     * invoke sqlite3_total_changes() function.
     * @return
     */
    public int totalChanges() {
        final SWIGTYPE_p_sqlite3 db = handle.getInstance();
        return sqlite3_total_changes(db);
    }
    
    
    public void addStatement(Statement stmt) throws SQLException {
        if (statements == null) {
            statements = new HashMap<Long, Statement>();
        }
        final long key = stmt.getHandle();
        if (statements.containsKey(key)) {
            throw new SQLException("Duplicate sqlite3_stmt handle error.");
        }
        statements.put(key, stmt);
    }
    
    public void removeStatement(Statement stmt) throws SQLException {
        final long key = stmt.getHandle();
        if (statements == null || statements.remove(key) == null) {
            throw new SQLException("Unmanaged sqlite3_stmt handle error.");
        }
    }
    
    private void closeStatements() throws SQLException {
        if (statements != null) {
            for (final Statement stmt : statements.values()) {
                stmt.closeForced();
            }
            statements = null;
        }
    }
    
    
    // TODO sqlite3_create_function()に対応する！
    
    // TODO Statement毎にtimeoutが設定できるようにする！
    
    @Override
    protected void finalize() throws Throwable {
        close();
        super.finalize();
    }
    
}